﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/os/sndfnd_Thread.h>
#include <nw/snd/snd_ThreadStack.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(void* ptrArg)
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NW_ASSERT_NOT_NULL(owner);
        NW_ASSERT_NOT_NULL(owner->m_Handler);

        owner->OnRun();

        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
    }
};

//---------------------------------------------------------------------------
Thread::Thread() :
m_State(STATE_NOT_RUN),
m_ID(INVALID_ID),
m_Priority(DEFAULT_THREAD_PRIORITY),
m_Name(""),
m_Handler(NULL)
{
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(s32 value)
{
    m_Handle.ChangePriority(value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan)
{
    nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromNanoSeconds(timeSpan.GetTick()));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, u32& id, const RunArgs& args)
{
    (void)handle;
    (void)id;

    nw::snd::internal::ThreadStack stack;
    stack.Initialize(reinterpret_cast<uptr>(const_cast<void*>(args.stack)), args.stackSize);

    nn::Result result = handle.TryStart(
        ThreadMain::Run,
        this,
        stack,
        args.priority);

    if (result.IsFailure())
    {
        m_ID = INVALID_ID;
        return false;
    }

    stack.Finalize();

    // nn::os::Thread へのポインタを ID として利用します。
    m_ID = reinterpret_cast<u32>(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach()
{
    if(IsTerminated())
    {
        return;
    }

    m_Handle.Join();
    m_Handle.Finalize();
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name)
{
    (void)name;
    // 対応APIがないので処理しません。
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AFFINITY_MASK value)
{
    (void)value;
    // 対応APIがないので処理しません。
}

//---------------------------------------------------------------------------
void
Thread::Resume()
{
    // 対応APIがないので処理しません。
}

//---------------------------------------------------------------------------
void
Thread::Join()
{
    m_Handle.Join();
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const
{
    return m_Handle.IsAlive();
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


