﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_Stream.h>

#if !defined(NW_PLATFORM_CTR)
#include <nw/ut/ut_String.h>
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {
static const char NEW_LINE_CODE[]      = "\r\n";
static const u32  NEW_LINE_CODE_LENGTH = sizeof(char) * 2;
}

//---------------------------------------------------------------------------
FndResult
StreamWriter::WriteLine(const char* text, u32 length)
{
    NW_ASSERT_NOT_NULL(text);
    (void)length;

    FndResult result(SNDFND_RESULT_TRUE);
    Write(text, std::strlen(text), &result);

    if(result.IsFailed())
    {
        return result;
    }

    Write(NEW_LINE_CODE, NEW_LINE_CODE_LENGTH, &result);

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamWriter::WriteFormat(const char* format, ...)
{
    NW_ASSERT_NOT_NULL(format);

    std::va_list vargs;
    va_start(vargs, format);

    FndResult result = VWriteFormat(format, vargs);

    va_end(vargs);

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamWriter::VWriteFormat(const char* format, va_list vargs)
{
    NW_ASSERT_NOT_NULL(m_WriteFormatBuffer);
    NW_ASSERT(m_WriteFormatBufferLength > 0);
    NW_ASSERT_NOT_NULL(format);

    s32 result = ut::vsnprintf(
        m_WriteFormatBuffer,
        m_WriteFormatBufferLength,
        m_WriteFormatBufferLength - 1,
        format,
        vargs);

    if(result < 0)
    {
        NW_ASSERT(0);
        return FndResult(SNDFND_RESULT_FAILED);
    }

    return Write(m_WriteFormatBuffer, result);
}

//---------------------------------------------------------------------------
void
StreamWriter::EnableWriteFormat(void* buffer, u32 length)
{
    NW_ASSERT( (buffer == NULL) == (length == 0) );
    m_WriteFormatBuffer = reinterpret_cast<char*>(buffer);
    m_WriteFormatBufferLength = length;
}

//---------------------------------------------------------------------------
void
StreamWriter::DisableWriteFormat()
{
    m_WriteFormatBuffer = NULL;
    m_WriteFormatBufferLength = 0;
}

//---------------------------------------------------------------------------
void
StreamAligner::Open(Stream* stream, u32 alignment, void* buf, u32 length)
{
    NW_ASSERT_NOT_NULL(stream);
    NW_ASSERT(alignment > 0);
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERT(length >= GetRequiredBufferLength(alignment));

    m_Stream = stream;
    m_Alignment = alignment;
    m_BufferForAlignment = ut::RoundUp(buf, alignment);
    m_BufferLengthForAlignment = length - ut::GetOffsetFromPtr(buf, m_BufferForAlignment);
}

//---------------------------------------------------------------------------
void
StreamAligner::Close()
{
    m_Stream = NULL;
    m_Alignment = 0;
    m_BufferForAlignment = NULL;
    m_BufferLengthForAlignment = 0;
}

//---------------------------------------------------------------------------
u32
StreamAligner::Read(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "AlignedStreamReader is not opened.\n");

    void* alignedBuf = ut::RoundUp(buf, m_Alignment);

    if(alignedBuf == buf)
    {
        return m_Stream->Read(buf, length, result);
    }

    u32 alignLength = ut::GetOffsetFromPtr(buf, alignedBuf);
    u32 restLength = length - alignLength;

    FndResult readResult(SNDFND_RESULT_TRUE);

    // アライメント調整されていない場合は、２回に分けて読み込みます。
    if(alignLength > 0)
    {
        NW_ASSERT(alignLength < m_Alignment);

        u32 readLength = m_Stream->Read(m_BufferForAlignment, alignLength, &readResult);

        if(result != NULL)
        {
            *result = readResult;
        }

        if(readResult.IsFailed())
        {
            return 0;
        }

        if(readLength <= alignLength)
        {
            return readLength;
        }
    }

    // 残りを読み込みます。
    u32 restReadLength = m_Stream->Read(alignedBuf, restLength, &readResult);

    if(result != NULL)
    {
        *result = readResult;
    }

    if(readResult.IsFailed())
    {
        return 0;
    }

    return alignLength + restReadLength;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
