﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#include <nw/snd/fnd/io/sndfnd_File.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
u32
FileStreamImpl::Read(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Read(buf, length, result);
    }

    return ReadDirect(buf, length, result);
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::Write(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Write(buf, length, result);
    }

    return WriteDirect(buf, length, result);
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::Seek(s32 offset, SeekOrigin origin)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Seek(offset, origin);
    }

    return SeekDirect(offset, origin);
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanRead() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanWrite() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanSeek() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
void
FileStreamImpl::EnableCache(void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);

    if(m_StreamCache.IsInitialized())
    {
        m_StreamCache.Finalize();
    }

    void* alignedBuffer = ut::RoundUp(buffer, FileTraits::IO_BUFFER_ALIGNMENT);

    m_StreamCache.Initialize(
        &m_DirectStream,
        alignedBuffer,
        length - ut::GetOffsetFromPtr(buffer, alignedBuffer));
}

//---------------------------------------------------------------------------
void
FileStreamImpl::DisableCache()
{
    m_StreamCache.Finalize();
}

//---------------------------------------------------------------------------
void
FileStreamImpl::ValidateAlignment(const void* buf) const
{
    if(ut::RoundUp(buf, FileTraits::IO_BUFFER_ALIGNMENT) != buf)
    {
        NW_FATAL_ERROR(
            "invalid alignment : buf(0x%08x) must be aligned by %d.\n",
            buf,
            FileTraits::IO_BUFFER_ALIGNMENT);
    }
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
