﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/vwrlyt/vwrlyt_Inputs.h>

#if defined(NW_VWRLYT_ENABLE)

namespace nw
{
namespace vwrlyt
{

namespace internal
{

//-----------------------------------------------------------------------------
// Pad入力を統合します。
//-----------------------------------------------------------------------------

UnifiedPad::UnifiedPad(Inputs* pInputs)
: m_pInputs(pInputs)
, m_PadLeftAnalogTrigger(0.f)
, m_PadRightAnalogTrigger(0.f)
, m_PadLeftStick(0.f, 0.f)
, m_PadRightStick(0.f, 0.f)
{}

/* protected virtual */
void UnifiedPad::UpdateImpl()
{
    if (!m_pInputs)
    {
        return;
    }

    // パッドをクリア
    m_PadHold.SetAllZero();

    if (const nw::dev::Pad* pPad = m_pInputs->GetPadSrc())
    {
        m_PadHold.SetMaskOn(pPad->GetHoldMask());
        this->UpdateValue(&m_LeftAnalogTrigger, &m_PadLeftAnalogTrigger, pPad->GetLeftAnalogTrigger());
        this->UpdateValue(&m_RightAnalogTrigger, &m_PadRightAnalogTrigger, pPad->GetRightAnalogTrigger());
        this->UpdateValue(&m_LeftStick, &m_PadLeftStick, pPad->GetLeftStick());
        this->UpdateValue(&m_RightStick, &m_PadRightStick, pPad->GetRightStick());
    }

    if (const nw::dev::Pad* pWPad = m_pInputs->GetWPadSrc())
    {
        m_PadHold.SetMaskOn(pWPad->GetHoldMask());
    }
}

//-----------------------------------------------------------------------------
// Mouse入力を統合します。
//-----------------------------------------------------------------------------

UnifiedMouse::UnifiedMouse(Inputs* pInputs)
: m_pInputs(pInputs)
{}

/* protected virtual */
void UnifiedMouse::UpdateImpl()
{
    if (!m_pInputs)
    {
        return;
    }

    m_PadHold.SetAllZero();

    nw::ut::BitFlag32 prevPointerFlag = m_PointerFlag;
    m_PointerFlag.SetAllZero();
    m_WheelDelta = 0;
    m_IsDoubleClick = false;

    m_PointerPrev = m_Pointer;
    const nw::dev::Mouse* pMouse = m_pInputs->GetMouseSrc();
    if (pMouse && pMouse->IsConnected())
    {
        if (pMouse->IsPointerOn())
        {
            m_PointerFlag.SetMaskOn(POINTER_ON);
        }
        m_PadHold.SetMaskOn(pMouse->GetHoldMask());
        if (pMouse->GetWheel() != 0)
        {
            m_WheelDelta = pMouse->GetWheel();
        }
        if (pMouse->IsDoubleClick())
        {
            m_IsDoubleClick = pMouse->IsDoubleClick();
        }
        if (pMouse->IsPointerOnNow() ||
            (pMouse->IsPointerOn() &&
            pMouse->GetPointerDiff().LengthSquare() > 0.f))
        {
            m_Pointer = pMouse->GetPointer();
        }
    }

#if defined(NW_PLATFORM_CAFE)
    const nw::dev::Pad* pWPad = m_pInputs->GetWPadSrc();
    if (pWPad && pWPad->IsConnected())
    {
       if (pWPad->IsPointerOn())
        {
            m_PointerFlag.SetMaskOn(POINTER_ON);
        }
        if (pWPad->IsHold(pWPad->MASK_A))
        {
            m_PadHold.SetMaskOn(this->MASK_LBUTTON);
        }
        if (pWPad->IsHold(pWPad->MASK_B))
        {
            m_PadHold.SetMaskOn(this->MASK_RBUTTON);
        }
        if (pWPad->IsPointerOn())
        {
            nw::math::VEC2 p = pWPad->GetPointer();
            const nw::lyt::Size& viewportSize = m_pInputs->GetMainViewportSize();
            p.x = (p.x + 1.f) * 0.5f * viewportSize.width;
            p.y = (p.y + 1.f) * 0.5f * viewportSize.height;

            if (pWPad->IsPointerOnNow())
            {
                m_Pointer = p;
            }
            else
            {
                const f32 WEIGHT_PREV = 0.5f;
                const f32 WEIGHT_NEW = 0.5f;

                m_Pointer.x = m_Pointer.x * WEIGHT_PREV + p.x * WEIGHT_NEW;
                m_Pointer.y = m_Pointer.y * WEIGHT_PREV + p.y * WEIGHT_NEW;
            }
        }
    }
#endif

    if (prevPointerFlag.IsMaskOff(POINTER_ON) && m_PointerFlag.IsMaskOn(POINTER_ON))
    {
        m_PointerFlag.SetMaskOn(POINTER_ON_NOW);
    }

    if (prevPointerFlag.IsMaskOn(POINTER_ON) && m_PointerFlag.IsMaskOff(POINTER_ON))
    {
        m_PointerFlag.SetMaskOn(POINTER_OFF_NOW);
    }
}

} // namespace internal

//-----------------------------------------------------------------------------
// 入力を統合します。
//-----------------------------------------------------------------------------

Inputs::Inputs()
: m_pPad(NULL)
, m_pVPad(NULL)
, m_pWPad(NULL)
, m_pMouse(NULL)
, m_UnifiedPad(this)
, m_UnifiedMouse(this)
{
}

/* virtual */
Inputs::~Inputs()
{
}

/* virtual */
void Inputs::Update()
{
    m_UnifiedPad.Update();
    m_UnifiedMouse.Update();
}

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
