﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/vwrlyt/vwrlyt_AnimationManager.h>
#include <nw/vwrlyt/vwrlyt_AllPaneAnimator.h>

#if defined(NW_VWRLYT_ENABLE)

namespace nw
{
namespace vwrlyt
{


AnimationManager::AnimationManager()
 : m_Layout(NULL)
 , m_ResourceAccessor(NULL)
 , m_AnimationNum(0)
 , m_CurrentAnimationNo(0)
 , m_TargetMode(TARGET_MODE_PANE)
 , m_Animator(NULL)
 , m_CurrentTargetNo(0)
 , m_CurrentTargetName("")
 , m_LayoutNameLength(0)
{
}

void AnimationManager::Setup(
    lyt::Layout* layout,
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    vwrlyt::DirResourceAccessor* resourceAccessor
#elif defined(NW_PLATFORM_CAFE)
    vwrlyt::FindableArcResourceAccessor* resourceAccessor
#endif
)
{
    m_Layout = layout;
    m_ResourceAccessor = resourceAccessor;

    ///---- このレイアウトが持っているアニメーションの数を数える
    const char* layoutName = layout->GetName();
    const char* animFileName = NULL;
    m_AnimationNum = 0;
    m_LayoutNameLength = static_cast<u32>(ut::SafeString(layoutName).calcLength());

    // まずは現在プレビューしているレイアウトに属しているアニメーションの数を数える
    do {
        animFileName = resourceAccessor->FindFile(lyt::res::RESOURCETYPE_ANIMATION, animFileName);
        if (animFileName != NULL)
        {
            if (ut::SafeString(animFileName).findIndex(layoutName) == 0)
            {
                // 現在プレビューしているレイアウトに属しているアニメーション
                m_AnimationNum++;
            }
        }
    } while (animFileName != NULL);

    // アニメーションの名前を取得しておく
    if (m_AnimationNum > 0)
    {
        do {
            animFileName = resourceAccessor->FindFile(lyt::res::RESOURCETYPE_ANIMATION, animFileName);
            if (animFileName != NULL)
            {
                if (ut::SafeString(animFileName).findIndex(layoutName) == 0)
                {
                    // 現在プレビューしているレイアウトに属しているアニメーション
                    mAnimInfoArray.push_back(AnimInfo());
                    AnimInfo& info = mAnimInfoArray.back();
                    info.name.copy(animFileName);
                    lyt::AnimResource animRes(resourceAccessor->GetResource(lyt::res::RESOURCETYPE_ANIMATION, animFileName));
                    info.tagOrder = animRes.GetTagOrder();
                    if (info.tagOrder == 0xffff)
                    {
                        // AllアニメのtagOrderは0xffffになる。これは先頭にする。
                        info.tagOrder = -1;
                    }
                }
            }
        } while (animFileName != NULL);

        // 区間タグの通し番号順にソートする。
        std::sort(mAnimInfoArray.begin(), mAnimInfoArray.end(), AnimInfo::compareTagOrder);
    }

    if (GetAnimationNum() > 0)
    {
        SetCurrentAnimationNo(0);
    }
}

void AnimationManager::Finalize()
{
    if (m_Animator)
    {
        m_Animator->Unbind();
        m_Layout->DeleteAnimTransform(m_Animator);
        m_Animator = NULL;
    }
    m_Layout = NULL;
    m_ResourceAccessor = NULL;
    m_AnimationNum = 0;
    m_CurrentAnimationNo = 0;
    m_CurrentAnimationName.clear();
    m_CurrentTargetNo = 0;
    m_CurrentTargetName = "";
    m_LayoutNameLength = 0;
    mAnimInfoArray.clear();
}

u32 AnimationManager::GetAnimationNum() const
{
    if (m_AnimationNum > 0)
    {
        if (m_TargetMode == TARGET_MODE_GROUP)
        {
            return m_AnimationNum - 1;
        }
        else
        {
            return m_AnimationNum;
        }
    }
    else
    {
        return 0;
    }
}

u32 AnimationManager::GetCurrentAnimationNo() const
{
    if (m_AnimationNum > 0)
    {
        if (m_TargetMode == TARGET_MODE_GROUP)
        {
            return m_CurrentAnimationNo - 1;
        }
        else
        {
            return m_CurrentAnimationNo;
        }
    }
    else
    {
        return 0;
    }
}

void AnimationManager::SetTargetMode(TargetMode mode)
{
    m_TargetMode = mode;
    if (GetAnimationNum() > 0)
    {
        SetCurrentAnimationNo(0);
    }
}

void AnimationManager::SetCurrentAnimationNo(u32 no)
{
    if (no < GetAnimationNum())
    {
        if (m_TargetMode == TARGET_MODE_PANE)
        {
            m_CurrentAnimationNo = no;
        }
        else
        {
            m_CurrentAnimationNo = no + 1;
        }
        if (m_CurrentAnimationNo == 0)
        {
            m_CurrentAnimationName.copy("All");
        }
        else
        {
            this->GetAnimTagNameFromAnimFileName(&m_CurrentAnimationName, mAnimInfoArray[m_CurrentAnimationNo].name);
        }
        m_CurrentAnimResource.Set(m_ResourceAccessor->GetResource(lyt::res::RESOURCETYPE_ANIMATION, mAnimInfoArray[m_CurrentAnimationNo].name.c_str()));
        SetCurrentTargetNo(0);
    }
    else
    {
        NW_ASSERTMSG(false, "no[%d] exceeds animation num[%d]", no, GetAnimationNum());
    }
}

u32 AnimationManager::GetCurrentTargetNum()
{
    if (m_Animator)
    {
        if (m_TargetMode == TARGET_MODE_PANE)
        {
            return m_CurrentAnimResource.GetResourceBlock()->animContNum + 1;
        }
        else
        {
            return m_CurrentAnimResource.GetGroupNum() + 1;
        }
    }
    else
    {
        return 0;
    }
}

void AnimationManager::SetCurrentTargetNo(u32 no)
{
    m_CurrentTargetNo = no;
    if (m_Animator)
    {
        m_Animator->Unbind();
        m_Layout->DeleteAnimTransform(m_Animator);
        m_Animator = NULL;
    }
    if (m_TargetMode == TARGET_MODE_PANE)
    {
        const lyt::res::AnimationBlock* pRes = m_CurrentAnimResource.GetResourceBlock();
        if (no <= pRes->animContNum)
        {
            if (no == 0)
            {
                m_CurrentTargetName = "All";
                vwrlyt::AllPaneAnimator* animator = m_Layout->CreateAnimTransform<vwrlyt::AllPaneAnimator>(m_CurrentAnimResource);
                animator->Setup(m_Layout, true);
                m_Animator = animator;
            }
            else
            {
                const ut::ResU32 *const animContOffsets = lyt::internal::ConvertOffsToPtr<ut::ResU32>(pRes, pRes->animContOffsetsOffset);
                const lyt::res::AnimationContent& animCont = *lyt::internal::ConvertOffsToPtr<lyt::res::AnimationContent>(pRes, animContOffsets[no - 1]);
                m_CurrentTargetName = animCont.name;
                if (m_CurrentAnimationNo == 0)
                {
                    nw::lyt::PaneAnimator* animator = m_Layout->CreateAnimTransform<nw::lyt::PaneAnimator>(m_CurrentAnimResource);
                    animator->Setup(m_Layout->GetRootPane()->FindPaneByName(m_CurrentTargetName), true);
                    m_Animator = animator;
                }
                else
                {
                    m_Animator = m_Layout->CreatePaneAnimator(m_CurrentAnimationName.c_str(), m_CurrentTargetName, true);
                }
            }
        }
        else
        {
            NW_ASSERTMSG(false, "no[%d] exceeds anim cont num[%d]", no, m_CurrentAnimResource.GetResourceBlock()->animContNum);
        }
    }
    else
    {
        if (no <= m_CurrentAnimResource.GetGroupNum())
        {
            if (no == 0)
            {
                m_CurrentTargetName = "All";
                if (m_CurrentAnimResource.GetGroupNum() == 0)
                {
                    // グループが関連付けられていない場合はAllPaneAnimatorで再生する
                    vwrlyt::AllPaneAnimator* animator = m_Layout->CreateAnimTransform<vwrlyt::AllPaneAnimator>(m_CurrentAnimResource);
                    animator->Setup(m_Layout, true);
                    m_Animator = animator;
                }
                else
                {
                    // グループが関連付けられていればGroupArrayAnimatorで再生する
                    ut::FixedSafeString<lyt::AnimTagNameStrMax> name;
                    GetAnimTagNameFromAnimFileName(&name, mAnimInfoArray[m_CurrentAnimationNo].name);
                    m_Animator = m_Layout->CreateGroupArrayAnimator(name.c_str(), true);
                }
            }
            else
            {
                m_CurrentTargetName = m_CurrentAnimResource.GetGroupArray()[no - 1].GetName();
                m_Animator = m_Layout->CreateGroupAnimator(m_CurrentAnimationName.c_str(), m_CurrentTargetName, true);
            }
        }
        else
        {
            NW_ASSERTMSG(false, "no[%d] exceeds group num[%d]", no, m_CurrentAnimResource.GetGroupNum());
        }
    }
}

bool AnimationManager::IsCurrentAnimationLoop() const
{
    if (m_Animator)
    {
        const lyt::res::AnimationBlock* pRes = m_CurrentAnimResource.GetResourceBlock();
        return (pRes != NULL && pRes->loop == 1);
    }
    else
    {
        return false;
    }
}

void AnimationManager::StartAnimation()
{
    if (m_Animator)
    {
        if (m_Animator->IsMaxFrame())
        {
            m_Animator->Play(IsCurrentAnimationLoop() ? lyt::Animator::PLAYTYPE_LOOP : lyt::Animator::PLAYTYPE_ONESHOT, 1.f);
        }
        else
        {
            m_Animator->PlayFromCurrent(IsCurrentAnimationLoop() ? lyt::Animator::PLAYTYPE_LOOP : lyt::Animator::PLAYTYPE_ONESHOT, 1.f);
        }
    }
}

void AnimationManager::StopAnimation()
{
    if (m_Animator)
    {
        m_Animator->StopCurrent();
    }
}

void AnimationManager::ResetAnimation()
{
    if (m_Animator)
    {
        m_Animator->StopAtMin();
    }
}

bool AnimationManager::IsAnimationPlaying() const
{
    if (m_Animator)
    {
        return m_Animator->IsPlaying();
    }
    else
    {
        return false;
    }
}

f32 AnimationManager::GetCurrentAnimationFrame() const
{
    if (m_Animator)
    {
        return m_Animator->GetFrame();
    }
    else
    {
        return 0;
    }
}

f32 AnimationManager::GetAnimationFrameMax() const
{
    if (m_Animator)
    {
        return m_Animator->GetFrameMax();
    }
    else
    {
        return 0;
    }
}

void AnimationManager::GetAnimTagNameFromAnimFileName(ut::BufferedSafeString* dst, const ut::SafeString& src)
{
    NW_ASSERT_NOT_NULL(dst);

    ut::SafeString part = src.getPart(m_LayoutNameLength + 1);
    dst->copy(part, part.calcLength() - 6);
}

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
