﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Window.h>

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_ResourceAccessor.h>

#include <nw/gfnd/gfnd_ShaderHelper.h>
namespace shhelp = nw::gfnd::shader_helper;

#include <nw/dev/dev_Profile.h>

namespace nw
{
namespace lyt
{

namespace wfnd = window_foundation;

//----------------------------------------
Window::Frame::~Frame()
{
    if (pMaterial && !pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(pMaterial);
    }
    pMaterial = NULL;
}

//----------------------------------------
Window::Window(
    u8  contentTexNum,
    u8  frameTexNum
)
: m_FrameMode(AROUND_MODE)
, m_WindowFlags(0)
{
    NW_ASSERT(contentTexNum <= TexMapMax);
    NW_ASSERT(frameTexNum <= TexMapMax);

    const u8 frameNum = 1;
    u8 frameTexNums[frameNum];
    frameTexNums[WINDOWFRAME_LT] = frameTexNum;

    InitTexNum(contentTexNum, frameTexNums, frameNum);
}

//----------------------------------------
Window::Window(
    u8  contentTexNum,
    u8  frameLTTexNum,
    u8  frameRTTexNum,
    u8  frameRBTexNum,
    u8  frameLBTexNum
)
: m_FrameMode(AROUND_MODE)
, m_WindowFlags(0)
{
    NW_ASSERT(contentTexNum <= TexMapMax);
    NW_ASSERT(frameLTTexNum <= TexMapMax);
    NW_ASSERT(frameRTTexNum <= TexMapMax);
    NW_ASSERT(frameRBTexNum <= TexMapMax);
    NW_ASSERT(frameLBTexNum <= TexMapMax);

    const u8 frameNum = 4;
    u8 frameTexNums[frameNum];
    frameTexNums[WINDOWFRAME_LT] = frameLTTexNum;
    frameTexNums[WINDOWFRAME_RT] = frameRTTexNum;
    frameTexNums[WINDOWFRAME_RB] = frameRBTexNum;
    frameTexNums[WINDOWFRAME_LB] = frameLBTexNum;

    InitTexNum(contentTexNum, frameTexNums, frameNum);
}

//----------------------------------------
Window::Window(
    u8  contentTexNum,
    u8  cornerLTTexNum,
    u8  cornerRTTexNum,
    u8  cornerRBTexNum,
    u8  cornerLBTexNum,
    u8  frameLTexNum,
    u8  frameTTexNum,
    u8  frameRTexNum,
    u8  frameBTexNum
)
: m_FrameMode(AROUND_MODE)
, m_WindowFlags(0)
{
    NW_ASSERT(contentTexNum <= TexMapMax);
    NW_ASSERT(cornerLTTexNum <= TexMapMax);
    NW_ASSERT(cornerRTTexNum <= TexMapMax);
    NW_ASSERT(cornerRBTexNum <= TexMapMax);
    NW_ASSERT(cornerLBTexNum <= TexMapMax);
    NW_ASSERT(frameLTexNum <= TexMapMax);
    NW_ASSERT(frameTTexNum <= TexMapMax);
    NW_ASSERT(frameRTexNum <= TexMapMax);
    NW_ASSERT(frameBTexNum <= TexMapMax);

    const u8 frameNum = 8;
    u8 frameTexNums[frameNum];
    frameTexNums[WINDOWFRAME_LT] = cornerLTTexNum;
    frameTexNums[WINDOWFRAME_RT] = cornerRTTexNum;
    frameTexNums[WINDOWFRAME_RB] = cornerRBTexNum;
    frameTexNums[WINDOWFRAME_LB] = cornerLBTexNum;
    frameTexNums[WINDOWFRAME_L ] = frameLTexNum;
    frameTexNums[WINDOWFRAME_T ] = frameTTexNum;
    frameTexNums[WINDOWFRAME_R ] = frameRTexNum;
    frameTexNums[WINDOWFRAME_B ] = frameBTexNum;

    InitTexNum(contentTexNum, frameTexNums, frameNum);

}

//----------------------------------------
Window::Window(
    const res::Window* pBaseBlock,
    const res::Window* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
: Base(pBaseBlock, buildArgSet)
, m_FrameMode(AROUND_MODE)
, m_WindowFlags(0)
{
    const res::Window* pBlock;
    if (pOverrideBlock && buildArgSet.overrideMaterialUsageFlag == 0)
    {
        // 丸ごと上書きする場合
        pBlock = pOverrideBlock;
    }
    else
    {
        // 上書きがない、もしくは部分上書きの場合
        pBlock = pBaseBlock;
    }

    m_FrameMode = static_cast<WindowFrameMode>(
        (pBlock->windowFlags >> WINDOWFLAG_WINDOWKIND0) & ((1 << WINDOWFLAG_WINDOWKIND_LENGTH) - 1));

    if (pBlock->windowFlags & (1 << WINDOWFLAG_USEVTXCOLALL)) { m_WindowFlags |= OVERALL_VERTEX_COLOR_ENABLED; }
    if (pBlock->windowFlags & (1 << WINDOWFLAG_USEONEMATERIALFORALL)) { m_WindowFlags |= ONEMATERIAL_FOR_ALLFRAME_ENABLED; }
    if (pBlock->windowFlags & (1 << WINDOWFLAG_NOT_DRAW_CONTENT)) { m_WindowFlags |= NOT_DRAW_CONTENT_ENABLED; }

    const res::WindowContent *const pResContent = internal::ConvertOffsToPtr<res::WindowContent>(pBlock, pBlock->contentOffset);
    const u8 texCoordNum = ut::Min(pResContent->texCoordNum, u8(TexMapMax));

    InitContent(texCoordNum);

    m_WindowSize.l = pBlock->inflation.l;
    m_WindowSize.r = pBlock->inflation.r;
    m_WindowSize.t = pBlock->inflation.t;
    m_WindowSize.b = pBlock->inflation.b;
    if (m_WindowSize.l != 0 || m_WindowSize.r != 0 || m_WindowSize.t != 0 || m_WindowSize.b != 0)
    {
        m_WindowFlags |= CONTENT_INFLATION_ENABLED;
    }
    switch (m_FrameMode)
    {
    case HORIZONTAL_MODE:
        m_WindowSize.frameSize.l = pBlock->frameSize.l;
        m_WindowSize.frameSize.r = pBlock->frameSize.r;
        m_WindowSize.frameSize.t = 0;
        m_WindowSize.frameSize.b = 0;
        SetSize(Size(GetSize().width, static_cast<f32>(pBlock->frameSize.t)));
        break;
    case HORIZONTAL_NOCONTENT_MODE:
        m_WindowSize.frameSize.l = static_cast<u16>(GetSize().width) - pBlock->frameSize.r;
        m_WindowSize.frameSize.r = pBlock->frameSize.r;
        m_WindowSize.frameSize.t = 0;
        m_WindowSize.frameSize.b = 0;
        SetSize(Size(GetSize().width, static_cast<f32>(pBlock->frameSize.t)));
        break;
    case AROUND_MODE:
    default:
        m_WindowSize.frameSize.l = pBlock->frameSize.l;
        m_WindowSize.frameSize.r = pBlock->frameSize.r;
        m_WindowSize.frameSize.t = pBlock->frameSize.t;
        m_WindowSize.frameSize.b = pBlock->frameSize.b;
        break;
    }

    // content
    // 頂点カラー
    for (int i = 0; i < VERTEXCOLOR_MAX; ++i)
    {
        m_Content.vtxColors[i] = pResContent->vtxCols[i];
    }

    // テクスチャ座標
    if (texCoordNum > 0)
    {
        if (! m_Content.texCoordAry.IsEmpty())
        {
            m_Content.texCoordAry.Copy(
                reinterpret_cast<const char*>(pResContent) + sizeof(*pResContent),
                texCoordNum);
        }
    }

    // マテリアルの作成
    {
        const res::WindowContent *const pBaseResContent = internal::ConvertOffsToPtr<res::WindowContent>(pBaseBlock, pBaseBlock->contentOffset);
        const res::Material *const pResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseResContent->materialIdx);
        const res::Material *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            const res::WindowContent *const pOverrideResContent = internal::ConvertOffsToPtr<res::WindowContent>(pOverrideBlock, pOverrideBlock->contentOffset);
            pOverrideResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideResContent->materialIdx);
        }
        m_pMaterial = Layout::NewObj<Material, const res::Material*, const res::Material*, const BuildArgSet&>(pResMaterial, pOverrideResMaterial, buildArgSet);
    }

    // Frame
    m_FrameNum = 0;
    m_Frames = 0;
    if (pBlock->frameNum > 0)
    {
        InitFrame(pBlock->frameNum);

        const ut::ResU32 *const frameOffsetTable = internal::ConvertOffsToPtr<ut::ResU32>(pBlock, pBlock->frameOffsetTableOffset);
        const ut::ResU32 *const baseFrameOffsetTable = internal::ConvertOffsToPtr<ut::ResU32>(pBaseBlock, pBaseBlock->frameOffsetTableOffset);
        const ut::ResU32 * overrideFrameOffsetTable = NULL;
        if (pOverrideBlock) {
            overrideFrameOffsetTable = internal::ConvertOffsToPtr<ut::ResU32>(pOverrideBlock, pOverrideBlock->frameOffsetTableOffset);
        }

        for (int i = 0; i < m_FrameNum; ++i)
        {
            const res::WindowFrame *const pResWindowFrame = internal::ConvertOffsToPtr<res::WindowFrame>(pBlock, frameOffsetTable[i]);
            m_Frames[i].textureFlip = pResWindowFrame->textureFlip;

            // マテリアルの初期化
            const res::WindowFrame *const pBaseResWindowFrame = internal::ConvertOffsToPtr<res::WindowFrame>(pBaseBlock, baseFrameOffsetTable[i]);
            const res::Material *const pResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseResWindowFrame->materialIdx);
            const res::Material *pOverrideResMaterial = NULL;
            if (pOverrideBlock) {
                const res::WindowFrame *const pOverrideResWindowFrame = internal::ConvertOffsToPtr<res::WindowFrame>(pOverrideBlock, overrideFrameOffsetTable[i]);
                pOverrideResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideResWindowFrame->materialIdx);
            }
            m_Frames[i].pMaterial = Layout::NewObj<Material, const res::Material*, const res::Material*, const BuildArgSet&>(pResMaterial, pOverrideResMaterial, buildArgSet);
        }
    }
}

//----------------------------------------
Window::Window(const Window& window)
 : Base(window)
 , m_WindowSize(window.m_WindowSize)
 , m_FrameMode(window.m_FrameMode)
 , m_FrameNum(window.m_FrameNum)
 , m_WindowFlags(window.m_WindowFlags)
{
    // content
    // 頂点カラー
    for (int i = 0; i < VERTEXCOLOR_MAX; ++i)
    {
        m_Content.vtxColors[i] = window.m_Content.vtxColors[i];
    }

    // テクスチャ座標
    const u8 texCoordNum = window.m_Content.texCoordAry.GetSize();
    if (texCoordNum > 0)
    {
        InitContent(texCoordNum);
        m_Content.texCoordAry.SetSize(texCoordNum);
        for (u32 i = 0; i < texCoordNum; i++)
        {
            m_Content.texCoordAry.SetCoord(i, window.m_Content.texCoordAry.GetArray()[i]);
        }
    }

    // コンテンツのマテリアルの複製
    m_pMaterial = Layout::NewObj<Material, const Material&>(*window.m_pMaterial);

    // Frame
    InitFrame(m_FrameNum);
    for (int i = 0; i < m_FrameNum; ++i)
    {
        m_Frames[i].textureFlip = window.m_Frames[i].textureFlip;
        m_Frames[i].pMaterial = Layout::NewObj<Material, const Material&>(*window.m_Frames[i].pMaterial);
    }
}

//----------------------------------------
void
Window::InitTexNum(
    u8  contentTexNum,
    u8  frameTexNums[],
    u8  frameNum
)
{
    InitContent(contentTexNum);

    m_WindowSize.l = 0;
    m_WindowSize.r = 0;
    m_WindowSize.t = 0;
    m_WindowSize.b = 0;
    m_WindowSize.frameSize.l = 0;
    m_WindowSize.frameSize.r = 0;
    m_WindowSize.frameSize.t = 0;
    m_WindowSize.frameSize.b = 0;

    // マテリアルの作成
    m_pMaterial = Layout::NewObj<Material>();
    if (m_pMaterial)
    {
        m_pMaterial->ReserveMem(contentTexNum, contentTexNum, contentTexNum);
    }

    // Frame
    InitFrame(frameNum);

    for (int i = 0; i < m_FrameNum; ++i)
    {
        // マテリアルの作成
        m_Frames[i].pMaterial = Layout::NewObj<Material>();
        if (m_Frames[i].pMaterial)
        {
            m_Frames[i].pMaterial->ReserveMem(frameTexNums[i], frameTexNums[i], frameTexNums[i]);
        }
    }
}

//----------------------------------------
void
Window::InitContent(u8 texNum)
{
    if (texNum > 0)
    {
        ReserveTexCoord(texNum);
    }
}

//----------------------------------------
void
Window::InitFrame(u8 frameNum)
{
    m_FrameNum = 0;
    m_Frames = Layout::NewArray<Frame>(frameNum);
    if (m_Frames)
    {
        m_FrameNum = frameNum;
    }
}

//----------------------------------------
Window::~Window()
{
    // OSReport("Window::~Window()\n");

    Layout::DeleteArray(m_Frames, m_FrameNum);

    if (m_pMaterial && ! m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    m_Content.texCoordAry.Free();
}

//----------------------------------------
const ut::Color8
Window::GetVtxColor(u32 idx) const
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] < VERTEXCOLOR_MAX for Window[%s]", idx, GetName());

    return m_Content.vtxColors[idx];
}

//----------------------------------------
void
Window::SetVtxColor(
    u32         idx,
    ut::Color8 value
)
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] < VERTEXCOLOR_MAX for Window[%s]", idx, GetName());

    m_Content.vtxColors[idx] = value;
}

//----------------------------------------
u8
Window::GetVtxColorElement(u32 idx) const
{
    return internal::GetVtxColorElement(m_Content.vtxColors, idx);
}

//----------------------------------------
void
Window::SetVtxColorElement(u32 idx, u8 value)
{
    internal::SetVtxColorElement(m_Content.vtxColors, idx, value);
}

//----------------------------------------
u8
Window::GetMaterialNum() const
{
    return u8(1 + m_FrameNum);
}

//----------------------------------------
Material*
Window::GetMaterial(u32 idx) const
{
    NW_WARNING(idx < GetMaterialNum(), "idx >= GetMaterialNum() : %d >= %d", idx, GetMaterialNum());

    return idx == 0 ? GetContentMaterial(): GetFrameMaterial(WindowFrame(idx - 1));
}

//----------------------------------------
Material*
Window::FindMaterialByName(
    const char* findName,
    bool        bRecursive
)
{
    if (m_pMaterial)
    {
        if (internal::EqualsMaterialName(m_pMaterial->GetName(), findName))
        {
            return m_pMaterial;
        }
    }
    for (int i = 0; i < m_FrameNum; ++i)
    {
        if (internal::EqualsMaterialName(m_Frames[i].pMaterial->GetName(), findName))
        {
            return m_Frames[i].pMaterial;
        }
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::Iterator it_end = GetChildList().GetEndIter();
        for (PaneList::Iterator it = GetChildList().GetBeginIter(); it != it_end; ++it)
        {
            if (Material* pMat = it->FindMaterialByName(findName, bRecursive))
            {
                return pMat;
            }
        }
    }

    return 0;
}

//----------------------------------------
void
Window::SetFrameMaterial(WindowFrame frameIdx, Material* pMaterial)
{
    NW_ASSERTMSG(frameIdx < WINDOWFRAME_MAX, "out of bounds: frameIdx[%u] < WINDOWFRAME_MAX for Window[%s]", frameIdx, GetName());

    if (m_Frames[frameIdx].pMaterial == pMaterial)
    {
        return;
    }

    if (m_Frames[frameIdx].pMaterial != NULL &&
        !m_Frames[frameIdx].pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_Frames[frameIdx].pMaterial);
    }

    m_Frames[frameIdx].pMaterial = pMaterial;
}

//----------------------------------------
void
Window::SetContentMaterial(Material* pMaterial)
{
    if (m_pMaterial == pMaterial)
    {
        return;
    }

    if (m_pMaterial != NULL && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }

    m_pMaterial = pMaterial;
}

//----------------------------------------
void
Window::DrawSelf(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Window::DrawSelf");

    LoadMtx(drawInfo);

    switch (m_FrameMode)
    {
    case AROUND_MODE:
        DrawAroundFrameWindow(drawInfo);
        break;
    case HORIZONTAL_MODE:
        DrawHorizontalFrameWindow(drawInfo);
        break;
    case HORIZONTAL_NOCONTENT_MODE:
        DrawHorizontalFrameNocontentWindow(drawInfo);
        break;
    default:
        NW_ERR("Unknow frame mode error!");
        break;
    }
}

//----------------------------------------
void
Window::DrawAroundFrameWindow(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Window::DrawAroundFrameWindow");

    WindowFrameSize frameSize;
    frameSize.l = m_WindowSize.frameSize.l;
    frameSize.r = m_WindowSize.frameSize.r;
    frameSize.t = m_WindowSize.frameSize.t;
    frameSize.b = m_WindowSize.frameSize.b;

    const nw::math::VEC2 basePt = GetVtxPos();

    // frameの描画
    switch (m_FrameNum)
    {
    case 1:
        DrawFrame(drawInfo, basePt, m_Frames[WINDOWFRAME_LT], frameSize, GetGlobalAlpha());
        break;
    case 4:
        DrawFrame4(drawInfo, basePt, m_Frames, frameSize, GetGlobalAlpha());
        break;
    case 8:
        DrawFrame8(drawInfo, basePt, m_Frames, frameSize, GetGlobalAlpha());
        break;
    }

    // contentの描画
    if (! (m_WindowFlags & NOT_DRAW_CONTENT_ENABLED))
    {
        bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
        if (vertexColorEnabled || GetGlobalAlpha() != 255)
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), STANDARD_SHADER_VARIATION, true, this->GetGlobalMtx(), &this->GetSize());
            if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
            {
                internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                shhelp::SetVertexUniformReg<gfnd::ShaderInt>(drawInfo.GetUniformId(UNIFORM_uFrameSpec), internal::FRAMESPECFLAG_CONTENT);
            }
            else
            {
                internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                shhelp::SetVertexUniformReg<gfnd::ShaderInt>(drawInfo.GetUniformId(UNIFORM_uFrameSpec), internal::FRAMESPECFLAG_NORMAL);
            }
        }
        else
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), WITHOUT_VERTEX_COLOR_SHADER_VARIATION, true, this->GetGlobalMtx(), &this->GetSize());
            shhelp::SetVertexUniformReg<gfnd::ShaderInt>(drawInfo.GetUniformId(UNIFORM_uFrameSpec), internal::FRAMESPECFLAG_NORMAL);
        }
        DrawContent(drawInfo, basePt, frameSize, GetGlobalAlpha());
    }
}

//----------------------------------------
void
Window::DrawHorizontalFrameWindow(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalFrameWindow");

    WindowFrameSize frameSize;
    frameSize.l = m_WindowSize.frameSize.l;
    frameSize.r = m_WindowSize.frameSize.r;
    frameSize.t = m_WindowSize.frameSize.t;
    frameSize.b = m_WindowSize.frameSize.b;

    const nw::math::VEC2 basePt = GetVtxPos();

    // frameの描画
    switch (m_FrameNum)
    {
    case 1:
        DrawHorizontalFrame(drawInfo, basePt, m_Frames[WINDOWFRAME_LT], frameSize, GetGlobalAlpha());
        break;
    case 2:
    case 4:
        DrawHorizontalFrame2(drawInfo, basePt, m_Frames, frameSize, GetGlobalAlpha());
        break;
    }

    // contentの描画
    if (! (m_WindowFlags & NOT_DRAW_CONTENT_ENABLED))
    {
        m_pMaterial->SetupSubmaterialOf_Texture(drawInfo);
        m_pMaterial->SetupSubmaterialOf_TextureMatrix(drawInfo, this->GetGlobalMtx(), &this->GetSize());
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            shhelp::SetVertexUniformReg<gfnd::ShaderInt>(drawInfo.GetUniformId(UNIFORM_uFrameSpec), internal::FRAMESPECFLAG_CONTENT);
        }
        else
        {
            internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
            shhelp::SetVertexUniformReg<gfnd::ShaderInt>(drawInfo.GetUniformId(UNIFORM_uFrameSpec), internal::FRAMESPECFLAG_NORMAL);
        }
        DrawContent(drawInfo, basePt, frameSize, GetGlobalAlpha());
    }
}

//----------------------------------------
void
Window::DrawHorizontalFrameNocontentWindow(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalFrameNocontentWindow");

    WindowFrameSize frameSize;
    frameSize.r = m_WindowSize.frameSize.r;
    frameSize.l = GetSize().width - frameSize.r;
    frameSize.t = 0.0f;
    frameSize.b = 0.0f;

    const nw::math::VEC2 basePt = GetVtxPos();

    // frameの描画
    switch (m_FrameNum)
    {
    case 1:
        DrawHorizontalNocontentFrame(drawInfo, basePt, m_Frames[WINDOWFRAME_LT], frameSize, GetGlobalAlpha());
        break;
    case 2:
    case 4:
        DrawHorizontalNocontentFrame2(drawInfo, basePt, m_Frames, frameSize, GetGlobalAlpha());
        break;
    }
}

//----------------------------------------
void
Window::DrawContent(
    DrawInfo& drawInfo,
    const math::VEC2& basePt,
    const WindowFrameSize& frameSize,
    u8 alpha
)
{
    NW_UNUSED_VARIABLE(alpha)

    math::VEC2 pos(
        basePt.x + frameSize.l,
        basePt.y - frameSize.t);
    Size size(
        GetSize().width - frameSize.l - frameSize.r,
        GetSize().height - frameSize.t - frameSize.b);

    if (m_WindowFlags & CONTENT_INFLATION_ENABLED)
    {
        const f32 WINDOWINFL_FP_RECIP_SCALING_FACTOR = 1.0f / static_cast<f32>(nw::lyt::WINDOWINFL_FP_SCALING_FACTOR);

        f32 winSizeL = static_cast<f32>(m_WindowSize.l) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        f32 winSizeR = static_cast<f32>(m_WindowSize.r) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        f32 winSizeT = static_cast<f32>(m_WindowSize.t) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        f32 winSizeB = static_cast<f32>(m_WindowSize.b) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;

        pos.x -= winSizeL;
        pos.y += winSizeT;
        size.width += winSizeL + winSizeR;
        size.height += winSizeT + winSizeB;
    }

    internal::DrawQuadWithTexCoords(
        drawInfo,
        pos,
        size,
        m_Content.texCoordAry.GetSize(),
        m_Content.texCoordAry.GetArray());
}

//----------------------------------------
void
Window::DrawFrame(
    DrawInfo& drawInfo,
    const nw::math::VEC2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    u8 alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawFrame");

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapNum() == 0)
    {
        return;
    }

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
            internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
        #endif
    }

    nw::math::VEC2 polPt;
    Size polSize;

    wfnd::GetLTFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupFrame4Transform(drawInfo, WINDOWFRAME_LT, TEXTUREFLIP_NONE, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);

    wfnd::GetRTFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupFrame4Transform(drawInfo, WINDOWFRAME_RT, TEXTUREFLIP_FLIPH, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);

    wfnd::GetRBFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupFrame4Transform(drawInfo, WINDOWFRAME_RB, TEXTUREFLIP_ROTATE180, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);

    wfnd::GetLBFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupFrame4Transform(drawInfo, WINDOWFRAME_LB, TEXTUREFLIP_FLIPV, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);
}

//----------------------------------------
void
Window::DrawHorizontalFrame(
    DrawInfo& drawInfo,
    const nw::math::VEC2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    u8 alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalFrame");

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapNum() == 0)
    {
        return;
    }

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
        #endif
    }

    nw::math::VEC2 polPt;
    Size polSize;

    wfnd::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalFrameTransform(drawInfo, WINDOWFRAME_LT, TEXTUREFLIP_NONE, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);

    wfnd::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalFrameTransform(drawInfo, WINDOWFRAME_RT, TEXTUREFLIP_FLIPH, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad_Repeat(drawInfo, polPt, polSize);
}

//----------------------------------------
void
Window::DrawHorizontalNocontentFrame(
    DrawInfo& drawInfo,
    const nw::math::VEC2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    u8 alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalNocontentFrame");

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapNum() == 0)
    {
        return;
    }

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
        #endif
    }

    nw::math::VEC2 polPt;
    Size polSize;

    wfnd::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalNocontextFrameTransform(drawInfo, WINDOWFRAME_LT, TEXTUREFLIP_NONE, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPt, polSize);

    wfnd::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalNocontextFrameTransform(drawInfo, WINDOWFRAME_RT, TEXTUREFLIP_FLIPH, m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad_Repeat(drawInfo, polPt, polSize);
}

//----------------------------------------
void
Window::DrawHorizontalFrame2(
    DrawInfo& drawInfo,
    const nw::math::VEC2& basePt,
    const Frame* frames,
    const WindowFrameSize& frameSize,
    u8 alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalFrame2");

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frames[WINDOWFRAME_LT].pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
            internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
        }
    }
    else
    {
        frames[WINDOWFRAME_LT].pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
        #endif
    }

    nw::math::VEC2 polPtL;
    Size polSizeL;
    wfnd::GetHorizontalLeftFrameSize(&polPtL, &polSizeL, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalFrameTransform(drawInfo, WindowFrame(WINDOWFRAME_LT), frames[WINDOWFRAME_LT].GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPtL, polSizeL);

    frames[WINDOWFRAME_RT].pMaterial->SetupSubmaterialOf_Texture(drawInfo);

    nw::math::VEC2 polPtR;
    Size polSizeR;
    wfnd::GetHorizontalRightFrameSize(&polPtR, &polSizeR, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalFrameTransform(drawInfo, WindowFrame(WINDOWFRAME_RT), frames[WINDOWFRAME_RT].GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad_Repeat(drawInfo, polPtR, polSizeR);
}

//----------------------------------------
void
Window::DrawHorizontalNocontentFrame2(
    DrawInfo&         drawInfo,
    const nw::math::VEC2&       basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    u8                      alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawHorizontalNocontentFrame2");

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frames[WINDOWFRAME_LT].pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
            internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
        }
    }
    else
    {
        frames[WINDOWFRAME_LT].pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
        {
            internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
        }
        #endif
    }

    nw::math::VEC2 polPtL;
    Size polSizeL;
    wfnd::GetHorizontalLeftFrameSize(&polPtL, &polSizeL, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalNocontextFrameTransform(drawInfo, WindowFrame(WINDOWFRAME_LT), frames[WINDOWFRAME_LT].GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad(drawInfo, polPtL, polSizeL);

    frames[WINDOWFRAME_RT].pMaterial->SetupSubmaterialOf_Texture(drawInfo);

    nw::math::VEC2 polPtR;
    Size polSizeR;
    wfnd::GetHorizontalRightFrameSize(&polPtR, &polSizeR, basePt, GetSize(), frameSize);
    wfnd::SetupHorizontalNocontextFrameTransform(drawInfo, WindowFrame(WINDOWFRAME_RT), frames[WINDOWFRAME_RT].GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
    internal::DrawQuad_Repeat(drawInfo, polPtR, polSizeR);
}

//----------------------------------------
void
Window::DrawFrame4(
    DrawInfo&         drawInfo,
    const nw::math::VEC2&       basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    u8                      alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawFrame4");

    nw::math::VEC2 polPt[WINDOWFRAME_RB + 1];
    Size polSize[WINDOWFRAME_RB + 1];
    bool bRepeat = false;

    wfnd::GetLTFrameSize(&polPt[WINDOWFRAME_LT], &polSize[WINDOWFRAME_LT], basePt, GetSize(), frameSize);
    wfnd::GetRTFrameSize(&polPt[WINDOWFRAME_RT], &polSize[WINDOWFRAME_RT], basePt, GetSize(), frameSize);
    wfnd::GetLBFrameSize(&polPt[WINDOWFRAME_LB], &polSize[WINDOWFRAME_LB], basePt, GetSize(), frameSize);
    wfnd::GetRBFrameSize(&polPt[WINDOWFRAME_RB], &polSize[WINDOWFRAME_RB], basePt, GetSize(), frameSize);

    for (int i = WINDOWFRAME_LT; i <= WINDOWFRAME_RB; ++i)
    {
        const Frame& frame = frames[i];

        if (frame.pMaterial && frame.pMaterial->GetTexMapNum() > 0)
        {
            bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);

            if (!bRepeat)
            {
                if (vertexColorEnabled || GetGlobalAlpha() != 255)
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                    }
                }
                else
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

                    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                    }
                    #endif
                }
                wfnd::SetupFrame4Transform(drawInfo, WindowFrame(i), frame.GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
                internal::DrawQuad(drawInfo, polPt[i], polSize[i]);
                bRepeat = true;
            }
            else
            {
                if (m_WindowFlags & ONEMATERIAL_FOR_ALLFRAME_ENABLED)
                {
                    frame.pMaterial->SetupSubmaterialOf_Texture(drawInfo);
                }
                else if (vertexColorEnabled || GetGlobalAlpha() != 255)
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                    }
                }
                else
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

                    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                    }
                    #endif
                }
                wfnd::SetupFrame4Transform(drawInfo, WindowFrame(i), frame.GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
                internal::DrawQuad_Repeat(drawInfo, polPt[i], polSize[i]);
            }
        }
    }
}

//----------------------------------------
void
Window::DrawFrame8(
    DrawInfo&         drawInfo,
    const nw::math::VEC2&       basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    u8                      alpha
)
{
    NW_PROFILE("nw::lyt::Window::DrawFrame8");

    const f32 x0 = basePt.x;
    const f32 x1 = basePt.x + frameSize.l;
    const f32 x2 = basePt.x + this->GetSize().width - frameSize.r;

    const f32 y0 = basePt.y;
    const f32 y1 = basePt.y - frameSize.t;
    const f32 y2 = basePt.y - this->GetSize().height + frameSize.b;

    const f32 w0 = frameSize.l;
    const f32 w1 = this->GetSize().width - frameSize.l - frameSize.r;
    const f32 w2 = frameSize.r;

    const f32 h0 = frameSize.t;
    const f32 h1 = this->GetSize().height - frameSize.t - frameSize.b;
    const f32 h2 = frameSize.b;

    const wfnd::Rect frameRect[WINDOWFRAME_MAX] =
    {
        { x0, y0, w0, h0 }, // LT
        { x2, y0, w2, h0 }, // RT
        { x0, y2, w0, h2 }, // LB
        { x2, y2, w2, h2 }, // RB
        { x0, y1, w0, h1 }, // L
        { x2, y1, w2, h1 }, // R
        { x1, y0, w1, h0 }, // T
        { x1, y2, w1, h2 }, // B
    };

    bool bRepeat = false;

    for (int i = 0; i < WINDOWFRAME_MAX; ++i)
    {
        const Frame& frame = frames[i];

        if (frame.pMaterial->GetTexMapNum() > 0)
        {
            bool vertexColorEnabled = internal::TestVertexColorEnabled(m_Content.vtxColors);
            if (!bRepeat)
            {
                if (vertexColorEnabled || GetGlobalAlpha() != 255)
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                    }
                }
                else
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

                    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                    }
                    #endif
                }
                wfnd::SetupFrameTransform(drawInfo, WindowFrame(i), frame.GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
                internal::DrawQuad(drawInfo, frameRect[i].Position(), frameRect[i].Size());
                bRepeat = true;
            }
            else
            {
                if (m_WindowFlags & ONEMATERIAL_FOR_ALLFRAME_ENABLED)
                {
                    frame.pMaterial->SetupSubmaterialOf_Texture(drawInfo);
                }
                else if (vertexColorEnabled || GetGlobalAlpha() != 255)
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, STANDARD_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                        internal::SetupVertexColor(drawInfo, m_Content.vtxColors);
                    }
                }
                else
                {
                    frame.pMaterial->SetupGraphics(drawInfo, alpha, WITHOUT_VERTEX_COLOR_SHADER_VARIATION, false, this->GetGlobalMtx(), &this->GetSize());

                    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    if (m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED)
                    {
                        internal::SetupFrameSize(drawInfo, GetSize(), frameSize);
                    }
                    #endif
                }
                wfnd::SetupFrameTransform(drawInfo, WindowFrame(i), frame.GetTextureFlip(), m_WindowFlags & OVERALL_VERTEX_COLOR_ENABLED);
                internal::DrawQuad_Repeat(drawInfo, frameRect[i].Position(), frameRect[i].Size());
            }
        }
    }
}

} // namespace lyt
} // namespace nw
