﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Util.h>
#include <nw/lyt/lyt_Bounding.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Group.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/lyt/lyt_Bounding.h>
#include <nw/lyt/lyt_Parts.h>
#include <nw/lyt/lyt_Picture.h>
#include <nw/lyt/lyt_TextBox.h>
#include <nw/lyt/lyt_Window.h>
#include <cafe/gx2/gx2Enum.h>
#include <cafe/gx2/gx2surface.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <cafe/gx2/gx2Constant.h>
#include <cafe/gx2/gx2Misc.h>
#include <cafe/gx2/gx2Shaders.h>
#ifdef _WIN32
#ifndef WIN32
#define WIN32
#endif
#endif
#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4063)
#include <nw/font/font_TextureTilingConverter.h>
#pragma warning(pop)
#include <sdk_ver.h>
#if CAFE_OS_SDK_VERSION < 20903
using namespace TexUtils;
#endif
#endif

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#include <nw/lyt/lyt_ShaderConnection.h>
#endif

namespace
{

bool
Contains(
    const nw::ut::Rect&   rect,
    const nw::math::VEC2& point
)
{
    return rect.left <= point.x && point.x <= rect.right && rect.bottom <= point.y && point.y <= rect.top;
}

} // namespace {no-name}

namespace
{

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
//########################################
struct TexSpec
{
    int lytFormat;
    GLenum internalformat;
    GLenum format;
    GLenum type;
    int minSize;
    bool compressed;
    GX2SurfaceFormat gx2_format;
    s32 bit_by_pixel;
};

//----------------------------------------------------------

void SetupGLTextureSwzzling_(nw::lyt::TexFormat format)
{
    if (format == nw::lyt::TEXFORMAT_BC4L) {
        // R成分を RGBの要素として利用する
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_R, GL_RED);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_G, GL_RED);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_B, GL_RED);
    } else if (format == nw::lyt::TEXFORMAT_BC4A) {
        // R成分をアルファとして利用する
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_A, GL_RED);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_R, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_G, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    } else if (format == nw::lyt::TEXFORMAT_A8) {
        // R成分をアルファとして利用する
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_A, GL_ALPHA);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_R, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_G, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    } else if (format == nw::lyt::TEXFORMAT_BC5) {
        // G成分をカラー(明度)として利用する
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_R, GL_GREEN);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_G, GL_GREEN);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_B, GL_GREEN);
    } else if (format == nw::lyt::TEXFORMAT_RGB565_INDIRECT) {
        // RとBがインダイレクトのずらし量、Gがアルファとして利用する
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_A, GL_GREEN);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_R, GL_RED);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_G, GL_BLUE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    }
}

//########################################
#elif defined(NW_PLATFORM_CAFE)
//########################################

//----------------------------------------
//! テクスチャの設定内容です。
struct GX2TextureDescription
{
    GX2TextureDescription(const void* image, u32 w, u32 h, GX2SurfaceFormat fmt, GX2TileMode tm, u32 sp, GX2CompSel selector)
    : textureImage(image)
    , width(w)
    , height(h)
    , format(fmt)
    , tileMode(tm)
    , swizzlePattern(sp)
    , componentSelector(selector)
    {}

    const void* textureImage;
    u32 width;
    u32 height;
    GX2SurfaceFormat format;
    GX2TileMode tileMode;
    u32 swizzlePattern;
    GX2CompSel componentSelector;
};

//----------------------------------------
void
InitializeTexture(GX2Texture* texture, const GX2TextureDescription& description)
{
    NW_ASSERT_NOT_NULL(texture);

    GX2InitTexture(
        texture,
        description.width,
        description.height,
        1,
        1,
        description.format,
        GX2_SURFACE_DIM_2D);
    texture->surface.tileMode = description.tileMode;
    texture->surface.swizzle |= description.swizzlePattern << 8;
    GX2CalcSurfaceSizeAndAlignment(&texture->surface);

    void* texAddr = const_cast<void*>(description.textureImage);
    NW_ASSERT_NOT_NULL(texAddr);
    NW_WARNING((reinterpret_cast<u32>(texAddr) % texture->surface.alignment) == 0, "Texture alignment is invalid!");

    GX2InitTexturePtrs(texture, texAddr, 0);
    GX2InitTextureCompSel(texture, description.componentSelector);
    GX2InitTextureRegs(texture);

    #if 0
    NW_LOG("## TextureDescription ##\n");
    NW_LOG("  texture Image = 0x%08x\n", description.textureImage);
    NW_LOG("  width = %d\n", texture->surface.width);
    NW_LOG("  height = %d\n", texture->surface.height);
    NW_LOG("  depth = %d\n", texture->surface.depth);
    NW_LOG("  pitch = %d\n", texture->surface.pitch);
    NW_LOG("  alignment = %d\n", texture->surface.alignment);
    NW_LOG("  format = 0x%08x\n", description.format);
    NW_LOG("  imageSize = %d\n", texture->surface.imageSize);
    NW_LOG("\n");
    #endif
}

//########################################
#endif
//########################################

} // namespace {no-name}

namespace nw {
namespace lyt {

//----------------------------------------
bool
LoadTexture(TextureInfo* textureInfo, const void* pImgRes, u32 size)
{
    NW_ASSERT_NOT_NULL(textureInfo);
    NW_ASSERT_NOT_NULL(pImgRes);
    NW_ASSERT((u32)(pImgRes) % 128 == 0);

    const int FILEHEADER_ALIGNMENT = 4;

    // アライメントのためテクスチャイメージはリソースファイルの先頭に配置。
    const void* pixels = pImgRes;

    // ファイルの末尾 4 バイトがイメージサイズ。
    const res::ImageSize* pImageSize
        = internal::ConvertOffsToPtr<res::ImageSize>(pImgRes, size - sizeof(u32));
    u32 imageSize = pImageSize->imageSize;
    NW_ASSERTMSG(imageSize < size, "imageSize(%d) < size(%d)", imageSize, size);

    const ut::BinaryFileHeader* pFileHead
        = internal::ConvertOffsToPtr<ut::BinaryFileHeader>(
            pImgRes,
            ut::RoundUp(imageSize, FILEHEADER_ALIGNMENT));

    if (!ut::IsValidBinaryFile(pFileHead, res::FILESIGNATURE_FLIM, res::BinaryFileFormatVersion))
    {
        NW_ERR("not valid layout image file.");
    }

    const res::Image* pImage = 0;

    const ut::BinaryBlockHeader* pBlockHead = NULL;
    for (int i = 0; i < pFileHead->dataBlocks; ++i)
    {
        pBlockHead = ut::GetNextBinaryBlockHeader(pFileHead, pBlockHead);
        NW_ASSERT_NOT_NULL(pBlockHead);

        ut::SigWord kind = pBlockHead->kind;
        switch (kind)
        {
        case res::DATABLOCKKIND_IMAGE:
            pImage = reinterpret_cast<const res::Image *>(pBlockHead);
            break;
        }
    }

    if (pImage == NULL)
    {
        NW_WARNING(false, "file does not contain texture image.");
        return false;
    }

    u16 width = pImage->width;
    u16 height = pImage->height;

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
//########################################

    static const TexSpec texSpec[TEXFORMAT_MAX] =
    {
        { TEXFORMAT_L8, GL_LUMINANCE8, GL_LUMINANCE, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TC_R8_UNORM, 8 },
        { TEXFORMAT_A8, GL_ALPHA8, GL_ALPHA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TC_R8_UNORM, 8 },
        { TEXFORMAT_LA4, GL_LUMINANCE4_ALPHA4, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_T_R4_G4_UNORM, 8 },
        { TEXFORMAT_LA8, GL_LUMINANCE8_ALPHA8, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, 16 },
        { TEXFORMAT_HILO8, GL_ALPHA16, GL_ALPHA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, 16 },
        { TEXFORMAT_RGB565,  GL_RGB, GL_RGB, GL_UNSIGNED_SHORT_5_6_5, 8, false, GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, 16 },
        { TEXFORMAT_RGB8, GL_RGB8, GL_RGB, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, 32 },
        { TEXFORMAT_RGB5A1, GL_RGB5_A1, GL_RGBA, GL_UNSIGNED_SHORT_5_5_5_1, 8, false, GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM, 16 },
        { TEXFORMAT_RGBA4, GL_RGBA4, GL_RGBA, GL_UNSIGNED_SHORT_4_4_4_4, 8, false, GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM, 16 },
        { TEXFORMAT_RGBA8, GL_RGBA8, GL_RGBA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, 32 },
        { TEXFORMAT_ETC1, GL_RGB_S3TC, GL_RGB, 0 /* N/A */, 16, true, GX2_SURFACE_FORMAT_T_BC1_UNORM, 4 },
        { TEXFORMAT_ETC1A4, GL_RGBA_S3TC, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC3_UNORM, 8 },
        { TEXFORMAT_BC1, GL_COMPRESSED_RGBA_S3TC_DXT1_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC1_UNORM, 4 }, // アルファありを指定するとCafeと同じ挙動になる
        { TEXFORMAT_BC2, GL_COMPRESSED_RGBA_S3TC_DXT3_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC2_UNORM, 8 },
        { TEXFORMAT_BC3, GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC3_UNORM, 8 },
        { TEXFORMAT_BC4L, GL_COMPRESSED_LUMINANCE_LATC1_EXT, GL_LUMINANCE8, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC4_UNORM, 4 },
        { TEXFORMAT_BC4A, GL_COMPRESSED_LUMINANCE_LATC1_EXT, GL_ALPHA8, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC4_UNORM, 4 }, // NG
        { TEXFORMAT_BC5, GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT, GL_LUMINANCE8_ALPHA8, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC5_UNORM, 8 },
        { TEXFORMAT_L4, GL_LUMINANCE4, GL_LUMINANCE, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_T_BC4_UNORM, 4 },
        { TEXFORMAT_A4, GL_ALPHA4, GL_ALPHA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_T_BC4_UNORM, 4 },
        { TEXFORMAT_RGBA8_SRGB, GL_SRGB8_ALPHA8_EXT, GL_RGBA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, 32 },
        { TEXFORMAT_BC1_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC1_SRGB, 4 },
        { TEXFORMAT_BC1_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC2_SRGB, 8 },
        { TEXFORMAT_BC1_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT, GL_RGBA, 0 /* N/A */, 8, true, GX2_SURFACE_FORMAT_T_BC3_SRGB, 8 },
        { TEXFORMAT_RGB10_A2, GL_RGB10_A2, GL_RGBA, GL_UNSIGNED_BYTE, 8, false, GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, 32 },
        { TEXFORMAT_RGB565_INDIRECT, GL_RGB, GL_RGB, GL_UNSIGNED_SHORT_5_6_5, 8, false, GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, 16 },
    };

    GLuint texName = TextureInfo::INVALID;

    {
        // テクスチャ名を取得。
        glGenTextures(1, &texName);
        // テクスチャオブジェクトを生成。
        glBindTexture(GL_TEXTURE_2D, texName);
        NW_GL_ASSERT();

        const GX2TileMode tileMode = static_cast<GX2TileMode>(pImage->tileModeAndSwizzlePattern & 0x1f);
        const bool needToConvertTiling = tileMode != GX2_TILE_MODE_DEFAULT;
        if(needToConvertTiling)
        {
            GX2Surface dst_surface;

            TC2Config config = {};
            config.gbTilingConfig = 0;

            GX2Surface src_surface;
            src_surface.dim = GX2_SURFACE_DIM_2D;
            src_surface.width = width;
            src_surface.height = height;
            src_surface.depth = 1;
            src_surface.numMips = 1;
            src_surface.format = texSpec[pImage->format].gx2_format;
            src_surface.aa = GX2_AA_MODE_1X;
            src_surface.use = GX2_SURFACE_USE_TEXTURE;
            src_surface.imageSize = imageSize;
            src_surface.imagePtr = const_cast<void*>(pixels);
            src_surface.mipSize = 0;
            src_surface.mipPtr = NULL;
            src_surface.tileMode = tileMode;
            src_surface.swizzle = ((pImage->tileModeAndSwizzlePattern >> 5) & 0x7) << 8;
            src_surface.alignment = 1;
            src_surface.pitch = 128;

            config.gpu = GPU_Cafe;

            nw::font::internal::TextureTilingConverter::Initialize(config);
            nw::font::internal::TextureTilingConverter::ConvertTiling(src_surface, dst_surface);

            if (texSpec[pImage->format].compressed)
            {
                glCompressedTexImage2D(GL_TEXTURE_2D, 0, texSpec[pImage->format].internalformat, width, height, 0, dst_surface.imageSize, dst_surface.imagePtr);
                NW_GL_ASSERT();
            }
            else
            {
                // 幅が4バイトアラインメントでないテクスチャが正しく表示されないため、
                // 1に設定する。デフォルトの4に戻す必要があるかどうかは要検討。
                glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
                glTexImage2D(GL_TEXTURE_2D, 0, texSpec[pImage->format].internalformat, width, height, 0, texSpec[pImage->format].format, texSpec[pImage->format].type, dst_surface.imagePtr);
                NW_GL_ASSERT();
            }

            // データはGL側が内部にコピーするので、タイリングを行う際に作ったsurfaceは削除してよい
            nw::font::internal::TextureTilingConverter::DestorySurface(dst_surface);
            nw::font::internal::TextureTilingConverter::Finalize();

        }else{

            if (texSpec[pImage->format].compressed)
            {
                glCompressedTexImage2D(GL_TEXTURE_2D, 0, texSpec[pImage->format].internalformat, width, height, 0, imageSize, pixels);
                NW_GL_ASSERT();
            }
            else
            {
                // 幅が4バイトアラインメントでないテクスチャが正しく表示されないため、
                // 1に設定する。デフォルトの4に戻す必要があるかどうかは要検討。
                glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
                glTexImage2D( GL_TEXTURE_2D, 0, texSpec[pImage->format].internalformat, width, height, 0, texSpec[pImage->format].format, texSpec[pImage->format].type, pixels);
                NW_GL_ASSERT();
            }
        }

        // 必要に応じてSWIZZLE 設定を行います。
        SetupGLTextureSwzzling_(static_cast<nw::lyt::TexFormat>(pImage->format));
    }

    textureInfo->Set(
        texName,
        TexSize(width, height),
        TexFormat(pImage->format));

    return true;
//########################################
#elif defined(NW_PLATFORM_CAFE)
//########################################

    static const GX2SurfaceFormat s_TextureFormatTable[nw::lyt::TEXFORMAT_MAX] =
    {
        GX2_SURFACE_FORMAT_TC_R8_UNORM, // TEXFORMAT_L8,
        GX2_SURFACE_FORMAT_TC_R8_UNORM, // TEXFORMAT_A8,
        GX2_SURFACE_FORMAT_T_R4_G4_UNORM, // TEXFORMAT_LA4,
        GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, // TEXFORMAT_LA8,
        GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, // TEXFORMAT_HILO8,
        GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, // TEXFORMAT_RGB565,
        GX2_SURFACE_FORMAT_INVALID, // not support TEXFORMAT_RGB8,
        GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM, // TEXFORMAT_RGB5A1,
        GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM, // TEXFORMAT_RGBA4,
        GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, // TEXFORMAT_RGBA8,
        GX2_SURFACE_FORMAT_INVALID, // TEXFORMAT_ETC1,
        GX2_SURFACE_FORMAT_INVALID, // TEXFORMAT_ETC1A4,
        GX2_SURFACE_FORMAT_T_BC1_UNORM, // TEXFORMAT_BC1,
        GX2_SURFACE_FORMAT_T_BC2_UNORM, // TEXFORMAT_BC2,
        GX2_SURFACE_FORMAT_T_BC3_UNORM, // TEXFORMAT_BC3,
        GX2_SURFACE_FORMAT_T_BC4_UNORM, // TEXFORMAT_BC4L,
        GX2_SURFACE_FORMAT_T_BC4_UNORM, // TEXFORMAT_BC4A,
        GX2_SURFACE_FORMAT_T_BC5_UNORM, // TEXFORMAT_BC5,
        GX2_SURFACE_FORMAT_INVALID, // not support TEXFORMAT_L4,
        GX2_SURFACE_FORMAT_INVALID, // not support TEXFORMAT_A4,
        GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, // TEXFORMAT_RGBA8_SRGB,
        GX2_SURFACE_FORMAT_T_BC1_SRGB, // TEXFORMAT_BC1_SRGB,
        GX2_SURFACE_FORMAT_T_BC2_SRGB, // TEXFORMAT_BC2_SRGB,
        GX2_SURFACE_FORMAT_T_BC3_SRGB, // TEXFORMAT_BC3_SRGB,
        GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, // TEXFORMAT_RGB10_A2,
        GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, // TEXFORMAT_RGB565_INDIRECT,
    };

    static const GX2CompSel s_TextureComponetSelectorTable[nw::lyt::TEXFORMAT_MAX] =
    {
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_C_1),
        GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R),
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G),
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G),
        GX2_COMP_SEL_XY01,
        GX2_GET_COMP_SEL(GX2_COMPONENT_Z_B, GX2_COMPONENT_Y_G, GX2_COMPONENT_X_R, GX2_COMPONENT_C_1),
        GX2_COMP_SEL_XYZ1,
        GX2_COMP_SEL_XYZW,
        GX2_SWAP_8_IN_32(static_cast<u32>(GX2_COMP_SEL_XYZW)),
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZ1,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_C_1),
        GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R),
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G),
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_C_1),
        GX2_GET_COMP_SEL(GX2_COMPONENT_C_0, GX2_COMPONENT_C_0, GX2_COMPONENT_C_0, GX2_COMPONENT_X_R),
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_COMP_SEL_XYZW,
        GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_Z_B, GX2_COMPONENT_C_1, GX2_COMPONENT_Y_G),
    };

    //NW_LOG("#### format = %d\n", pImage->format);

    NW_ASSERT_MAXLT(pImage->format, nw::lyt::TEXFORMAT_MAX);
    const GX2SurfaceFormat format = s_TextureFormatTable[pImage->format];
    NW_ASSERT(format != GX2_SURFACE_FORMAT_INVALID);

    InitializeTexture(
        &textureInfo->GetTexture(),
        GX2TextureDescription(
            pixels,
            width,
            height,
            format,
            static_cast<GX2TileMode>(pImage->tileModeAndSwizzlePattern & 0x1f),
            (pImage->tileModeAndSwizzlePattern >> 5) & 0x7,
            s_TextureComponetSelectorTable[pImage->format])
        );

    textureInfo->Set(
        1,
        TexSize(width, height),
        TexFormat(pImage->format));

    return true;

#else
//########################################
#error "Not support platform."
#endif
//########################################
}

bool
LoadArchiveShader(ArchiveShaderInfo* pShaderInfo, const void* pShaderRes, u32 size, const char *name)
{
#if defined(NW_PLATFORM_CAFE)
    GX2VertexShader* vertexShader = NULL;
    GX2PixelShader* pixelShader = NULL;
    GX2FetchShader* fetchShader = NULL;

    for (u32 i = 0; i < SHADER_VARIATION_QUANTITY; i++)
    {
        {
            u32 headerSize = GFDGetVertexShaderHeaderSize(i, pShaderRes);
            u32 programSize = GFDGetVertexShaderProgramSize(i, pShaderRes);
            if (!headerSize || !programSize)
            {
                return false;
            }
            GX2VertexShader* header = static_cast<GX2VertexShader*>(Layout::AllocMemory(headerSize, PPC_IO_BUFFER_ALIGN));
            void* program = static_cast<void*>(Layout::AllocMemory(programSize, GX2_SHADER_ALIGNMENT));
            u32 ret = GFDGetVertexShader(header, program, i, pShaderRes);
            if (!ret)
            {
                NW_LOG("Warning: Invalid Vertex Shader :%d", ret);
                if (header)
                {
                    Layout::FreeMemory(header);
                }
                if (program)
                {
                    Layout::FreeMemory(program);
                }
                return false;
            }
            GX2Invalidate(GX2_INVALIDATE_CPU, header->shaderPtr, header->shaderSize);
            vertexShader = header;
        }

        {
            u32 headerSize = GFDGetPixelShaderHeaderSize(i, pShaderRes);
            u32 programSize = GFDGetPixelShaderProgramSize(i, pShaderRes);
            if (!headerSize || !programSize)
            {
                return false;
            }
            GX2PixelShader* header = static_cast<GX2PixelShader*>(Layout::AllocMemory(headerSize, PPC_IO_BUFFER_ALIGN));
            void* program = static_cast<void*>(Layout::AllocMemory(programSize, GX2_SHADER_ALIGNMENT));
            u32 ret = GFDGetPixelShader(header, program, i, pShaderRes);
            if (!ret)
            {
                NW_LOG("Warning: Invalid Pixel Shader :%d", ret);
                if (header)
                {
                    Layout::FreeMemory(header);
                }
                if (program)
                {
                    Layout::FreeMemory(program);
                }
                return false;
            }
            GX2Invalidate(GX2_INVALIDATE_CPU, header->shaderPtr, header->shaderSize);
            pixelShader = header;
        }

        const int VERTEX_ATTRIBUTE_QUANTITY = 1;
        ShaderConnection* connection = Layout::NewObj<ShaderConnection>();
        connection->m_Attributes = static_cast<GX2AttribStream*>(Layout::AllocMemory(sizeof(GX2AttribStream) * VERTEX_ATTRIBUTE_QUANTITY));
        NW_ASSERT_NOT_NULL(connection->m_Attributes);
        std::uninitialized_fill_n(connection->m_Attributes, VERTEX_ATTRIBUTE_QUANTITY, GX2AttribStream());
        connection->m_UniformIds = static_cast<gfnd::ShaderUniformId*>(Layout::AllocMemory(sizeof(gfnd::ShaderUniformId) * nw::lyt::UNIFORM_MAX));
        NW_ASSERT_NOT_NULL(connection->m_UniformIds);
        std::uninitialized_fill_n(connection->m_UniformIds, static_cast<u32>(nw::lyt::UNIFORM_MAX), GX2_UNIFORM_VAR_INVALID_OFFSET);
#if defined(NW_DISPLAY_LIST_ENABLED)
        {
            void* displaylist_memory = Layout::AllocMemory(ShaderSlotDescription::DISPLAY_LIST_CAPACITY_DEFAULT, gfnd::DisplayList::ALIGNMENT);
            connection->m_ShaderDisplayList.Initialize(displaylist_memory, ShaderSlotDescription::DISPLAY_LIST_CAPACITY_DEFAULT);
        }
#endif

        connection->m_VertexShader = vertexShader;
        connection->m_PixelShader = pixelShader;
        connection->m_IsMultiTextureEnabled = true;

        GX2AttribStream* attribute = connection->m_Attributes;
        u32 location = static_cast<u32>(GX2GetVertexAttribVarLocation(vertexShader, "aVertexIndex"));
        GX2InitAttribStream(attribute, location, 0, 0, GX2_ATTRIB_FORMAT_32_32_FLOAT);

        fetchShader = &connection->m_FetchShader;

        const int MAX_ATTRIBUTE_STREAM_QUANTITY = 1;
        int fetchShaderSize = GX2CalcFetchShaderSize(MAX_ATTRIBUTE_STREAM_QUANTITY);
        connection->m_FetchShaderBuffer = Layout::AllocMemory(fetchShaderSize, GX2_SHADER_ALIGNMENT);

        GX2InitFetchShader(fetchShader, connection->m_FetchShaderBuffer, MAX_ATTRIBUTE_STREAM_QUANTITY, connection->m_Attributes);
        GX2Invalidate(GX2_INVALIDATE_CPU, connection->m_FetchShaderBuffer, fetchShaderSize);

        bool binded = nw::gfnd::shader_helper::CacheUniforms(
            connection->m_UniformIds,
            nw::lyt::UNIFORM_MAX,
            nw::lyt::GraphicsResource::GetUniformList(name, i == WITHOUT_VERTEX_COLOR_SHADER_VARIATION),
            connection->m_VertexShader,
            connection->m_PixelShader);
        NW_ASSERT(binded);

        connection->m_ShaderDisplayList.Clear();
        NW_MAKE_DISPLAY_LIST(cache, connection->m_ShaderDisplayList)
        {
            GX2SetShaders(&connection->m_FetchShader, connection->m_VertexShader, connection->m_PixelShader);
        }

        pShaderInfo->SetConnection(i, connection);
    }
    return true;
#else
    NW_UNUSED_VARIABLE(pShaderInfo);
    NW_UNUSED_VARIABLE(pShaderRes);
    NW_UNUSED_VARIABLE(size);
    NW_UNUSED_VARIABLE(name);
    return false;
#endif
}

void
FreeArchiveShader(ArchiveShaderInfo* pShaderInfo)
{
#if defined(NW_PLATFORM_CAFE)
    for (s32 i = 0; i < SHADER_VARIATION_QUANTITY; i++)
    {
        ShaderConnection* connection = pShaderInfo->GetConnection(i);
        if (connection->m_VertexShader != NULL)
        {
            if (connection->m_VertexShader->shaderPtr != NULL)
            {
                Layout::FreeMemory(connection->m_VertexShader->shaderPtr);
            }
            Layout::FreeMemory(connection->m_VertexShader);
            connection->m_VertexShader = NULL;
        }
        if (connection->m_PixelShader != NULL)
        {
            if (connection->m_PixelShader->shaderPtr != NULL)
            {
                Layout::FreeMemory(connection->m_PixelShader->shaderPtr);
            }
            Layout::FreeMemory(connection->m_PixelShader);
            connection->m_PixelShader = NULL;
        }
        if (connection->m_Attributes != NULL)
        {
            Layout::FreeMemory(connection->m_Attributes);
            connection->m_Attributes = NULL;
        }
        if (connection->m_UniformIds != NULL)
        {
            Layout::FreeMemory(connection->m_UniformIds);
            connection->m_UniformIds = NULL;
        }
#if defined(NW_DISPLAY_LIST_ENABLED)
        {
            void* displaylist_memory = connection->m_ShaderDisplayList.Shutdown();
            if (displaylist_memory != NULL)
            {
                Layout::FreeMemory(displaylist_memory);
            }
        }
#endif
        if (connection->m_FetchShaderBuffer != NULL)
        {
            Layout::FreeMemory(connection->m_FetchShaderBuffer);
            connection->m_FetchShaderBuffer = NULL;
        }
        Layout::DeleteObj<ShaderConnection>(connection);
    }
#else
    NW_UNUSED_VARIABLE(pShaderInfo);
#endif
}

} // namespace lyt
} // namespace nw

namespace nw {
namespace lyt {

void
BindAnimation(
    Group*          pGroup,
    AnimTransform*  pAnimTrans,
    bool            bEnable
)
{
    pAnimTrans->BindGroup(pGroup);
    pAnimTrans->SetEnable(bEnable);
}

void
UnbindAnimation(
    Group*          pGroup,
    AnimTransform*  pAnimTrans
)
{
    pAnimTrans->UnbindGroup(pGroup);
}

bool
IsContain(
    Pane*               pPane,
    const math::VEC2&   pos
)
{
    math::MTX34 invGlbMtx;
    math::MTX34Inverse(&invGlbMtx, &pPane->GetGlobalMtx());       // ペインのグローバル行列の逆行列を求める

    math::VEC3 pos3(pos.x, pos.y, 0.f);
    math::VEC3 lclPos;
    math::VEC3Transform(&lclPos, &invGlbMtx, &pos3);

    return Contains(pPane->GetPaneRect(), math::VEC2(lclPos.x, lclPos.y));
}

Pane*
FindHitPane(
    Pane*               pPane,
    const math::VEC2&   pos
)
{
    // 非表示のペインはヒットチェックの対象としない。
    if (! pPane->IsVisible())
    {
        return 0;
    }

    // 子供のヒットチェック
    for (PaneList::ReverseIterator it = pPane->GetChildList().GetBeginReverseIter(); it != pPane->GetChildList().GetEndReverseIter(); ++it)
    {
        if (Pane *const ret = FindHitPane(&(*it), pos))
        {
            return ret;
        }
    }

    // Bounding ペインか
    if (nw::lyt::Bounding *const pBounding = nw::ut::DynamicCast<nw::lyt::Bounding*>(pPane))
    {
        if (IsContain(pBounding, pos))
        {
            return pBounding;  // 含まれている。
        }
    }

    return 0;
}

Pane*
FindHitPane(
    Layout*             pLayout,
    const math::VEC2&   pos
)
{
    return FindHitPane(pLayout->GetRootPane(), pos);
}

/*
    次のペインを返す。
*/
Pane*
GetNextPane(Pane* pPane)
{
    if (! pPane->GetChildList().IsEmpty())  // 子供がいれば、最初の子供を返す。
    {
        PaneList::Iterator paneIt = pPane->GetChildList().GetBeginIter();
        return &(*paneIt);
    }

    // 弟を探す。
    // 弟が無ければ、親の弟へとたどる。
    for(;;)
    {
        if (pPane->GetParent() == 0)    // 親が NULL の場合はルートペイン
        {
            return 0;
        }

        PaneList::Iterator nextIt = PaneList::GetIteratorFromPointer(pPane->m_Link.GetNext());
        PaneList::Iterator endIt = pPane->GetParent()->GetChildList().GetEndIter();
        if (nextIt != endIt)
        {
            break;
        }

        pPane = pPane->GetParent();
    }

    return PaneList::GetPointerFromNode(pPane->m_Link.GetNext());
}

Pane*
ClonePaneTree(const Pane* pPane)
{
    Pane* new_pane = NULL;
    const Picture* picture = ut::DynamicCast<const Picture*>(pPane);
    if (picture)
    {
        new_pane = Layout::NewObj<Picture, const Picture&>(*picture);
    }
    else
    {
        const TextBox* textBox = ut::DynamicCast<const TextBox*>(pPane);
        if (textBox)
        {
            new_pane = Layout::NewObj<TextBox, const TextBox&>(*textBox);
        }
        else
        {
            const Window* window = ut::DynamicCast<const Window*>(pPane);
            if (window)
            {
                new_pane = Layout::NewObj<Window, const Window&>(*window);
            }
            else
            {
                const Bounding* bounding = ut::DynamicCast<const Bounding*>(pPane);
                if (bounding)
                {
                    new_pane = Layout::NewObj<Bounding, const Bounding&>(*bounding);
                }
                else
                {
                    const Parts* parts = ut::DynamicCast<const Parts*>(pPane);
                    if (parts)
                    {
                        new_pane = Layout::NewObj<Parts, const Parts&>(*parts);
                    }
                    else
                    {
                        new_pane = Layout::NewObj<Pane, const Pane&>(*pPane);
                    }
                }
            }
        }
    }

    for (PaneList::ConstIterator it = pPane->GetChildList().GetBeginIter(); it != pPane->GetChildList().GetEndIter(); ++it)
    {
        Pane* child_tree = ClonePaneTree(&(*it));
        new_pane->AppendChild(child_tree);
    }

    return new_pane;
}

Pane*
ClonePaneTreeWithPartsLayout(const Pane* pPane, Layout* pPartsLayout /* = NULL */)
{
    Pane* new_pane = NULL;
    bool need_clone_child = true;
    const Picture* picture = ut::DynamicCast<const Picture*>(pPane);
    if (picture)
    {
        new_pane = Layout::NewObj<Picture, const Picture&>(*picture);
    }
    else
    {
        const TextBox* textBox = ut::DynamicCast<const TextBox*>(pPane);
        if (textBox)
        {
            new_pane = Layout::NewObj<TextBox, const TextBox&>(*textBox);
        }
        else
        {
            const Window* window = ut::DynamicCast<const Window*>(pPane);
            if (window)
            {
                new_pane = Layout::NewObj<Window, const Window&>(*window);
            }
            else
            {
                const Bounding* bounding = ut::DynamicCast<const Bounding*>(pPane);
                if (bounding)
                {
                    new_pane = Layout::NewObj<Bounding, const Bounding&>(*bounding);
                }
                else
                {
                    const Parts* parts = ut::DynamicCast<const Parts*>(pPane);
                    if (parts)
                    {
                        if (pPartsLayout)
                        {
                            // 部品レイアウトが指定されている場合は、既に構築済みのレイアウトがあるので、
                            // 部品ペインを作成して、レイアウトをセットする。
                            Parts* new_parts = Layout::NewObj<Parts, const Parts&>(*parts);
                            new_parts->SetLayout(pPartsLayout);
                            new_pane = new_parts;
                        }
                        else
                        {
                            // 部品レイアウトが指定されていない場合は、まだ部品レイアウトが作られていないので、この場でコピーする。
                            // コピーしたレイアウトはルートペインが部品ペインになる。レイアウトをコピーするとペインツリーごとコピー
                            // されるので、子供をコピーする必要はない。
                            Layout* parts_layout = Layout::NewObj<Layout, const Layout&, Layout*>(*parts->GetLayout(), parts->GetLayout());
                            static_cast<Parts*>(parts_layout->GetRootPane())->SetLayout(parts_layout);
                            new_pane = parts_layout->GetRootPane();
                            need_clone_child = false;
                        }
                    }
                    else
                    {
                        new_pane = Layout::NewObj<Pane, const Pane&>(*pPane);
                    }
                }
            }
        }
    }

    if (need_clone_child) {
        for (PaneList::ConstIterator it = pPane->GetChildList().GetBeginIter(); it != pPane->GetChildList().GetEndIter(); ++it)
        {
            Pane* child_tree = ClonePaneTreeWithPartsLayout(&(*it), NULL);
            new_pane->AppendChild(child_tree);
        }
    }

    return new_pane;
}

bool
GX2SurfaceFormatToLytFormat(TexFormat* pTexFormat, int gx2Format, bool is1chUsaAlpha)
{
    switch (gx2Format) {
    case GX2_SURFACE_FORMAT_TC_R8_UNORM:
        if (is1chUsaAlpha) {
            *pTexFormat = TEXFORMAT_A8;
        } else {
            *pTexFormat = TEXFORMAT_L8;
        }
        break;
    case GX2_SURFACE_FORMAT_T_R4_G4_UNORM:
        *pTexFormat = TEXFORMAT_LA4;
        break;
    case GX2_SURFACE_FORMAT_TC_R8_G8_UNORM:
        *pTexFormat = TEXFORMAT_LA8;
        break;
    case GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM:
        *pTexFormat = TEXFORMAT_RGB565;
        break;
    case GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM:
        *pTexFormat = TEXFORMAT_RGB5A1;
        break;
    case GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM:
        *pTexFormat = TEXFORMAT_RGBA4;
        break;
    case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM:
        *pTexFormat = TEXFORMAT_RGBA8;
        break;
    case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB:
        *pTexFormat = TEXFORMAT_RGBA8_SRGB;
        break;
    case GX2_SURFACE_FORMAT_T_BC1_UNORM:
        *pTexFormat = TEXFORMAT_BC1;
        break;
    case GX2_SURFACE_FORMAT_T_BC1_SRGB:
        *pTexFormat = TEXFORMAT_BC1_SRGB;
        break;
    case GX2_SURFACE_FORMAT_T_BC2_UNORM:
        *pTexFormat = TEXFORMAT_BC2;
        break;
    case GX2_SURFACE_FORMAT_T_BC2_SRGB:
        *pTexFormat = TEXFORMAT_BC2_SRGB;
        break;
    case GX2_SURFACE_FORMAT_T_BC3_UNORM:
        *pTexFormat = TEXFORMAT_BC3;
        break;
    case GX2_SURFACE_FORMAT_T_BC3_SRGB:
        *pTexFormat = TEXFORMAT_BC3_SRGB;
        break;
    case GX2_SURFACE_FORMAT_T_BC4_UNORM:
        if (is1chUsaAlpha) {
            *pTexFormat = TEXFORMAT_BC4A;
        } else {
            *pTexFormat = TEXFORMAT_BC4L;
        }
        break;
    case GX2_SURFACE_FORMAT_T_BC5_UNORM:
        *pTexFormat = TEXFORMAT_BC5;
        break;
    case GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM:
        *pTexFormat = TEXFORMAT_RGB10_A2;
        break;
    default:
        return false;
    }

    return true;
}

void DrawNullAndBoundingPane(DrawInfo& drawInfo, const Pane* pPane, ut::Color8 nullColor, ut::Color8 boundingColor)
{
    // ルートペインかどうかをチェック
    if (pPane->GetParent())
    {
        if (ut::IsTypeOf<Pane>(pPane))
        {
            drawInfo.SetModelViewMtx(pPane->GetGlobalMtx());

            ut::Rect rect = pPane->GetPaneRect();
            internal::DrawBox(drawInfo, math::VEC2(rect.left, rect.top), pPane->GetSize(), nullColor);

            drawInfo.ResetGlState();
        }
        else if (ut::IsTypeOf<Bounding>(pPane))
        {
            drawInfo.SetModelViewMtx(pPane->GetGlobalMtx());

            ut::Rect rect = pPane->GetPaneRect();
            internal::DrawBox(drawInfo, math::VEC2(rect.left, rect.top), pPane->GetSize(), boundingColor);

            drawInfo.ResetGlState();
        }
    }

    PaneList::ConstIterator it_end = pPane->GetChildList().GetEndIter();
    for (PaneList::ConstIterator it = pPane->GetChildList().GetBeginIter(); it != it_end; ++it)
    {
        DrawNullAndBoundingPane(drawInfo, &(*it), nullColor, boundingColor);
    }
}

f32 GetHermiteCurveValue(f32 frame, const res::HermiteKey*  keyArray, u32 keySize)
{
    NW_ASSERTMSG(keySize > 0, "out of bounds: keySize[%u] > 0", keySize);

    // outside of curve
    if (keySize == 1 || frame <= keyArray[0].frame)
    {
        return keyArray[0].value;
    }
    else if (frame >= keyArray[keySize - 1].frame)
    {
        return keyArray[keySize - 1].value;
    }

    // find 2 keys
    u32 ikeyL = 0;
    u32 ikeyR = keySize - 1;
    while (ikeyL != ikeyR - 1 && ikeyL != ikeyR)
    {
        int ikeyCenter = static_cast<int>(ikeyL + ikeyR) / 2;
        if (frame <= keyArray[ikeyCenter].frame)
        {
            ikeyR = static_cast<u32>(ikeyCenter);
        }
        else
        {
            ikeyL = static_cast<u32>(ikeyCenter);
        }
    }

    // calculate hermite interpolation
    const res::HermiteKey& key0 = keyArray[ikeyL];
    const res::HermiteKey& key1 = keyArray[ikeyR];
    const f32 R_FRAME_TOLERANCE = 0.001F;
    const f32 diff = frame - key1.frame;
    if (-R_FRAME_TOLERANCE < diff && diff < R_FRAME_TOLERANCE)
    {
        if (ikeyR < keySize - 1 &&
            key1.frame == keyArray[ikeyR + 1].frame)
        {
            // 同一フレームに値の異なるキーがある場合
            return keyArray[ikeyR + 1].value;
        }
        else
        {
            return key1.value;
        }
    }
    f32 t1 = frame - key0.frame;
    f32 t2 = 1.0F / (key1.frame - key0.frame);
    f32 v0 = key0.value;
    f32 v1 = key1.value;
    f32 s0 = key0.slope;
    f32 s1 = key1.slope;

    f32 t1t1t2 = t1 * t1 * t2;
    f32 t1t1t2t2 = t1t1t2 * t2;
    f32 t1t1t1t2t2 = t1 * t1t1t2t2;
    f32 t1t1t1t2t2t2 = t1t1t1t2t2 * t2;

    return v0 * (2.0F * t1t1t1t2t2t2 - 3.0F * t1t1t2t2 + 1.0F)
        + v1 * (-2.0F * t1t1t1t2t2t2 + 3.0F * t1t1t2t2)
        + s0 * (t1t1t1t2t2 - 2.0F * t1t1t2 + t1)
        + s1 * (t1t1t1t2t2 - t1t1t2);
}

} // namespace nw::lyt
} // namespace nw

