﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Layout.h>

#include <nw/ut/ut_Inlines.h>
#include <nw/gfnd/gfnd_ShaderHelper.h>
namespace shhelp = nw::gfnd::shader_helper;

#include <nw/dev/dev_Profile.h>

namespace nw
{
namespace lyt
{
namespace internal
{

using namespace nw::math;

TexCoordAry::TexCoordAry()
:   m_Cap(0),
    m_Num(0),
    m_pData(NULL)
{
}

void
TexCoordAry::Free()
{
    if (m_pData)
    {
        const u32 coordNum = m_Cap;
        Layout::DeleteArray<math::VEC2>(&m_pData[0][0], VERTEX_MAX * coordNum);
        m_pData = 0;

        m_Cap = 0;
        m_Num = 0;
    }
}

void
TexCoordAry::Reserve(u8 num)
{
    NW_ASSERTMSG(num <= TexMapMax, "out of bounds: num[%d] <= TexMapMax", num);

    if (m_Cap < num)
    {
        Free();

        // テクスチャ座標
        const u32 coordNum = num;
        math::VEC2 *const pVecAry = Layout::NewArray<math::VEC2>(VERTEX_MAX * coordNum);
        m_pData = reinterpret_cast<TexCoordQuad *>(pVecAry);
        if (m_pData)
        {
            m_Cap = num;
        }
    }
}

void
TexCoordAry::SetSize(u8 num)
{
    if (m_pData && num <= m_Cap)
    {
        static const VEC2 texCoords[VERTEX_MAX] =
        {
            VEC2(0.f, 0.f),
            VEC2(1.f, 0.f),
            VEC2(0.f, 1.f),
            VEC2(1.f, 1.f)
        };

        for (int j = m_Num; j < num; ++j)
        {
            for (int i = 0; i < VERTEX_MAX; ++i)
            {
                m_pData[j][i] = texCoords[i];
            }
        }
        m_Num = num;
    }
}

void
TexCoordAry::GetCoord(
    u32 idx,
    TexCoordQuad coord
) const
{
    NW_ASSERTMSG(idx < m_Num, "out of bounds: idx[%d] < m_Num[%d]", idx, m_Num);

    for (int i = 0; i < VERTEX_MAX; ++i)
    {
        coord[i] = m_pData[idx][i];
    }
}

void
TexCoordAry::SetCoord(
    u32 idx,
    const TexCoordQuad coord
)
{
    NW_ASSERTMSG(idx < m_Num, "out of bounds: idx[%d] < m_Num[%d]", idx, m_Num);

    for (int i = 0; i < VERTEX_MAX; ++i)
    {
        m_pData[idx][i] = coord[i];
    }
}

void
TexCoordAry::Copy(
    const void* pResTexCoord,
    u8 texCoordNum
)
{
    NW_ASSERTMSG(texCoordNum <= m_Cap, "out of bounds: texCoordNum[%d] <= m_Cap[%d]", texCoordNum, m_Cap);

    m_Num = ut::Max(m_Num, texCoordNum);
    const res::Vec2 (*src)[VERTEX_MAX] = static_cast<const res::Vec2 (*)[VERTEX_MAX]>(pResTexCoord);
    for (int j = 0; j < texCoordNum; ++j)
    {
        for (int i = 0; i < VERTEX_MAX; ++i)
        {
            m_pData[j][i] = src[j][i];
        }
    }
}

//----------------------------------------
const res::Material *const
GetResMaterial(
    const BuildResSet* buildResSet,
    u16 materialIdx)
{
    NW_ASSERT_NOT_NULL(buildResSet->pMaterialList);
    const ut::ResU32 *const matOffsTbl = internal::ConvertOffsToPtr<ut::ResU32>(buildResSet->pMaterialList, sizeof(*buildResSet->pMaterialList));
    return internal::ConvertOffsToPtr<res::Material>(buildResSet->pMaterialList, matOffsTbl[materialIdx]);
}

//----------------------------------------
void
DrawQuad(
    DrawInfo&     drawInfo,
    const VEC2&         basePt,
    const Size&         size
)
{
    NW_PROFILE("nw::lyt::internal::DrawQuad");

    // プログラムが設定されてから一度だけ行う初期化。
    drawInfo.SetupProgram();

    // ローカル座標変換設定
    shhelp::SetVertexUniformReg(drawInfo.GetUniformId(UNIFORM_uTransform), size.width, size.height, basePt.x, basePt.y);

    drawInfo.LoadMtxModelView();
    NW_GL_ASSERT();

#if defined(NW_PLATFORM_CAFE)
    // GX2: 描画
    GraphicsResource& gres = *drawInfo.GetGraphicsResource();
    gres.DrawVBO();
#else
    glDrawElements(GL_TRIANGLE_STRIP, GraphicsResource::VBO_INDEX_COUNT, GL_UNSIGNED_SHORT, 0);
#endif
    NW_GL_ASSERT();
}

//----------------------------------------
void
DrawQuadWithTexCoords(
    DrawInfo& drawInfo,
    const VEC2& basePt,
    const Size& size,
    u8 texCoordNum,
    const VEC2 (*texCoords)[VERTEX_MAX]
)
{
    NW_PROFILE("nw::lyt::internal::DrawQuadWithTexCoords");

    drawInfo.SetupProgram(); // プログラムが設定されてから一度だけ行う初期化。

    // テクスチャ座標設定
    if (0 < texCoordNum && texCoords != NULL)
    {
        for (int i = 0; i < TexMapMax; ++i)
        {
            int src = drawInfo.GetTexCoordSrc(i);
            if (src < 0)
            {
                break;
            }

            if (texCoords != NULL && src < texCoordNum)
            {
                math::MTX44 texCoordsMatrix(
                    texCoords[src][0].x, texCoords[src][0].y, 0.0f, 1.0f,
                    texCoords[src][1].x, texCoords[src][1].y, 0.0f, 1.0f,
                    texCoords[src][2].x, texCoords[src][2].y, 0.0f, 1.0f,
                    texCoords[src][3].x, texCoords[src][3].y, 0.0f, 1.0f
                );
                gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uVertexTexCoord0 + i);
                shhelp::SetVertexUniformReg(id, texCoordsMatrix);
            }
        }
        NW_GL_ASSERT();
    }

    // ローカル座標変換設定
    shhelp::SetVertexUniformReg(drawInfo.GetUniformId(UNIFORM_uTransform), size.width, size.height, basePt.x, basePt.y);

    drawInfo.LoadMtxModelView();
    NW_GL_ASSERT();

#if defined(NW_PLATFORM_CAFE)
    // GX2: 描画
    GraphicsResource& gres = *drawInfo.GetGraphicsResource();
    gres.DrawVBO();
#else
    glDrawElements(GL_TRIANGLE_STRIP, GraphicsResource::VBO_INDEX_COUNT, GL_UNSIGNED_SHORT, 0);
#endif
    NW_GL_ASSERT();
}

//----------------------------------------
// @brief 前回と同じ設定で描画を繰り返す
void
DrawQuad_Repeat(
    const DrawInfo&     drawInfo,
    const VEC2&         basePt,
    const Size&         size
)
{
    NW_PROFILE("nw::lyt::internal::DrawQuad_Repeat");

    // ローカル座標変換を設定
    shhelp::SetVertexUniformReg(drawInfo.GetUniformId(UNIFORM_uTransform), size.width, size.height, basePt.x, basePt.y);

#if defined(NW_PLATFORM_CAFE)
    // GX2: 描画
    GraphicsResource& gres = *drawInfo.GetGraphicsResource();
    gres.DrawVBO();
#else
    glDrawElements(GL_TRIANGLE_STRIP, GraphicsResource::VBO_INDEX_COUNT, GL_UNSIGNED_SHORT, 0);
#endif
    NW_GL_ASSERT();
}

//----------------------------------------
void
DrawBox(
    DrawInfo&           drawInfo,
    const math::VEC2&   pos,
    const Size&         size,
    ut::Color8          color)
{
    // テクスチャなし、頂点カラーなしのシェーダを設定
    drawInfo.SetShader(NULL_TEXTURE_SHADER + SHADER_ID_QUANTITY);

#if defined(NW_PLATFORM_CAFE)
    const int indeicesSize = sizeof(u16) * 4;
    static u16 indices[indeicesSize + GX2_INDEX_BUFFER_ALIGNMENT];
    u16* aligned_indecies = static_cast<u16*>(ut::RoundUp(indices, GX2_INDEX_BUFFER_ALIGNMENT));
#endif
    {
        static bool isVBOinitialized = false;
#if defined(NW_PLATFORM_CAFE)
        const int verticesSize = sizeof(f32) * 2 * 4;
        static f32 vertices[verticesSize + GX2_VERTEX_BUFFER_ALIGNMENT];
        f32* aligned_vertices = static_cast<f32*>(ut::RoundUp(vertices, GX2_VERTEX_BUFFER_ALIGNMENT));
#else
        static GLuint lineVBO[GraphicsResource::VBO_MAX];
        static const GLushort lineVertexIndex[GraphicsResource::VBO_INDEX_COUNT] =
        {
            VERTEX_RT, VERTEX_LT, VERTEX_RB, VERTEX_LB
        };
        const GLshort lineUniformVertexIndex[VERTEX_MAX][VERTEXATTRSIZE_INDEX] =
        {
            { 0, 0 }, // LT
            { 1, 0 }, // RT
            { 1, 1 }, // RB
            { 0, 1 }  // LB
        };
#endif
        if ( ! isVBOinitialized)
        {
#if defined(NW_PLATFORM_CAFE)
            // GX2: バーテックスオブジェクト生成
            aligned_vertices[0] = 0.0f; aligned_vertices[1] = 0.0f;
            aligned_vertices[2] = 1.0f; aligned_vertices[3] = 0.0f;
            aligned_vertices[4] = 1.0f; aligned_vertices[5] = 1.0f;
            aligned_vertices[6] = 0.0f; aligned_vertices[7] = 1.0f;
            GX2Invalidate(GX2_INVALIDATE_CPU, aligned_vertices, verticesSize);

            aligned_indecies[0] = VERTEX_LT;
            aligned_indecies[1] = VERTEX_RT;
            aligned_indecies[2] = VERTEX_LB;
            aligned_indecies[3] = VERTEX_RB;
            GX2Invalidate(GX2_INVALIDATE_CPU, aligned_indecies, indeicesSize);
#else
            // VAOの無効化
            glBindVertexArray(0);

            glGenBuffers(GraphicsResource::VBO_MAX, lineVBO);
            NW_GL_ASSERT();

            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, lineVBO[GraphicsResource::VBO_ELEMENT]);
            glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(lineVertexIndex), lineVertexIndex, GL_STATIC_DRAW);
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
            NW_GL_ASSERT();

            glBindBuffer(GL_ARRAY_BUFFER, lineVBO[GraphicsResource::VBO_VERTEX_INDEX]);
            glBufferData(GL_ARRAY_BUFFER, sizeof(lineUniformVertexIndex), lineUniformVertexIndex, GL_STATIC_DRAW);
            glBindBuffer(GL_ARRAY_BUFFER, 0);
            NW_GL_ASSERT();
#endif
            isVBOinitialized = true;
        }
#if defined(NW_PLATFORM_CAFE)
        // GX2: バーテックスオブジェクト有効化
        GX2SetAttribBuffer(0, sizeof(f32) * 2 * 4, sizeof(f32) * 2, aligned_vertices);
#else
        glBindVertexArray(0);

        // 頂点インデックス
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, lineVBO[0]);

        // 頂点座標
        glBindBuffer(GL_ARRAY_BUFFER, lineVBO[1]);
        glEnableVertexAttribArray(VERTEXATTR_VERTEX_INDEX);
        glVertexAttribPointer(VERTEXATTR_VERTEX_INDEX, VERTEXATTRSIZE_INDEX, GL_SHORT, GL_FALSE, 0, NULL);
#endif
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    {
        const f32 scale = 1.0f / 255.0f; // 頂点カラーを[0, 1]に正規化するためのスケールです。
        static const ut::Color8 colors[VERTEXCOLOR_MAX] = {
            ut::Color8(255, 255, 255, 255),
            ut::Color8(255, 255, 255, 255),
            ut::Color8(255, 255, 255, 255),
            ut::Color8(255, 255, 255, 255)
        };
        internal::SetupVertexColor(drawInfo, colors);
        // 頂点カラーへ積算するカラーを設定します。
        shhelp::SetVertexUniformReg(
            drawInfo.GetUniformId(UNIFORM_uColor),
            scale * 1.0f,
            scale * 1.0f,
            scale * 1.0f,
            scale * scale * 255.f);
    }
#endif

    {
        ut::FloatColor float_color(color);
        gfnd::ShaderUniformId loc = drawInfo.GetUniformId(UNIFORM_uInterpolateWidth);
        shhelp::SetPixelUniformReg(loc, float_color);
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        loc = drawInfo.GetUniformId(UNIFORM_uInterpolateOffset);
        shhelp::SetPixelUniformReg(loc, 0.f, 0.f, 0.f, 0.f);
#endif
    }

    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glDisable(GL_COLOR_LOGIC_OP); // 論理演算はしない
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); // ソースアルファでブレンド
        glBlendEquation(GL_FUNC_ADD);
        NW_GL_ASSERT();
#elif defined(NW_PLATFORM_CAFE)
        GX2SetColorControl(
            GX2_LOGIC_OP_COPY,
            0x01,
            GX2_DISABLE,
            GX2_ENABLE);
        GX2SetBlendControl(
            GX2_RENDER_TARGET_0,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD,
            GX2_FALSE,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD);
#endif
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glDisable(GL_ALPHA_TEST);
#elif defined(NW_PLATFORM_CAFE)
        GX2SetAlphaTest(GX2_FALSE, GX2_COMPARE_NEVER, 0.0f);
#endif
    }

    // フレームの設定を初期化
    gfnd::ShaderUniformId loc = drawInfo.GetUniformId(UNIFORM_uFrameSpec);
    shhelp::SetVertexUniformReg<gfnd::ShaderInt>(loc, TEXTUREFLIP_NONE);
    NW_GL_ASSERT();

    // ローカル座標変換設定
    shhelp::SetVertexUniformReg(drawInfo.GetUniformId(UNIFORM_uTransform), size.width, size.height, pos.x, pos.y);

    drawInfo.LoadMtxModelView();
    drawInfo.LoadProjectionMtx();
    NW_GL_ASSERT();

#if defined(NW_PLATFORM_CAFE)
    // GX2: 描画
    GX2SetLineWidth(2.f);
    GX2DrawIndexed(GX2_PRIMITIVE_LINE_LOOP, GraphicsResource::VBO_INDEX_COUNT, GX2_INDEX_FORMAT_U16, aligned_indecies);
#else
    // テクスチャモードを0にする -> テクスチャをフェッチしない設定
    gfnd::ShaderUniformId textureModeLoc = drawInfo.GetUniformId(UNIFORM_uTextureMode);
    shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 0);

    glDrawElements(GL_LINE_LOOP, GraphicsResource::VBO_INDEX_COUNT, GL_UNSIGNED_SHORT, 0);
#endif
    NW_GL_ASSERT();
}

} // namespace nw::lyt::internal

} // namespace nw::lyt
} // namespace nw
