﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Group.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <nw/lyt/lyt_Util.h>
#include <nw/lyt/lyt_TextBox.h>

#include <nw/dev/dev_Profile.h>

namespace
{
const f32 R_SAME_TOLERANCE = 1.0e-5F;
const f32 R_FRAME_TOLERANCE = 0.001F;
}

namespace nw
{
namespace lyt
{
namespace
{

/*---------------------------------------------------------------------------*
  @brief f32 型の値を許容値付きで比較します。

  @param a 比較する値です。
  @param b 比較する値です。
  @param tolerance 許容値です。

  @return ２つの値の差の絶対値が許容値未満なら true を返します。
 *---------------------------------------------------------------------------*/
inline bool
RIsSame(
    const f32 a,
    const f32 b,
    const f32 tolerance = R_SAME_TOLERANCE
)
{
    f32 c = a - b;
    return (-tolerance < c && c < tolerance);
}

/*---------------------------------------------------------------------------*
  @brief ステップ形式カーブの特定フレームでの値を取得します。

  @param frame 値を取得するフレームです。
  @param keyArray ステップ形式キーの配列です。
  @param keySize キー数です。

  @return 値です。
 *---------------------------------------------------------------------------*/
u16
GetStepCurveValue(
    f32 frame,
    const res::StepKey* keyArray,
    u32 keySize
)
{
    NW_ASSERTMSG(keySize > 0, "out of bounds: keySize[%u] > 0", keySize);

    if (keySize == 1 || frame <= keyArray[0].frame)
    {
        return keyArray[0].value;
    }
    else if (frame >= keyArray[keySize - 1].frame)
    {
        return keyArray[keySize - 1].value;
    }

    int ikeyL = 0;
    int ikeyR = (int)keySize - 1;
    while (ikeyL != ikeyR - 1 && ikeyL != ikeyR)
    {
        int ikeyCenter = (ikeyL + ikeyR) / 2;
        const res::StepKey& centerKey = keyArray[ikeyCenter];
        if (frame < centerKey.frame)
        {
            ikeyR = ikeyCenter;
        }
        else
        {
            ikeyL = ikeyCenter;
        }
    }

    if (RIsSame(frame, keyArray[ikeyR].frame, R_FRAME_TOLERANCE))
    {
        return keyArray[ikeyR].value;
    }
    else
    {
        return keyArray[ikeyL].value;
    }
}

void
AnimatePaneSRT(
    Pane* pPane,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);

        NW_ASSERT(pAnimTarget->target < ANIMTARGET_PANE_MAX);
        NW_ASSERT(pAnimTarget->curveType == ANIMCURVE_HERMITE);   // 現時点ではHERMITEのみ

        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        pPane->SetSRTElement(pAnimTarget->target, GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum));
    }
}

void
AnimateVisibility(
    Pane* pPane,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);

        NW_ASSERT(pAnimTarget->target < ANIMTARGET_VISIBILITY_MAX);
        NW_ASSERT(pAnimTarget->curveType == ANIMCURVE_STEP);

        const res::StepKey* keys = internal::ConvertOffsToPtr<res::StepKey>(pAnimTarget, pAnimTarget->keysOffset);
        pPane->SetVisible(0 != GetStepCurveValue(frame, keys, pAnimTarget->keyNum));
    }
}

void
AnimateVertexColor(
    Pane* pPane,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);

        NW_ASSERT(pAnimTarget->target < ANIMTARGET_PANE_COLOR_MAX);
        NW_ASSERT(pAnimTarget->curveType == ANIMCURVE_HERMITE);   // 現時点ではHERMITEのみ

        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        f32 value = GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum);
        value += 0.5f;  // 四捨五入のため
        u8 u8Val = static_cast<u8>(ut::Min(ut::Max(value, 0.f), 255.f));
        pPane->SetColorElement(pAnimTarget->target, u8Val);
    }
}

void
AnimatePerCharacterTransform(
    Pane* pPane,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);

        NW_ASSERT(pAnimTarget->target < ANIMTARGET_PERCHARACTERTRANSFORM_MAX);
        NW_ASSERT(pAnimTarget->curveType == ANIMCURVE_HERMITE);   // 現時点ではHERMITEのみ

        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        static_cast<TextBox*>(pPane)->SetPerCharacterTransformOffset(GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum));
    }
}

void
AnimateMaterialColor(
    Material* pMaterial,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);
        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        f32 value = GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum);
        value += 0.5f;  // 四捨五入のため
        u8 val = static_cast<u8>(ut::Min(ut::Max(value, 0.f), 255.f));
        pMaterial->SetColorElement(pAnimTarget->target, val);
    }
}

void
AnimateTextureSRT(
    Material* pMaterial,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);
        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        pMaterial->SetTexSRTElement(pAnimTarget->id, pAnimTarget->target, GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum));
    }
}

void
AnimateTexturePattern(
    Material* pMaterial,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame,
    const TextureInfo** texInfos
)
{
    for (int j = 0; j < pAnimInfo->num; ++j)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[j]);
        const res::StepKey* keys = internal::ConvertOffsToPtr<res::StepKey>(pAnimTarget, pAnimTarget->keysOffset);
        const u16 fileIdx = GetStepCurveValue(frame, keys, pAnimTarget->keyNum);

        if (texInfos[fileIdx]->IsValid())
        {
            pMaterial->SetTexMap(pAnimTarget->id, texInfos[fileIdx]);
        }
    }
}

void
AnimateIndirectSRT(
    Material*                   pMaterial,
    const res::AnimationInfo*   pAnimInfo,
    const ut::ResU32*           animTargetOffsets,
    f32                         frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);
        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        pMaterial->SetIndirectSRTElement(pAnimTarget->target, GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum));
    }
}

void
AnimateAlphaCompare(
    Material* pMaterial,
    const res::AnimationInfo* pAnimInfo,
    const ut::ResU32* animTargetOffsets,
    f32 frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);
        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        f32 value = GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum);
        f32 val = ut::Min(ut::Max(value, 0.f), 1.f);
        res::AlphaCompare compare(pMaterial->GetAlphaCompare().GetFunc(), val);
        pMaterial->SetAlphaCompare(compare);
    }
}

void
AnimateFontShadow(
    Material*                   pMaterial,
    const res::AnimationInfo*   pAnimInfo,
    const ut::ResU32*           animTargetOffsets,
    f32                         frame
)
{
    for (int i = 0; i < pAnimInfo->num; ++i)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(pAnimInfo, animTargetOffsets[i]);
        const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
        f32 value = GetHermiteCurveValue(frame, keys, pAnimTarget->keyNum);
        value += 0.5f;  // 四捨五入のため
        u8 value8 = static_cast<u8>(ut::Min(ut::Max(value, 0.f), 255.f));
        pMaterial->SetFontShadowParameterElement(pAnimTarget->target, value8);
    }
}

} // namespace nw::lyt::{no-name}

AnimTransform::AnimTransform()
: m_pRes(0)
, m_Frame(0)
, m_IsEnable(true)
{
}

AnimTransform::~AnimTransform()
{
}

u16
AnimTransform::GetFrameSize() const
{
    return m_pRes->frameSize;
}

void
AnimTransform::UpdateFrame(f32 progress_frame)
{
    NW_UNUSED_VARIABLE(progress_frame);
}

void
AnimTransform::SetEnable(bool bEnable)
{
    m_IsEnable = bEnable;
}

bool
AnimTransform::IsLoopData() const
{
    return m_pRes->loop != 0;
}

bool
AnimTransform::IsWaitData() const
{
    return m_pRes->frameSize == 0;
}

AnimTransformBasic::AnimTransformBasic()
: m_pTexAry(0)
, m_BindPairAry(0)
, m_BindPairNum(0)
, m_BindPairNumMax(0)
{
}

AnimTransformBasic::~AnimTransformBasic()
{
    Layout::DeletePrimArray(m_BindPairAry);
    Layout::DeletePrimArray(m_pTexAry);
}

void
AnimTransformBasic::SetResource(
    const res::AnimationBlock* pRes,
    ResourceAccessor* pResAccessor
)
{
    NW_ASSERT_NOT_NULL(pRes);
    SetResource(pRes, pResAccessor, pRes->animContNum);
}

void
AnimTransformBasic::SetResource(
    const res::AnimationBlock*  pRes,
    ResourceAccessor* pResAccessor,
    u16 animNum
)
{
    NW_ASSERT(m_pTexAry == 0);
    NW_ASSERT(m_BindPairAry == 0);
    NW_ASSERT_NOT_NULL(pRes);

    this->SetAnimResource(pRes);
    m_pTexAry = 0;

     if (pRes->fileNum > 0)
    {
        NW_ASSERT_NOT_NULL(pResAccessor);

        m_pTexAry = Layout::NewArray<const TextureInfo*>(pRes->fileNum);
        if (m_pTexAry)
        {
            const ut::ResU32* fileNameOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pRes, sizeof(*pRes));

            for (int i = 0; i < pRes->fileNum; ++i)
            {
                const char *const fileName = internal::GetStrTableStr(fileNameOffsets, i);

                // テクスチャオブジェクトをロードする。
                m_pTexAry[i] = pResAccessor->GetTexture(fileName);
            }
        }
    }

    m_BindPairAry = Layout::NewArray<BindPair>(animNum);
    if (m_BindPairAry)
    {
        m_BindPairNumMax = animNum;
    }
}

void
AnimTransformBasic::BindPane(
    Pane* pPane,
    bool bRecursive
)
{
    NW_ASSERT_NOT_NULL(pPane);

    const res::AnimationBlock* pRes = this->GetAnimResource();

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pRes, pRes->animContOffsetsOffset);
    for (u16 i = 0; i < pRes->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(pRes, animContOffsets[i]);
        if (animCont.type == ANIMCONTENTTYPE_PANE)
        {
            if (Pane *const pFindPane = pPane->FindPaneByName(animCont.name, bRecursive))
            {
                if (! checkBindAnimationDoubly(pFindPane, &animCont))
                {
                    if (! BindPaneImpl(pFindPane, &animCont))
                    {
                        break;
                    }
                }
            }
        }
        else
        {
            if (Material *const pFindMat = pPane->FindMaterialByName(animCont.name, bRecursive))
            {
                if (! checkBindAnimationDoubly(pFindMat, &animCont))
                {
                    if (! BindMaterialImpl(pFindMat, &animCont))
                    {
                        break;
                    }
                }
            }
        }
    }
}

void
AnimTransformBasic::BindGroup(
    Group* pGroup
)
{
    NW_ASSERT_NOT_NULL(pGroup);

    const res::AnimationBlock* pRes = this->GetAnimResource();

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pRes, pRes->animContOffsetsOffset);
    for (u16 i = 0; i < pRes->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(pRes, animContOffsets[i]);

        if (animCont.type == ANIMCONTENTTYPE_PANE) {
            // グループに所属しているペインの中から探す
            PaneLinkList::Iterator paneEnd = pGroup->GetPaneList().GetEndIter();
            Pane* targetPane = NULL;
            for (PaneLinkList::Iterator it = pGroup->GetPaneList().GetBeginIter(); it != paneEnd; ++it)
            {
                if (internal::EqualsResName(animCont.name, it->target->GetName()))
                {
                    targetPane = it->target;
                    break;
                }
            }
            if (targetPane) {
                if (! checkBindAnimationDoubly(targetPane, &animCont))
                {
                    if (! BindPaneImpl(targetPane, &animCont))
                    {
                        break;
                    }
                }
            }
        }
        else
        {
            // グループに所属しているペインの中からマテリアルを探す
            PaneLinkList& paneList = pGroup->GetPaneList();
            PaneLinkList::Iterator paneEnd = paneList.GetEndIter();
            Material* pFindMat = NULL;
            for (PaneLinkList::Iterator it = paneList.GetBeginIter(); it != paneEnd; ++it)
            {
                pFindMat = it->target->FindMaterialByName(animCont.name, false);
                if (pFindMat)
                {
                    break;
                }
            }
            if (pFindMat) {
                if (! checkBindAnimationDoubly(pFindMat, &animCont))
                {
                    if (! BindMaterialImpl(pFindMat, &animCont))
                    {
                        break;
                    }
                }
            }
        }
    }
}

void
AnimTransformBasic::BindMaterial(
    Material* pMaterial
)
{
    NW_ASSERT_NOT_NULL(pMaterial);

    const res::AnimationBlock* pRes = this->GetAnimResource();

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pRes, pRes->animContOffsetsOffset);
    for (u32 i = 0; i < pRes->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(pRes, animContOffsets[i]);
        if (animCont.type == ANIMCONTENTTYPE_MATERIAL)
        {
            if (internal::EqualsMaterialName(pMaterial->GetName(), animCont.name))
            {
                if (! checkBindAnimationDoubly(pMaterial, &animCont))
                {
                    if (! BindMaterialImpl(pMaterial, &animCont))
                    {
                        break;
                    }
                }
            }
        }
    }
}

void AnimTransformBasic::ForceBindPane(
    Pane* pDstPane,
    const Pane* pSrcPane)
{
    NW_ASSERT_NOT_NULL(pDstPane);
    NW_ASSERT_NOT_NULL(pSrcPane);

    const res::AnimationBlock* pRes = this->GetAnimResource();

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pRes, pRes->animContOffsetsOffset);
    for (u16 i = 0; i < pRes->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(pRes, animContOffsets[i]);
        if (animCont.type == ANIMCONTENTTYPE_PANE)
        {
            // src側の名前と一致したanimContをdst側にバインドする
            if (internal::EqualsResName(pSrcPane->GetName(), animCont.name))
            {
                if (! checkBindAnimationDoubly(pDstPane, &animCont))
                {
                    if (! BindPaneImpl(pDstPane, &animCont))
                    {
                        return;
                    }
                }
            }
        }
        else
        {
            u32 nbMaterial = pSrcPane->GetMaterialNum();
            for (u32 idx = 0; idx < nbMaterial; ++idx)
            {
                const Material* pSrcMaterial = pSrcPane->GetMaterial(idx);
                if (pSrcMaterial)
                {
                    // src側の名前と一致したanimContをdst側にバインドする。
                    // バインドするマテリアルの対象は、同じインデックスのマテリアルとする。
                    if (internal::EqualsMaterialName(pSrcMaterial->GetName(), animCont.name))
                    {
                        Material* pDstMaterial = pDstPane->GetMaterial(idx);
                        if (pDstMaterial)
                        {
                            if (! BindMaterialImpl(pDstMaterial, &animCont))
                            {
                                return;
                            }
                        }
                    }
                }
            }
        }
    }
}

void
AnimTransformBasic::UnbindPane(Pane* pPane)
{
    int end = m_BindPairNum;
    for (int i = 0; i < end; i++)
    {
        BindPair* pair = &m_BindPairAry[i];
        if (pair->target == pPane)
        {
            eraseBindPair(i);
            break;
        }
    }

    u32 nbMaterial = pPane->GetMaterialNum();
    for (u32 idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = pPane->GetMaterial(idx);
        if (pMaterial)
        {
            UnbindMaterial(pMaterial);
        }
    }
}

void
AnimTransformBasic::UnbindGroup(Group* pGroup)
{
    PaneLinkList& paneList = pGroup->GetPaneList();
    PaneLinkList::Iterator it_end = paneList.GetEndIter();
    for (PaneLinkList::Iterator it = paneList.GetBeginIter(); it != it_end; ++it)
    {
        UnbindPane(it->target);
    }
}

void
AnimTransformBasic::UnbindMaterial(Material* pMaterial)
{
    int end = m_BindPairNum;
    for (int i = 0; i < end; i++)
    {
        BindPair* pair = &m_BindPairAry[i];
        if (pair->target == pMaterial)
        {
            eraseBindPair(i);
            break;
        }
    }
}

void
AnimTransformBasic::UnbindAll()
{
    m_BindPairNum = 0;
}

//----------------------------------------
bool
AnimTransformBasic::BindPaneImpl(
    Pane* pTarget,
    const res::AnimationContent* animCont
)
{
    NW_ASSERT_NOT_NULL(pTarget);

    if (m_BindPairNum >= m_BindPairNumMax)
    {
        NW_WARNING(false, "all BindPair used.");
        return false;
    }

    BindPair* pair = &m_BindPairAry[m_BindPairNum];
    m_BindPairNum++;

    pair->target = pTarget;
    pair->animCont = animCont;

    return true;
}

//----------------------------------------
bool
AnimTransformBasic::BindMaterialImpl(
    Material* pTarget,
    const res::AnimationContent* animCont
)
{
    NW_ASSERT_NOT_NULL(pTarget);

    if (m_BindPairNum >= m_BindPairNumMax)
    {
        NW_WARNING(false, "all BindPair used.");
        return false;
    }

    const ut::ResU32* animInfoOffsets = internal::ConvertOffsToPtr<ut::ResU32>(animCont, sizeof(*animCont));
    bool isBinding = true;

    if (animCont != NULL)
    {
        for (int j = 0; j < animCont->num; ++j)
        {
            const res::AnimationInfo* animInfo = internal::ConvertOffsToPtr<res::AnimationInfo>(animCont, animInfoOffsets[j]);
            const ut::ResU32* animTargetOffsets = internal::ConvertOffsToPtr<ut::ResU32>(animInfo, sizeof(*animInfo));

            for (int k = 0; k < animInfo->num; ++k)
            {
                const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(animInfo, animTargetOffsets[k]);

                switch (animInfo->kind)
                {
                case res::ANIMATIONTYPE_MATCOLOR:
                    NW_FAILSAFE_IF(ANIMTARGET_MATCOLOR_MAX <= pAnimTarget->target) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_HERMITE) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case res::ANIMATIONTYPE_TEXSRT:
                    NW_FAILSAFE_IF(TexMapMax <= pAnimTarget->id) { isBinding = false; }
                    NW_FAILSAFE_IF(pTarget->GetTexSRTCap() <= pAnimTarget->id) { isBinding = false; }
                    NW_FAILSAFE_IF(ANIMTARGET_TEXSRT_MAX <= pAnimTarget->target) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_HERMITE) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case res::ANIMATIONTYPE_TEXPATTERN:
                    NW_FAILSAFE_IF(m_pTexAry == NULL) { isBinding = false; }
                    NW_FAILSAFE_IF(pTarget->GetTexMapNum() <= pAnimTarget->id) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_STEP) { isBinding = false; }              // 現時点ではSTEPのみ
                    NW_FAILSAFE_IF(pAnimTarget->target != ANIMTARGET_TEXPATTURN_IMAGE) { isBinding = false; }    // 現状ではimageのみ

                    for (int fileIdx = 0; fileIdx < this->GetAnimResource()->fileNum; ++fileIdx)
                    {
                        NW_FAILSAFE_IF(!m_pTexAry[fileIdx]->IsValid()) { isBinding = false; break; }
                    }
                    break;
                case res::ANIMATIONTYPE_INDIRECTSRT:
                    NW_FAILSAFE_IF(!pTarget->IsIndirectParameterCap()) { isBinding = false; }
                    NW_FAILSAFE_IF(ANIMTARGET_INDIRECTSRT_MAX <= pAnimTarget->target) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_HERMITE) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case res::ANIMATIONTYPE_ALPHACOMPARE:
                    NW_FAILSAFE_IF(!pTarget->IsAlphaCompareCap()) { isBinding = false; }
                    NW_FAILSAFE_IF(ANIMTARGET_ALPHACOMPARE_MAX <= pAnimTarget->target) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_HERMITE) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case res::ANIMATIONTYPE_FONTSHADOW:
                    NW_FAILSAFE_IF(!pTarget->IsFontShadowParameterCap()) { isBinding = false; }
                    NW_FAILSAFE_IF(ANIMTARGET_FONTSHADOW_MAX <= pAnimTarget->target) { isBinding = false; }
                    NW_FAILSAFE_IF(pAnimTarget->curveType != ANIMCURVE_HERMITE) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                default:
                    NW_WARNING(false, "Invalid animation type. ");
                    break;
                }
            }
        }
    }

    if (isBinding)
    {
        BindPair* pair = &m_BindPairAry[m_BindPairNum];
        m_BindPairNum++;

        pair->target = pTarget;
        pair->animCont = animCont;
    }

    return true;
}

void
AnimTransformBasic::Animate()
{
    if (IsEnable())
    {
        int end = m_BindPairNum;
        for (int i = 0; i < end; i++)
        {
            BindPair* pair = &m_BindPairAry[i];
            if (pair->animCont->type == ANIMCONTENTTYPE_PANE)
            {
                AnimatePaneImpl(reinterpret_cast<Pane*>(pair->target), pair->animCont);
            }
            else
            {
                AnimateMaterialImpl(reinterpret_cast<Material*>(pair->target), pair->animCont);
            }
        }
    }
}

void
AnimTransformBasic::AnimatePane(Pane* pPane)
{
    if (IsEnable())
    {
        int end = m_BindPairNum;
        for (int i = 0; i < end; i++)
        {
            BindPair* pair = &m_BindPairAry[i];
            if (pair->target == pPane)
            {
                AnimatePaneImpl(pPane, pair->animCont);

                u32 nbMaterial = pPane->GetMaterialNum();
                for (u32 idx = 0; idx < nbMaterial; ++idx)
                {
                    Material* pMaterial = pPane->GetMaterial(idx);
                    if (pMaterial)
                    {
                        AnimateMaterial(pMaterial);
                    }
                }
                break;
            }
        }
    }
}

void
AnimTransformBasic::AnimateMaterial(Material* pMaterial)
{
    if (IsEnable())
    {
        int end = m_BindPairNum;
        for (int i = 0; i < end; i++)
        {
            BindPair* pair = &m_BindPairAry[i];
            if (pair->target == pMaterial)
            {
                AnimateMaterialImpl(pMaterial, pair->animCont);
                break;
            }
        }
    }
}

void
AnimTransformBasic::AnimatePaneImpl(Pane* pPane, const res::AnimationContent* animCont)
{
    NW_PROFILE("nw::lyt::AnimTransformBasic::Animate_Pane");

    const ut::ResU32* animInfoOffsets = internal::ConvertOffsToPtr<ut::ResU32>(animCont, sizeof(*animCont));
    for (int i = 0; i < animCont->num; ++i)
    {
        const res::AnimationInfo* pAnimInfo = internal::ConvertOffsToPtr<res::AnimationInfo>(animCont, animInfoOffsets[i]);
        const ut::ResU32* animTargetOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pAnimInfo, sizeof(*pAnimInfo));

        switch (pAnimInfo->kind)
        {
        case res::ANIMATIONTYPE_PANESRT:
            AnimatePaneSRT(pPane, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_VISIBILITY:
            AnimateVisibility(pPane, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_VTXCOLOR:
            AnimateVertexColor(pPane, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_PERCHARACTERTRANSFORM:
            AnimatePerCharacterTransform(pPane, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        default:
            NW_WARNING(false, "Invalid animation type. ");
            break;
        }
    }
}

void
AnimTransformBasic::AnimateMaterialImpl(Material* pMaterial, const res::AnimationContent* animCont)
{
    NW_PROFILE("nw::lyt::AnimTransformBasic::Animate_Material");

    const ut::ResU32* animInfoOffsets = internal::ConvertOffsToPtr<ut::ResU32>(animCont, sizeof(*animCont));
    for (int i = 0; i < animCont->num; ++i)
    {
        const res::AnimationInfo* pAnimInfo = internal::ConvertOffsToPtr<res::AnimationInfo>(animCont, animInfoOffsets[i]);
        const ut::ResU32* animTargetOffsets = internal::ConvertOffsToPtr<ut::ResU32>(pAnimInfo, sizeof(*pAnimInfo));

        switch (pAnimInfo->kind)
        {
        case res::ANIMATIONTYPE_MATCOLOR:
            AnimateMaterialColor(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_TEXSRT:
            AnimateTextureSRT(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_TEXPATTERN:
            AnimateTexturePattern(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame(), m_pTexAry);
            break;
        case res::ANIMATIONTYPE_INDIRECTSRT:
            AnimateIndirectSRT(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_ALPHACOMPARE:
            AnimateAlphaCompare(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        case res::ANIMATIONTYPE_FONTSHADOW:
            AnimateFontShadow(pMaterial, pAnimInfo, animTargetOffsets, this->GetFrame());
            break;
        default:
            NW_WARNING(false, "Invalid animation type. ");
            break;
        }
    }
}

bool
AnimTransformBasic::checkBindAnimationDoubly(const void* target, const res::AnimationContent* animCont) const
{
#if defined(NW_RELEASE)

    NW_UNUSED_VARIABLE(target)
    NW_UNUSED_VARIABLE(animCont)

#else
    int end = m_BindPairNum;
    for (int i = 0; i < end; i++)
    {
        NW_ASSERTMSG(m_BindPairAry[i].target != target || m_BindPairAry[i].animCont != animCont,
            "already bind animation[%s].", animCont->name);
    }

#endif  // #if !defined(NW_RELEASE)

    return false;
}

void
AnimTransformBasic::eraseBindPair(int pos)
{
    if (pos + 1 < m_BindPairNum)
    {
        // m_BindPairAry内の順番はアニメーションの挙動には影響しないため、配列の空きを詰める処理は、
        // 一番後ろの要素を持ってくるだけでよい
        m_BindPairAry[pos] = m_BindPairAry[m_BindPairNum - 1];
    }
    m_BindPairNum -= 1;
}

AnimResource::AnimResource()
{
    Init();
}

bool
AnimResource::CheckResource() const
{
    if (!m_pResBlock)
    {
        NW_WARNING(false, "Animation resource is not set.");
        return false;
    }
    return true;
}

void
AnimResource::Set(const void* anmResBuf)
{
    NW_ASSERT_NOT_NULL(anmResBuf);

    Init();

    const ut::BinaryFileHeader *const pFileHeader = static_cast<const ut::BinaryFileHeader*>(anmResBuf);

    if (!ut::IsValidBinaryFile(pFileHeader, res::FILESIGNATURE_FLAN, res::BinaryFileFormatVersion))
    {
        NW_ERR("not valid layout animation file.");
    }

    m_pFileHeader = pFileHeader;

    const ut::BinaryBlockHeader* pDataBlockHead = internal::ConvertOffsToPtr<ut::BinaryBlockHeader>(m_pFileHeader, m_pFileHeader->headerSize);
    for (int i = 0; i < m_pFileHeader->dataBlocks; ++i)
    {
        ut::SigWord kind = pDataBlockHead->kind;
        switch (kind)
        {
        case res::DATABLOCKKIND_PANEANIMTAG:
            m_pTagBlock = reinterpret_cast<const res::AnimationTagBlock*>(pDataBlockHead);
            break;

        case res::DATABLOCKKIND_PANEANIMSHARE:
            m_pShareBlock = reinterpret_cast<const res::AnimationShareBlock*>(pDataBlockHead);
            break;

        case res::DATABLOCKKIND_PANEANIMINFO:
            m_pResBlock = reinterpret_cast<const res::AnimationBlock*>(pDataBlockHead);
            break;
        }

        // 次のブロック位置へ
        pDataBlockHead = internal::ConvertOffsToPtr<ut::BinaryBlockHeader>(pDataBlockHead, pDataBlockHead->size);
    }

    NW_WARNING(m_pResBlock != NULL, "Animation resource is empty.");
}

void
AnimResource::Init()
{
    m_pFileHeader = 0;
    m_pResBlock = 0;
    m_pTagBlock = 0;
    m_pShareBlock = 0;
}

u16
AnimResource::GetTagOrder() const
{
    if (! m_pTagBlock)
    {
        return u16(-1);
    }

    return m_pTagBlock->tagOrder;
}

const char*
AnimResource::GetTagName() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    return internal::ConvertOffsToPtr<const char>(m_pTagBlock, m_pTagBlock->nameOffset);
}

u16
AnimResource::GetGroupNum() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    return m_pTagBlock->groupNum;
}

const res::AnimationGroupRef*
AnimResource::GetGroupArray() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    const res::AnimationGroupRef *const groups = internal::ConvertOffsToPtr<const res::AnimationGroupRef>(m_pTagBlock, m_pTagBlock->groupsOffset);

    return groups;

}

bool
AnimResource::IsDescendingBind() const
{
    if (! m_pTagBlock)
    {
        return false;
    }

    return internal::TestBit(m_pTagBlock->flag, ANIMTAGFLAG_DESCENDINGBIND);
}

u16
AnimResource::GetAnimationShareInfoNum() const
{
    if (! m_pShareBlock)
    {
        return 0;
    }

    return m_pShareBlock->shareNum;
}

const res::AnimationShareInfo*
AnimResource::GetAnimationShareInfoArray() const
{
    if (! m_pShareBlock)
    {
        return 0;
    }

    return internal::ConvertOffsToPtr<const res::AnimationShareInfo>(m_pShareBlock, m_pShareBlock->animShareInfoOffset);
}

u16
AnimResource::CalcAnimationNum(
    Pane* pPane,
    bool bRecursive
) const
{
    if (! CheckResource())
    {
        return 0;
    }

    u16 linkNum = 0;

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(m_pResBlock, m_pResBlock->animContOffsetsOffset);
    for (u16 i = 0; i < m_pResBlock->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(m_pResBlock, animContOffsets[i]);
        if (animCont.type == ANIMCONTENTTYPE_PANE)
        {
            if (Pane *const pFindPane = pPane->FindPaneByName(animCont.name, bRecursive))
            {
                ++linkNum;
            }
        }
        else
        {
            if (Material *const pFindMat = pPane->FindMaterialByName(animCont.name, bRecursive))
            {
                ++linkNum;
            }
        }
    }

    return linkNum;
}

u16
AnimResource::CalcAnimationNum(Material* pMaterial) const
{
    if (! CheckResource())
    {
        return 0;
    }

    u16 linkNum = 0;

    const ut::ResU32 *const animContOffsets = internal::ConvertOffsToPtr<ut::ResU32>(m_pResBlock, m_pResBlock->animContOffsetsOffset);
    for (u16 i = 0; i < m_pResBlock->animContNum; ++i)
    {
        const res::AnimationContent& animCont = *internal::ConvertOffsToPtr<res::AnimationContent>(m_pResBlock, animContOffsets[i]);
        if (animCont.type == ANIMCONTENTTYPE_MATERIAL)
        {
            if (internal::EqualsMaterialName(pMaterial->GetName(), animCont.name))
            {
                ++linkNum;
                break;  // 同じ名前の res::AnimationContent が複数存在することはないため、
                // 見つかったらすぐに抜ける。
            }
        }
    }

    return linkNum;
}

u16
AnimResource::CalcAnimationNum(
    Group* pGroup,
    bool bRecursive
) const
{
    NW_ASSERT_NOT_NULL(pGroup);

    u16 linkNum = 0;

    PaneLinkList& paneList = pGroup->GetPaneList();
    PaneLinkList::Iterator it_end = paneList.GetEndIter();
    for (PaneLinkList::Iterator it = paneList.GetBeginIter(); it != it_end; ++it)
    {
        linkNum += CalcAnimationNum(it->target, bRecursive);
    }

    return linkNum;
}

namespace internal
{

AnimPaneTree::AnimPaneTree()
{
    Init();
}

AnimPaneTree::AnimPaneTree(
    Pane* pTargetPane,
    const AnimResource& animRes)
{
    Init();
    Set(pTargetPane, animRes);
}

const res::AnimationContent*
AnimPaneTree::FindAnimContent(
    const res::AnimationBlock* pAnimBlock,
    const char* animContName,
    u8 animContType
)
{
    NW_ASSERT_NOT_NULL(pAnimBlock);

    const ut::ResU32 *const animContOffsets = ConvertOffsToPtr<ut::ResU32>(pAnimBlock, pAnimBlock->animContOffsetsOffset);
    for (u16 i = 0; i < pAnimBlock->animContNum; ++i)
    {
        const res::AnimationContent *const pAnimCont = ConvertOffsToPtr<res::AnimationContent>(pAnimBlock, animContOffsets[i]);
        if (pAnimCont->type == animContType &&
            internal::EqualsMaterialName(pAnimCont->name, animContName))
        {
            return pAnimCont;
        }
    }

    return NULL;
}

void
AnimPaneTree::Init()
{
    m_LinkNum = 0;

    m_PaneAnimCont = NULL;
    m_AnimMatCnt = 0;
    for (u8 i = 0; i < MATERIAL_NUM_MAX; ++i)
    {
        m_MatAnimConts[i] = NULL;
    }
}

void
AnimPaneTree::Set(
    Pane* pTargetPane,
    const AnimResource& animRes
)
{
    NW_ASSERT_NOT_NULL(pTargetPane);

    u16 linkNum = 0;
    const res::AnimationBlock* pAnimBlock = animRes.GetResourceBlock();

    const res::AnimationContent* animCont = FindAnimContent(pAnimBlock, pTargetPane->GetName(), ANIMCONTENTTYPE_PANE);
    if (animCont != NULL)
    {
        ++linkNum;
    }

    // マテリアルの処理
    const u8 animMatCnt = pTargetPane->GetMaterialNum();
    NW_ASSERT(animMatCnt <= MATERIAL_NUM_MAX);
    const res::AnimationContent* animMatIdxs[MATERIAL_NUM_MAX];

    for (u8 i = 0; i < animMatCnt; ++i)
    {
        animMatIdxs[i] = FindAnimContent(pAnimBlock, pTargetPane->GetMaterial(i)->GetName(), ANIMCONTENTTYPE_MATERIAL);
        if (animMatIdxs[i] != NULL)
        {
            ++linkNum;
        }
    }

    // アニメーション対象がひとつも無い
    if (linkNum == 0)
    {
        return;
    }

    m_AnimRes = animRes;
    m_PaneAnimCont = animCont;
    m_AnimMatCnt = animMatCnt;
    for (u8 i = 0; i < animMatCnt; ++i)
    {
        m_MatAnimConts[i] = animMatIdxs[i];
    }
    m_LinkNum = linkNum;
}

AnimTransform*
AnimPaneTree::Bind(
    Layout* pLayout,
    Pane* pTargetPane,
    ResourceAccessor* pResAccessor
) const
{
    NW_ASSERT_NOT_NULL(pLayout);
    NW_ASSERT_NOT_NULL(pTargetPane);

    AnimTransformBasic *const pAnimTrans = static_cast<AnimTransformBasic*>(pLayout->CreateAnimTransformBasic());
    pAnimTrans->SetResource(m_AnimRes.GetResourceBlock(), pResAccessor, m_LinkNum);

    // ペインのアニメのバインド
    if (m_PaneAnimCont != NULL)
    {
        pAnimTrans->BindPaneImpl(pTargetPane, m_PaneAnimCont);
    }

    // マテリアルアニメのバインド
    // ※マテリアルの個数が共有元ペインと等しいとは限りません。
    const u32 animMatMax = ut::Min(m_AnimMatCnt, pTargetPane->GetMaterialNum());
    for (u32 i = 0; i < animMatMax; ++i)
    {
        if (m_MatAnimConts[i] != NULL)
        {
            Material *const pMaterial = pTargetPane->GetMaterial(i);
            NW_ASSERT_NOT_NULL(pMaterial);
            pAnimTrans->BindMaterialImpl(pMaterial, m_MatAnimConts[i]);
        }
    }

    return pAnimTrans;
}

} // namespace nw::lyt::internal
} // namespace nw::lyt
} // namespace nw
