﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_AXFX_H_
#define NW_WINEXT_AXFX_H_

#include<winext/cafe/ax.h>

namespace nw {
namespace internal {
namespace winext {

/*==========================================================================*
    lib version
 *==========================================================================*/

#define AXFX_VERSION          0x0200      // Version 2.0


/*==========================================================================*
    memory allocation
 *==========================================================================*/

typedef void*   (*AXFXAlloc) (u32);
typedef void    (*AXFXFree)  (void*);

void AXFXSetHooks(AXFXAlloc  alloc, AXFXFree  free);
void AXFXGetHooks(AXFXAlloc *alloc, AXFXFree *free);

BOOL AXFX2SetMemAllocFns(AXFXAlloc alloc, AXFXFree free);
BOOL AXFX2GetMemAllocFns(AXFXAlloc *alloc, AXFXFree *free);

/*==========================================================================*
    buffer update
 *==========================================================================*/

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_BUFFERUPDATE
{
    s32       *left;
    s32       *right;
    s32       *surround;

} AXFX_BUFFERUPDATE;


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_BUFFERUPDATE_DPL2
{
    s32       *L;
    s32       *R;
    s32       *Ls;
    s32       *Rs;

} AXFX_BUFFERUPDATE_DPL2;


/*==========================================================================*
    send bus (for connecting FXs)
 *==========================================================================*/

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_BUS
{
    s32       *left;
    s32       *right;
    s32       *surround;

} AXFX_BUS;


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_BUS_DPL2
{
    s32       *L;
    s32       *R;
    s32       *Ls;
    s32       *Rs;

} AXFX_BUS_DPL2;


/*==========================================================================*
  ==========================================================================

    High Quality Reverb

  ==========================================================================
 *==========================================================================*/

/*==========================================================================*
    expanded version
 *==========================================================================*/

// early reflection type
#define AXFX_REVERBHI_EXP_EARLY_MODE_5MS           0
#define AXFX_REVERBHI_EXP_EARLY_MODE_10MS          1
#define AXFX_REVERBHI_EXP_EARLY_MODE_15MS          2
#define AXFX_REVERBHI_EXP_EARLY_MODE_20MS          3
#define AXFX_REVERBHI_EXP_EARLY_MODE_25MS          4
#define AXFX_REVERBHI_EXP_EARLY_MODE_30MS          5
#define AXFX_REVERBHI_EXP_EARLY_MODE_35MS          6
#define AXFX_REVERBHI_EXP_EARLY_MODE_40MS          7
#define AXFX_REVERBHI_EXP_EARLY_MODE_MAX           8

// fused reverb type
#define AXFX_REVERBHI_EXP_FUSED_MODE_OLD_AXFX      0
#define AXFX_REVERBHI_EXP_FUSED_MODE_METAL_TANK    1
#define AXFX_REVERBHI_EXP_FUSED_MODE_SMALL_ROOM    2
#define AXFX_REVERBHI_EXP_FUSED_MODE_LARGE_ROOM    3
#define AXFX_REVERBHI_EXP_FUSED_MODE_HALL          4
#define AXFX_REVERBHI_EXP_FUSED_MODE_CAVERNOUS     5
#define AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST       6  // do not use this!
#define AXFX_REVERBHI_EXP_FUSED_MODE_MAX           7

#define AXFX_REVHI_EXP_MIN_PREDELAYTIMEMAX      0.0f
//#define AXFX_REVHI_EXP_MAX_PREDELAYTIMEMAX

#define AXFX_REVHI_EXP_MIN_PREDELAYTIME         0.0f
//#define AXFX_REVHI_EXP_MAX_PREDELAYTIME

#define AXFX_REVHI_EXP_MIN_FUSEDTIME            0.0f
//#define AXFX_REVHI_EXP_MAX_FUSEDTIME

#define AXFX_REVHI_EXP_MIN_COLORATION           0.0f
#define AXFX_REVHI_EXP_MAX_COLORATION           1.0f

#define AXFX_REVHI_EXP_MIN_DAMPING              0.0f
#define AXFX_REVHI_EXP_MAX_DAMPING              1.0f

#define AXFX_REVHI_EXP_MIN_CROSSTALK            0.0f
#define AXFX_REVHI_EXP_MAX_CROSSTALK            1.0f

#define AXFX_REVHI_EXP_MIN_EARLYGAIN            0.0f
#define AXFX_REVHI_EXP_MAX_EARLYGAIN            1.0f

#define AXFX_REVHI_EXP_MIN_FUSEDGAIN            0.0f
#define AXFX_REVHI_EXP_MAX_FUSEDGAIN            1.0f

#define AXFX_REVHI_EXP_MIN_OUTGAIN              0.0f
#define AXFX_REVHI_EXP_MAX_OUTGAIN              1.0f

#define AXFX_REVHI_EXP_MIN_SENDGAIN             0.0f
#define AXFX_REVHI_EXP_MAX_SENDGAIN             1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_REVERBHI_EXP
{
    // do not touch these!
    f32       *earlyLine[3];
    u32        earlyPos[3];
    u32        earlyLength;
    u32        earlyMaxLength;
    f32        earlyCoef[3];
    f32       *preDelayLine[3];
    u32        preDelayPos;
    u32        preDelayLength;
    u32        preDelayMaxLength;
    f32       *combLine[3][3];
    u32        combPos[3];
    u32        combLength[3];
    u32        combMaxLength[3];
    f32        combCoef[3];
    f32       *allpassLine[3][2];
    u32        allpassPos[2];
    u32        allpassLength[2];
    u32        allpassMaxLength[2];
    f32       *lastAllpassLine[3];
    u32        lastAllpassPos[3];
    u32        lastAllpassLength[3];
    u32        lastAllpassMaxLength[3];
    f32        allpassCoef;
    f32        lastLpfOut[3];
    f32        lpfCoef;
    u32        active;

    // user params
    u32        earlyMode;        // early reflection mode
    f32        preDelayTimeMax;  // max of pre delay time of fused reverb (sec)
    f32        preDelayTime;     // pre delay time of fused reverb (sec)
    u32        fusedMode;        // fused reverb mode
    f32        fusedTime;        // fused reverb time (sec)
    f32        coloration;       // coloration of all-pass filter (0.f - 1.f)
    f32        damping;          // damping of timbre  (0.f - 1.f)
    f32        crosstalk;        // crosstalk of each channels
    f32        earlyGain;        // output gain of early reflection (0.f - 1.f)
    f32        fusedGain;        // output gain of fused reverb (0.f - 1.f)

    AXFX_BUS  *busIn;
    AXFX_BUS  *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_REVERBHI_EXP;


s32  AXFXReverbHiExpGetMemSize     (AXFX_REVERBHI_EXP *reverb);
BOOL AXFXReverbHiExpInit           (AXFX_REVERBHI_EXP *reverb);
void AXFXReverbHiExpShutdown       (AXFX_REVERBHI_EXP *reverb);
BOOL AXFXReverbHiExpSettings       (AXFX_REVERBHI_EXP *reverb);
BOOL AXFXReverbHiExpSettingsUpdate (AXFX_REVERBHI_EXP *reverb);
void AXFXReverbHiExpCallback       (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBHI_EXP *reverb);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_REVERBHI_EXP_DPL2
{
    // do not touch these!
    f32       *earlyLine[4];
    u32        earlyPos[3];
    u32        earlyLength;
    u32        earlyMaxLength;
    f32        earlyCoef[3];
    f32       *preDelayLine[4];
    u32        preDelayPos;
    u32        preDelayLength;
    u32        preDelayMaxLength;
    f32       *combLine[4][3];
    u32        combPos[3];
    u32        combLength[3];
    u32        combMaxLength[3];
    f32        combCoef[3];
    f32       *allpassLine[4][2];
    u32        allpassPos[2];
    u32        allpassLength[2];
    u32        allpassMaxLength[2];
    f32       *lastAllpassLine[4];
    u32        lastAllpassPos[4];
    u32        lastAllpassLength[4];
    u32        lastAllpassMaxLength[4];
    f32        allpassCoef;
    f32        lastLpfOut[4];
    f32        lpfCoef;
    u32        active;

    // user params
    u32        earlyMode;      // early reflection mode
    f32        preDelayTimeMax;// pre-delay max time of fused reverb (sec)
    f32        preDelayTime;   // pre-delay time of fused reverb (sec)
    u32        fusedMode;      // fused reverb mode
    f32        fusedTime;      // fused reverb time (sec)
    f32        coloration;     // coloration of all-pass filter (0.f-1.f)
    f32        damping;        // damping of timbre  (0.f-1.f)
    f32        crosstalk;      // crosstalk of each channels
    f32        earlyGain;      // output gain of early reflection(0.f-1.f)
    f32        fusedGain;      // output gain of fused reverb (0.f-1.f)

    AXFX_BUS_DPL2 *busIn;
    AXFX_BUS_DPL2 *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_REVERBHI_EXP_DPL2;


s32  AXFXReverbHiExpGetMemSizeDpl2     (AXFX_REVERBHI_EXP_DPL2 *reverb);
BOOL AXFXReverbHiExpInitDpl2           (AXFX_REVERBHI_EXP_DPL2 *reverb);
void AXFXReverbHiExpShutdownDpl2       (AXFX_REVERBHI_EXP_DPL2 *reverb);
BOOL AXFXReverbHiExpSettingsDpl2       (AXFX_REVERBHI_EXP_DPL2 *reverb);
BOOL AXFXReverbHiExpSettingsUpdateDpl2 (AXFX_REVERBHI_EXP_DPL2 *reverb);
void AXFXReverbHiExpCallbackDpl2       (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_REVERBHI_EXP_DPL2 *reverb);


/*==========================================================================*
    classic version
 *==========================================================================*/

#define AXFX_REVHI_MIN_TIME                     0.0f
//#define AXFX_REVHI_MAX_TIME

#define AXFX_REVHI_MIN_PREDELAY                 0.0f
//#define AXFX_REVHI_MAX_PREDELAY

#define AXFX_REVHI_MIN_COLORATION               0.0f
#define AXFX_REVHI_MAX_COLORATION               1.0f

#define AXFX_REVHI_MIN_DAMPING                  0.0f
#define AXFX_REVHI_MAX_DAMPING                  1.0f

#define AXFX_REVHI_MIN_MIX                      0.0f
#define AXFX_REVHI_MAX_MIX                      1.0f

#define AXFX_REVHI_MIN_CROSSTALK                0.0f
#define AXFX_REVHI_MAX_CROSSTALK                1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_REVERBHI
{
    // do not write to these
    AXFX_REVERBHI_EXP  reverbInner;

    // user params
    f32        coloration;
    f32        mix;
    f32        time;
    f32        damping;
    f32        preDelay;
    f32        crosstalk;

} AXFX_REVERBHI;


s32  AXFXReverbHiGetMemSize (AXFX_REVERBHI *reverb);
BOOL AXFXReverbHiInit       (AXFX_REVERBHI *reverb);
BOOL AXFXReverbHiShutdown   (AXFX_REVERBHI *reverb);
BOOL AXFXReverbHiSettings   (AXFX_REVERBHI *reverb);
void AXFXReverbHiCallback   (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBHI *reverb);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_REVERBHI_DPL2
{
    // do not write to these
    AXFX_REVERBHI_EXP_DPL2  reverbInner;

    // user params
    f32        coloration;
    f32        mix;
    f32        time;
    f32        damping;
    f32        preDelay;
    f32        crosstalk;

} AXFX_REVERBHI_DPL2;


s32  AXFXReverbHiGetMemSizeDpl2 (AXFX_REVERBHI_DPL2 *reverb);
BOOL AXFXReverbHiInitDpl2       (AXFX_REVERBHI_DPL2 *reverb);
BOOL AXFXReverbHiShutdownDpl2   (AXFX_REVERBHI_DPL2 *reverb);
BOOL AXFXReverbHiSettingsDpl2   (AXFX_REVERBHI_DPL2 *reverb);
void AXFXReverbHiCallbackDpl2   (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_REVERBHI_DPL2 *reverb);


/*==========================================================================*
  ==========================================================================

    Standard Reverb

  ==========================================================================
 *==========================================================================*/

/*==========================================================================*
    expanded version
 *==========================================================================*/

// early reflection type
#define AXFX_REVERBSTD_EXP_EARLY_MODE_5MS           0
#define AXFX_REVERBSTD_EXP_EARLY_MODE_10MS          1
#define AXFX_REVERBSTD_EXP_EARLY_MODE_15MS          2
#define AXFX_REVERBSTD_EXP_EARLY_MODE_20MS          3
#define AXFX_REVERBSTD_EXP_EARLY_MODE_25MS          4
#define AXFX_REVERBSTD_EXP_EARLY_MODE_30MS          5
#define AXFX_REVERBSTD_EXP_EARLY_MODE_35MS          6
#define AXFX_REVERBSTD_EXP_EARLY_MODE_40MS          7
#define AXFX_REVERBSTD_EXP_EARLY_MODE_MAX           8

// fused reverb type
#define AXFX_REVERBSTD_EXP_FUSED_MODE_OLD_AXFX      0
#define AXFX_REVERBSTD_EXP_FUSED_MODE_METAL_TANK    1
#define AXFX_REVERBSTD_EXP_FUSED_MODE_SMALL_ROOM    2
#define AXFX_REVERBSTD_EXP_FUSED_MODE_LARGE_ROOM    3
#define AXFX_REVERBSTD_EXP_FUSED_MODE_HALL          4
#define AXFX_REVERBSTD_EXP_FUSED_MODE_CAVERNOUS     5
#define AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST       6  // do not use this!
#define AXFX_REVERBSTD_EXP_FUSED_MODE_MAX           7


#define AXFX_REVSTD_EXP_MIN_PREDELAYTIMEMAX     0.0f
//#define AXFX_REVSTD_EXP_MAX_PREDELAYTIMEMAX

#define AXFX_REVSTD_EXP_MIN_PREDELAYTIME        0.0f
//#define AXFX_REVSTD_EXP_MAX_PREDELAYTIME

#define AXFX_REVSTD_EXP_MIN_FUSEDTIME           0.0f
//#define AXFX_REVSTD_EXP_MAX_FUSEDTIME

#define AXFX_REVSTD_EXP_MIN_COLORATION          0.0f
#define AXFX_REVSTD_EXP_MAX_COLORATION          1.0f

#define AXFX_REVSTD_EXP_MIN_DAMPING             0.0f
#define AXFX_REVSTD_EXP_MAX_DAMPING             1.0f

#define AXFX_REVSTD_EXP_MIN_EARLYGAIN           0.0f
#define AXFX_REVSTD_EXP_MAX_EARLYGAIN           1.0f

#define AXFX_REVSTD_EXP_MIN_FUSEDGAIN           0.0f
#define AXFX_REVSTD_EXP_MAX_FUSEDGAIN           1.0f

#define AXFX_REVSTD_EXP_MIN_OUTGAIN             0.0f
#define AXFX_REVSTD_EXP_MAX_OUTGAIN             1.0f

#define AXFX_REVSTD_EXP_MIN_SENDGAIN            0.0f
#define AXFX_REVSTD_EXP_MAX_SENDGAIN            1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_REVERBSTD_EXP
{
    // do not touch these!
    f32       *earlyLine[3];
    u32        earlyPos;
    u32        earlyLength;
    u32        earlyMaxLength;
    f32        earlyCoef;
    f32       *preDelayLine[3];
    u32        preDelayPos;
    u32        preDelayLength;
    u32        preDelayMaxLength;
    f32       *combLine[3][2];
    u32        combPos[2];
    u32        combLength[2];
    u32        combMaxLength[2];
    f32        combCoef[2];
    f32       *allpassLine[3][2];
    u32        allpassPos[2];
    u32        allpassLength[2];
    u32        allpassMaxLength[2];
    f32        allpassCoef;
    f32        lastLpfOut[3];
    f32        lpfCoef;
    u32        active;

    // user params
    u32        earlyMode;        // early reflection mode
    f32        preDelayTimeMax;  // pre-delay max time of fused reverb (sec)
    f32        preDelayTime;     // pre-delay time of fused reverb (sec)
    u32        fusedMode;        // fused reverb mode
    f32        fusedTime;        // fused reverb time (sec)
    f32        coloration;       // coloration of all-pass filter (0.f - 1.f)
    f32        damping;          // damping of timbre  (0.f - 1.f)
    f32        earlyGain;        // output gain of early reflection (0.f - 1.f)
    f32        fusedGain;        // output gain of fused reverb (0.f - 1.f)

    AXFX_BUS  *busIn;
    AXFX_BUS  *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_REVERBSTD_EXP;


s32  AXFXReverbStdExpGetMemSize     (AXFX_REVERBSTD_EXP *reverb);
BOOL AXFXReverbStdExpInit           (AXFX_REVERBSTD_EXP *reverb);
void AXFXReverbStdExpShutdown       (AXFX_REVERBSTD_EXP *reverb);
BOOL AXFXReverbStdExpSettings       (AXFX_REVERBSTD_EXP *reverb);
BOOL AXFXReverbStdExpSettingsUpdate (AXFX_REVERBSTD_EXP *reverb);
void AXFXReverbStdExpCallback       (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBSTD_EXP *reverb);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_REVERBSTD_EXP_DPL2
{
    // do not touch these!
    f32       *earlyLine[4];
    u32        earlyPos;
    u32        earlyLength;
    u32        earlyMaxLength;
    f32        earlyCoef;
    f32       *preDelayLine[4];
    u32        preDelayPos;
    u32        preDelayLength;
    u32        preDelayMaxLength;
    f32       *combLine[4][2];
    u32        combPos[2];
    u32        combLength[2];
    u32        combMaxLength[2];
    f32        combCoef[2];
    f32       *allpassLine[4][2];
    u32        allpassPos[2];
    u32        allpassLength[2];
    u32        allpassMaxLength[2];
    f32        allpassCoef;
    f32        lastLpfOut[4];
    f32        lpfCoef;
    u32        active;

    // user params
    u32        earlyMode;      // early reflection mode
    f32        preDelayTimeMax;// pre-delay max time of fused reverb(sec)
    f32        preDelayTime;   // pre-delay time of fused reverb (sec)
    u32        fusedMode;      // fused reverb mode
    f32        fusedTime;      // fused reverb time (sec)
    f32        coloration;     // coloration of all-pass filter (0.f-1.f)
    f32        damping;        // damping of timbre  (0.f-1.f)
    f32        earlyGain;      // output gain of early reflection(0.f-1.f)
    f32        fusedGain;      // output gain of fused reverb (0.f-1.f)

    AXFX_BUS_DPL2 *busIn;
    AXFX_BUS_DPL2 *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_REVERBSTD_EXP_DPL2;


s32  AXFXReverbStdExpGetMemSizeDpl2     (AXFX_REVERBSTD_EXP_DPL2 *reverb);
BOOL AXFXReverbStdExpInitDpl2           (AXFX_REVERBSTD_EXP_DPL2 *reverb);
void AXFXReverbStdExpShutdownDpl2       (AXFX_REVERBSTD_EXP_DPL2 *reverb);
BOOL AXFXReverbStdExpSettingsDpl2       (AXFX_REVERBSTD_EXP_DPL2 *reverb);
BOOL AXFXReverbStdExpSettingsUpdateDpl2 (AXFX_REVERBSTD_EXP_DPL2 *reverb);
void AXFXReverbStdExpCallbackDpl2       (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_REVERBSTD_EXP_DPL2 *reverb);


/*==========================================================================*
    classic version
 *==========================================================================*/

#define AXFX_REVSTD_MIN_TIME                    0.0f
//#define AXFX_REVSTD_MAX_TIME

#define AXFX_REVSTD_MIN_PREDELAY                0.0f
//#define AXFX_REVSTD_MAX_PREDELAY

#define AXFX_REVSTD_MIN_COLORATION              0.0f
#define AXFX_REVSTD_MAX_COLORATION              1.0f

#define AXFX_REVSTD_MIN_DAMPING                 0.0f
#define AXFX_REVSTD_MAX_DAMPING                 1.0f

#define AXFX_REVSTD_MIN_MIX                     0.0f
#define AXFX_REVSTD_MAX_MIX                     1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_REVERBSTD
{
    AXFX_REVERBSTD_EXP  reverbInner;

    // user params
    f32    coloration;
    f32    mix;
    f32    time;
    f32    damping;
    f32    preDelay;

} AXFX_REVERBSTD;


s32  AXFXReverbStdGetMemSize (AXFX_REVERBSTD *reverb);
BOOL AXFXReverbStdInit       (AXFX_REVERBSTD *reverb);
BOOL AXFXReverbStdShutdown   (AXFX_REVERBSTD *reverb);
BOOL AXFXReverbStdSettings   (AXFX_REVERBSTD *reverb);
void AXFXReverbStdCallback   (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBSTD *reverb);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_REVERBSTD_DPL2
{
    AXFX_REVERBSTD_EXP_DPL2  reverbInner;

    // user params
    f32        coloration;
    f32        mix;
    f32        time;
    f32        damping;
    f32        preDelay;

} AXFX_REVERBSTD_DPL2;


s32  AXFXReverbStdGetMemSizeDpl2 (AXFX_REVERBSTD_DPL2 *reverb);
BOOL AXFXReverbStdInitDpl2       (AXFX_REVERBSTD_DPL2 *reverb);
BOOL AXFXReverbStdShutdownDpl2   (AXFX_REVERBSTD_DPL2 *reverb);
BOOL AXFXReverbStdSettingsDpl2   (AXFX_REVERBSTD_DPL2 *reverb);
void AXFXReverbStdCallbackDpl2   (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_REVERBSTD_DPL2 *reverb);


/*==========================================================================*
  ==========================================================================

    Delay

  ==========================================================================
 *==========================================================================*/

/*==========================================================================*
    expanded version
 *==========================================================================*/

#define AXFX_DELAY_EXP_MIN_MAXDELAY             0.04f // 0.0f < maxDelay
//#define AXFX_DELAY_EXP_MAX_MAXDELAY

#define AXFX_DELAY_EXP_MIN_DELAY                0.04f
//#define AXFX_DELAY_EXP_MAX_DELAY

#define AXFX_DELAY_EXP_MIN_FEEDBACK             0.0f
#define AXFX_DELAY_EXP_MAX_FEEDBACK             0.99f // feedback < 1.0f

#define AXFX_DELAY_EXP_MIN_LPF                  0.0f
#define AXFX_DELAY_EXP_MAX_LPF                  1.0f

#define AXFX_DELAY_EXP_MIN_OUTGAIN              0.0f
#define AXFX_DELAY_EXP_MAX_OUTGAIN              1.0f

#define AXFX_DELAY_EXP_MIN_SENDGAIN             0.0f
#define AXFX_DELAY_EXP_MAX_SENDGAIN             1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_DELAY_EXP
{
    // do not touch these!
    s32       *lineL;
    s32       *lineR;
    s32       *lineS;
    u32        curPos;
    u32        length;
    u32        maxLength;
    s32        feedbackGain;
    s32        lastLpfOut[3];
    s32        lpfCoef;
    s32        outGainCalc;
    s32        sendGainCalc;
    u32        active;

    // user params
    f32        maxDelay;  // max delay time (msec) (>0.f)
    f32        delay;     // delay time (msec) (0.f< - <=maxDelay)
    f32        feedback;  // feedback gain (0.f - <1.f)
    f32        lpf;       // LPF param (0.f - 1.f) (cutoff : low - high)

    AXFX_BUS  *busIn;
    AXFX_BUS  *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_DELAY_EXP;


s32  AXFXDelayExpGetMemSize     (AXFX_DELAY_EXP *delay);
BOOL AXFXDelayExpInit           (AXFX_DELAY_EXP *delay);
void AXFXDelayExpShutdown       (AXFX_DELAY_EXP *delay);
BOOL AXFXDelayExpSettings       (AXFX_DELAY_EXP *delay);
BOOL AXFXDelayExpSettingsUpdate (AXFX_DELAY_EXP *delay);
void AXFXDelayExpCallback       (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_DELAY_EXP *delay);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_DELAY_EXP_DPL2
{
    // do not touch these!
    s32       *line[4];
    u32        curPos;
    u32        length;
    u32        maxLength;
    s32        feedbackGain;
    s32        lastLpfOut[4];
    s32        lpfCoef;
    s32        outGainCalc;
    s32        sendGainCalc;
    u32        active;

    // user params
    f32        maxDelay;  // max delay time (msec) (>0.f)
    f32        delay;     // delay time (msec) (0.f< - <=maxDelay)
    f32        feedback;  // feedback gain (0.f - <1.f)
    f32        lpf;       // LPF param (0.f - 1.f) (cutoff : low - high)

    AXFX_BUS_DPL2 *busIn;
    AXFX_BUS_DPL2 *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_DELAY_EXP_DPL2;


s32  AXFXDelayExpGetMemSizeDpl2     (AXFX_DELAY_EXP_DPL2 *delay);
BOOL AXFXDelayExpInitDpl2           (AXFX_DELAY_EXP_DPL2 *delay);
void AXFXDelayExpShutdownDpl2       (AXFX_DELAY_EXP_DPL2 *delay);
BOOL AXFXDelayExpSettingsDpl2       (AXFX_DELAY_EXP_DPL2 *delay);
BOOL AXFXDelayExpSettingsUpdateDpl2 (AXFX_DELAY_EXP_DPL2 *delay);
void AXFXDelayExpCallbackDpl2       (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_DELAY_EXP_DPL2 *delay);


/*==========================================================================*
    classic version
 *==========================================================================*/

#define AXFX_DELAY_MIN_DELAY                    1
//#define AXFX_DELAY_MAX_DELAY

#define AXFX_DELAY_MIN_FEEDBACK                 0
#define AXFX_DELAY_MAX_FEEDBACK                99

#define AXFX_DELAY_MIN_OUTPUT                   0
#define AXFX_DELAY_MAX_OUTPUT                 100

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_DELAY
{
    // do not touch these!
    s32       *line[3];
    u32        curPos[3];
    u32        length[3];
    s32        feedbackGain[3];
    s32        outGain[3];
    u32        active;

    // user params
    u32        delay[3];       // Delay buffer length in ms per channel
    u32        feedback[3];    // Feedback volume in % per channel
    u32        output[3];      // Output volume in % per channel

} AXFX_DELAY;


s32  AXFXDelayGetMemSize (AXFX_DELAY *delay);
BOOL AXFXDelayInit       (AXFX_DELAY *delay);
void AXFXDelayShutdown   (AXFX_DELAY *delay);
BOOL AXFXDelaySettings   (AXFX_DELAY *delay);
void AXFXDelayCallback   (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_DELAY *delay);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_DELAY_DPL2
{
    // do not touch these!
    s32       *line[4];
    u32        curPos[4];
    u32        length[4];
    s32        feedbackGain[4];
    s32        outGain[4];
    u32        active;

    // user params
    u32        delay[4];       // Delay buffer length in ms per channel
    u32        feedback[4];    // Feedback volume in % per channel
    u32        output[4];      // Output volume in % per channel

} AXFX_DELAY_DPL2;


s32  AXFXDelayGetMemSizeDpl2 (AXFX_DELAY_DPL2 *delay);
BOOL AXFXDelayInitDpl2       (AXFX_DELAY_DPL2 *delay);
void AXFXDelayShutdownDpl2   (AXFX_DELAY_DPL2 *delay);
BOOL AXFXDelaySettingsDpl2   (AXFX_DELAY_DPL2 *delay);
void AXFXDelayCallbackDpl2   (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_DELAY_DPL2 *delay);


/*==========================================================================*
  ==========================================================================

    Chorus

  ==========================================================================
 *==========================================================================*/

/*==========================================================================*
    expanded version
 *==========================================================================*/

#define AXFX_CHORUS_EXP_MIN_DELAYTIME           0.1f
#define AXFX_CHORUS_EXP_MAX_DELAYTIME          50.0f

#define AXFX_CHORUS_EXP_MIN_DEAPTH              0.0f
#define AXFX_CHORUS_EXP_MAX_DEAPTH              1.0f

#define AXFX_CHORUS_EXP_MIN_RATE                0.1f
#define AXFX_CHORUS_EXP_MAX_RATE                2.0f

#define AXFX_CHORUS_EXP_MIN_FEEDBACK            0.0f
#define AXFX_CHORUS_EXP_MAX_FEEDBACK            0.99f // feedback < 1.0f

#define AXFX_CHORUS_EXP_MIN_OUTGAIN             0.0f
#define AXFX_CHORUS_EXP_MAX_OUTGAIN             1.0f

#define AXFX_CHORUS_EXP_MIN_SENDGAIN            0.0f
#define AXFX_CHORUS_EXP_MAX_SENDGAIN            1.0f

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_CHORUS_EXP_DELAY
{
    f32       *line[3];
    u32        inPos;
    u32        outPos;
    u32        lastPos;
    u32        sizeFP;
    u32        size;

} AXFX_CHORUS_EXP_DELAY;


typedef struct AXFX_CHORUS_EXP_LFO
{
    s32       *table;
    s32        phaseAdd;
    s32        stepSamp;
    s32        depthSamp;
    u32        phase;
    u32        sign;
    u32        lastNum;
    s32        lastValue;
    s32        grad;
    s32        gradFactor;

} AXFX_CHORUS_EXP_LFO;


typedef struct AXFX_CHORUS_EXP
{
    // don't touch these
    AXFX_CHORUS_EXP_DELAY  delay;
    AXFX_CHORUS_EXP_LFO    lfo;
    f32        history[3][4];
    u32        histIndex;
    u32        active;

    // user params
    f32        delayTime;   // 0.1 - 50.f (msec)
    f32        depth;       // 0.f - 1.f
    f32        rate;        // 0.1 - 2.f (Hz)
    f32        feedback;    // 0.f - < 1.f

    AXFX_BUS  *busIn;
    AXFX_BUS  *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_CHORUS_EXP;


s32  AXFXChorusExpGetMemSize     (AXFX_CHORUS_EXP *chorus);
BOOL AXFXChorusExpInit           (AXFX_CHORUS_EXP *chorus);
BOOL AXFXChorusExpSettings       (AXFX_CHORUS_EXP *chorus);
BOOL AXFXChorusExpSettingsUpdate (AXFX_CHORUS_EXP *chorus);
void AXFXChorusExpShutdown       (AXFX_CHORUS_EXP *chorus);
void AXFXChorusExpCallback       (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_CHORUS_EXP *chorus);


/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_CHORUS_EXP_DELAY_DPL2
{
    f32       *line[4];
    u32        inPos;
    u32        outPos;
    u32        lastPos;
    u32        sizeFP;
    u32        size;

} AXFX_CHORUS_EXP_DELAY_DPL2;


typedef struct AXFX_CHORUS_EXP_LFO_DPL2
{
    s32       *table;
    s32        phaseAdd;
    s32        stepSamp;
    s32        depthSamp;
    u32        phase;
    u32        sign;
    u32        lastNum;
    s32        lastValue;
    s32        grad;
    s32        gradFactor;

} AXFX_CHORUS_EXP_LFO_DPL2;


typedef struct AXFX_CHORUS_EXP_DPL2
{
    // don't touch these
    AXFX_CHORUS_EXP_DELAY_DPL2  delay;
    AXFX_CHORUS_EXP_LFO_DPL2    lfo;
    f32        history[4][4];
    u32        histIndex;
    u32        active;

    // user params
    f32        delayTime;   // 0.1 - 50.f (msec)
    f32        depth;       // 0.f - 1.f
    f32        rate;        // 0.1f - 2.f (Hz)
    f32        feedback;    // 0.f - < 1.f

    AXFX_BUS_DPL2 *busIn;
    AXFX_BUS_DPL2 *busOut;
    f32        outGain;
    f32        sendGain;

} AXFX_CHORUS_EXP_DPL2;


s32  AXFXChorusExpGetMemSizeDpl2     (AXFX_CHORUS_EXP_DPL2 *chorus);
BOOL AXFXChorusExpInitDpl2           (AXFX_CHORUS_EXP_DPL2 *chorus);
BOOL AXFXChorusExpSettingsDpl2       (AXFX_CHORUS_EXP_DPL2 *chorus);
BOOL AXFXChorusExpSettingsUpdateDpl2 (AXFX_CHORUS_EXP_DPL2 *chorus);
void AXFXChorusExpShutdownDpl2       (AXFX_CHORUS_EXP_DPL2 *chorus);
void AXFXChorusExpCallbackDpl2       (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_CHORUS_EXP_DPL2 *chorus);


/*==========================================================================*
    classic version
 *==========================================================================*/

#define AXFX_CHORUS_MIN_DELAY                   1
#define AXFX_CHORUS_MAX_DELAY                  50

#define AXFX_CHORUS_MIN_VARIATION               0
#define AXFX_CHORUS_MAX_VARIATION              50

#define AXFX_CHORUS_MIN_PERIOD                500
#define AXFX_CHORUS_MAX_PERIOD              10000

/*----------------------------------------------*
    for mono, stereo, surround
 *----------------------------------------------*/

typedef struct AXFX_CHORUS
{
    AXFX_CHORUS_EXP  chorusInner;

    // user params
    u32        baseDelay;      // Base delay of chorus effect in ms
    u32        variation;      // Variation of base delay in ms
    u32        period;         // Period of variational oscilation in ms

} AXFX_CHORUS;


s32  AXFXChorusGetMemSize (AXFX_CHORUS *chorus);
BOOL AXFXChorusInit       (AXFX_CHORUS *chorus);
BOOL AXFXChorusShutdown   (AXFX_CHORUS *chorus);
BOOL AXFXChorusSettings   (AXFX_CHORUS *chorus);
void AXFXChorusCallback   (AXFX_BUFFERUPDATE *bufferUpdate, AXFX_CHORUS *chorus);

/*----------------------------------------------*
    for Dolby prologic2
 *----------------------------------------------*/

typedef struct AXFX_CHORUS_DPL2
{
    AXFX_CHORUS_EXP_DPL2  chorusInner;

    // user params
    u32        baseDelay;      // Base delay of chorus effect in ms
    u32        variation;      // Variation of base delay in ms
    u32        period;         // Period of variational oscilation in ms

} AXFX_CHORUS_DPL2;


s32  AXFXChorusGetMemSizeDpl2 (AXFX_CHORUS_DPL2 *chorus);
BOOL AXFXChorusInitDpl2       (AXFX_CHORUS_DPL2 *chorus);
BOOL AXFXChorusShutdownDpl2   (AXFX_CHORUS_DPL2 *chorus);
BOOL AXFXChorusSettingsDpl2   (AXFX_CHORUS_DPL2 *chorus);
void AXFXChorusCallbackDpl2   (AXFX_BUFFERUPDATE_DPL2 *bufferUpdate, AXFX_CHORUS_DPL2 *chorus);



/*--------------------------------------------------------------------------*

   Sound 1 and 2 common declarations for two, four, and six channel (5.1
   Surround Sound) effects.

 *--------------------------------------------------------------------------*/

#define AXFX_MULTI_CH_MAX_CHANNELS 6

typedef enum _AXFX_SAMPLE_RATE
{
    AXFX_SAMPLE_RATE_INVALID=0,
    AXFX_SAMPLE_RATE_32000,
    AXFX_SAMPLE_RATE_48000,
    AXFX_SAMPLE_RATE_COUNT=AXFX_SAMPLE_RATE_48000
} AXFX_SAMPLE_RATE;

/* Left, Right, LeftSurround, RightSurround, Center, and Subwoofer
   Each an array of AXFX_AUX_BLOCKSIZE=96 or 144 samples */
typedef struct _AXFX_6CH_BUFFERUPDATE
{
    s32 *L;
    s32 *R;
    s32 *Ls;
    s32 *Rs;
    s32 *Ct;
    s32 *Sw;
} AXFX_6CH_BUFFERUPDATE;

// Reverb presets
typedef enum _REVERB_CONFIGURATION_TYPE
{
    REVERB_CONFIGURATION_SMALL_ROOM=1,
    REVERB_CONFIGURATION_LARGE_ROOM,
    REVERB_CONFIGURATION_HALL,
    REVERB_CONFIGURATION_CAVERNOUS_CATHEDRAL,
    REVERB_CONFIGURATION_METAL_CORRIDOR
} REVERB_CONFIGURATION_TYPE;

typedef enum _AXFX_REVERB_MODE
{
    AXFX_REVERB_MODE_INVALID=0,
    AXFX_REVERB_MODE_2CH_MONO_MIXED,
    AXFX_REVERB_MODE_4CH_MONO_MIXED,
    AXFX_REVERB_MODE_6CH_MONO_MIXED,
    AXFX_REVERB_MODE_2CH,
    AXFX_REVERB_MODE_4CH,
    AXFX_REVERB_MODE_6CH,
    AXFX_REVERB_MODE_COUNT=AXFX_REVERB_MODE_6CH
} AXFX_REVERB_MODE;

typedef enum _AXFX_CHORUS_MODE
{
    AXFX_CHORUS_MODE_INVALID=0,
    AXFX_CHORUS_MODE_2CH,
    AXFX_CHORUS_MODE_4CH,
    AXFX_CHORUS_MODE_6CH,
    AXFX_CHORUS_MODE_2CH_2,
    AXFX_CHORUS_MODE_4CH_2,
    AXFX_CHORUS_MODE_6CH_2,
    AXFX_CHORUS_MODE_COUNT=AXFX_CHORUS_MODE_6CH
} AXFX_CHORUS_MODE;

typedef enum _AXFX_DELAY_MODE
{
    AXFX_DELAY_MODE_INVALID=0,
    AXFX_DELAY_MODE_2CH,
    AXFX_DELAY_MODE_4CH,
    AXFX_DELAY_MODE_6CH,
    AXFX_DELAY_MODE_COUNT=AXFX_DELAY_MODE_6CH
} AXFX_DELAY_MODE;

typedef enum AXFX_FLANGER_MODE
{
    AXFX_FLANGER_MODE_INVALID = 0,
    AXFX_FLANGER_MODE_2CH,
    AXFX_FLANGER_MODE_4CH,
    AXFX_FLANGER_MODE_6CH,
    AXFX_FLANGER_MODE_COUNT = AXFX_FLANGER_MODE_6CH
} AXFX_FLANGER_MODE;

typedef enum AXFX_PHASER_MODE
{
    AXFX_PHASER_MODE_INVALID = 0,
    AXFX_PHASER_MODE_2CH,
    AXFX_PHASER_MODE_4CH,
    AXFX_PHASER_MODE_6CH,
    AXFX_PHASER_MODE_COUNT = AXFX_PHASER_MODE_6CH
} AXFX_PHASER_MODE;

typedef enum AXFX_OVERDRIVE_MODE
{
    AXFX_OVERDRIVE_MODE_INVALID = 0,
    AXFX_OVERDRIVE_MODE_2CH,
    AXFX_OVERDRIVE_MODE_4CH,
    AXFX_OVERDRIVE_MODE_6CH,
    AXFX_OVERDRIVE_MODE_COUNT = AXFX_OVERDRIVE_MODE_6CH
} AXFX_OVERDRIVE_MODE;

typedef enum AXFX_COMPRESSOR_MODE
{
    AXFX_COMPRESSOR_MODE_INVALID = 0,
    AXFX_COMPRESSOR_MODE_2CH,
    AXFX_COMPRESSOR_MODE_4CH,
    AXFX_COMPRESSOR_MODE_6CH,
    AXFX_COMPRESSOR_MODE_COUNT = AXFX_COMPRESSOR_MODE_6CH
} AXFX_COMPRESSOR_MODE;

typedef enum AXFX_PITCHSHIFT_MODE
{
    AXFX_PITCHSHIFT_MODE_INVALID = 0,
    AXFX_PITCHSHIFT_MODE_2CH,
    AXFX_PITCHSHIFT_MODE_4CH,
    AXFX_PITCHSHIFT_MODE_6CH,
    AXFX_PITCHSHIFT_MODE_COUNT = AXFX_PITCHSHIFT_MODE_6CH
} AXFX_PITCHSHIFT_MODE;



/*---------------------------------------------------------------------------*
   Sound 1 multi-channel delay line library declarations used by reverb, chorus,
   and delay effects
 *---------------------------------------------------------------------------*/
typedef enum _AXFX_DELAY_TYPE
{
    AXFX_DELAY_TYPE_NONE=0,
    AXFX_DELAY_TYPE_DELAY,
    AXFX_DELAY_TYPE_ALLPASS
} AXFX_DELAY_TYPE;

typedef struct _AXFX_DELAY_LINE {
    AXFX_DELAY_TYPE type;
    s32             size;   // Buffer length in samples.  Must be 1 more than maximum delay size.
    f32             *buff;
    s32             delay;  // Delay sample size.  Maximum value can't be larger than size-1.
    s32             in;
    s32             out;
    f32             coef;   // Required for allpass
} AXFX_DELAY_LINE;

#if 0
typedef struct _AXFX_PS_DELAY_LINE {
    AXFX_DELAY_TYPE type;
    s32             size;   // Buffer length in samples.  Must be 1 more than maximum delay size.
    f32x2           *buff;
    s32             delay;  // Delay sample size.  Maximum value must be size-1.
    s32             in;     // Internal pointer
    s32             out;    // Internal pointer
    f32x2           coef;   // Required for allpass
} AXFX_PS_DELAY_LINE;
#endif


/*---------------------------------------------------------------------------*
   Sound 1 multi-channel reverb effect
 *---------------------------------------------------------------------------*/
#define AXFX_NUM_FDN           4
#define AXFX_NUM_ER_TAPS       10

typedef struct _AXFX_MULTI_CH_REVERB {
    // User configurable parameters
    u32 early_mode;          // presets 0 to 4 for early reflections
    u32 late_mode;           // presets 0 to 4 for late reverb
    f32 predelay_time;       // msec 0 to 100
    f32 decay_time;          // sec 0.1 to 20.0
    f32 hf_decay_ratio;      // 0.1 to 1.0
    f32 coloration;          // 0.0 to 1.0
    f32 early_gain;          // 0.0 to 1.0
    f32 late_gain;           // 0.0 to 1.0
    f32 reverb_gain;         // 0.0 to 1.0

    // DO NOT TOUCH THESE!
    // These are set once to configure the specific reverb algorithm.
    AXFX_REVERB_MODE mode;
    u32 fs;
    // Derived parameters and objects
    f32 early_level;
    f32 late_level;
    f32 reverb_level;
    // early reflections
    AXFX_DELAY_LINE   early_delay[AXFX_MULTI_CH_MAX_CHANNELS]; // Not all filters use 6 delay lines
    u32               early_tap[AXFX_NUM_ER_TAPS];
    f32               early_coef[AXFX_NUM_ER_TAPS];
    // late reverb
    u32          late_tap;
    f32          cos_w_hf;
    f32          ap_coef;
    AXFX_DELAY_LINE center_delay;
    AXFX_DELAY_LINE      fdn_delay[AXFX_NUM_FDN];
    AXFX_DELAY_LINE      ap[AXFX_NUM_FDN];
    f32        fdn_fb[AXFX_NUM_FDN];       // feedback coefficients
    f32        fdn_lp_coef[AXFX_NUM_FDN];  // low-pass filter coefficients
    f32        fdn_lp_z[AXFX_NUM_FDN];
} AXFX_MULTI_CH_REVERB;

s32  AXFXMultiChReverbGetMemSize            (AXFX_MULTI_CH_REVERB *reverb);
BOOL AXFXMultiChReverbInit                  (AXFX_MULTI_CH_REVERB *reverb, AXFX_REVERB_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFXMultiChReverbShutdown              (AXFX_MULTI_CH_REVERB *reverb);
BOOL AXFXMultiChReverbParametersPreset      (AXFX_MULTI_CH_REVERB *reverb, REVERB_CONFIGURATION_TYPE preset);
BOOL AXFXMultiChReverbSettingsUpdate        (AXFX_MULTI_CH_REVERB *reverb);
BOOL AXFXMultiChReverbSettingsUpdateNoReset (AXFX_MULTI_CH_REVERB *reverb);
void AXFXMultiChReverbCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX_MULTI_CH_REVERB *reverb, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 1 multi-channel delay effect
 *---------------------------------------------------------------------------*/
typedef struct _AXFX_MULTI_CH_DELAY {
    // User configurable parameters
    f32 delay_time;        // 0 to 2000.0
    f32 feedback;          // 0.1 to 1.0
    f32 low_pass_amount;   // 0.1 to 1.0
    f32 channel_spread;    // 0.1 to 1.0
    f32 out_gain;          // 0.1 to 1.0
    // DO NOT TOUCH THESE!
    // Derived parameters and objects
    AXFX_DELAY_MODE mode;
    u32 fs;
#if 0
    AXFX_PS_DELAY_LINE psdelay[AXFX_MULTI_CH_MAX_CHANNELS/2]; // Paired singles operate on two f32's at a time
#endif
    f32 fb;
    f32 fb_direct;
    f32 fb_cross;
    f32 lpf_b0;
    f32 lpf_a1;
#if 0
    f32x2 pslpf_z[AXFX_MULTI_CH_MAX_CHANNELS/2];
#endif
    f32 gain;
    f32 max_delay;
} AXFX_MULTI_CH_DELAY;

s32  AXFXMultiChDelayGetMemSize     (AXFX_MULTI_CH_DELAY *delay);
BOOL AXFXMultiChDelayInit           (AXFX_MULTI_CH_DELAY *delay, AXFX_DELAY_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFXMultiChDelayShutdown       (AXFX_MULTI_CH_DELAY *delay);
BOOL AXFXMultiChDelaySettingsUpdate (AXFX_MULTI_CH_DELAY *delay);
BOOL AXFXMultiChDelaySettingsUpdateNoReset (AXFX_MULTI_CH_DELAY *delay);
void AXFXMultiChDelayCallback       (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX_MULTI_CH_DELAY *delay, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 1 multi-channel chorus effect

   Includes support for the previous algorithm as well as an
   updated optimized algorithm
 *---------------------------------------------------------------------------*/
#define AXFX_CHORUS_HIST_SIZE 4

typedef struct AXFX_6CH_CHORUS_DELAY
{
    f32       *line[AXFX_MULTI_CH_MAX_CHANNELS];
    u32        inPos;
    u32        outPos;
    u32        lastPos;
    u32        sizeFP;
    u32        size;
} AXFX_6CH_CHORUS_DELAY;

typedef struct AXFX_6CH_CHORUS_LFO
{
    s32       *table;
    s32        phaseAdd;
    s32        stepSamp;
    s32        depthSamp;
    u32        phase;
    u32        sign;
    u32        lastNum;
    s32        lastValue;
    s32        grad;
    s32        gradFactor;

} AXFX_6CH_CHORUS_LFO;

typedef struct AXFX_MULTI_CH_CHORUS
{
    // don't touch these
    AXFX_CHORUS_MODE mode;
    u32 fs;
    AXFX_6CH_CHORUS_DELAY  delay;
    AXFX_6CH_CHORUS_LFO    lfo;
    f32        history[AXFX_MULTI_CH_MAX_CHANNELS][AXFX_CHORUS_HIST_SIZE];
    u32        histIndex;
    u32        active;
    // user params          // Original Chorus      Chorus_2
                            // -----------------    ------------------------
    f32        delayTime;   // 0.1 - 50.0 (msec)    same as original version
    f32        depth;       // 0.1 -  1.0           0.1 - 1.0
    f32        rate;        // 0.1 -  2.0 (Hz)      same as original version
    f32        feedback;    // 0.0 - <1.0           0.1 - 1.0
    AXFX_6CH_BUFFERUPDATE  *busIn;  //              unused
    AXFX_6CH_BUFFERUPDATE  *busOut; //              unused
    f32        outGain;     // 0.0 - 1.0            same as original version
    f32        sendGain;    // 0.0 - 1.0            unused
} AXFX_MULTI_CH_CHORUS;

s32  AXFXMultiChChorusGetMemSize     (AXFX_MULTI_CH_CHORUS *chorus);
BOOL AXFXMultiChChorusInit           (AXFX_MULTI_CH_CHORUS *chorus, AXFX_CHORUS_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFXMultiChChorusShutdown       (AXFX_MULTI_CH_CHORUS *chorus);
BOOL AXFXMultiChChorusSettings       (AXFX_MULTI_CH_CHORUS *chorus);
BOOL AXFXMultiChChorusSettingsUpdate (AXFX_MULTI_CH_CHORUS *chorus);
BOOL AXFXMultiChChorusSettingsUpdateNoReset (AXFX_MULTI_CH_CHORUS *chorus);
void AXFXMultiChChorusCallback       (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX_MULTI_CH_CHORUS *chorus, AXAUXCBSTRUCT *info);


/*---------------------------------------------------------------------------*
   Sound 2 reverb effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 reverb_gain;         // 0.0 to 1.0
    u32 early_mode;          // presets 0 to 4 for early reflections
    u32 late_mode;           // presets 0 to 4 for late reverb
    f32 predelay_time;       // msec 0 to 100
    f32 decay_time;          // sec 0.1 to 20.0
    f32 hf_decay_ratio;      // 0.1 to 1.0
    f32 coloration;          // 0.0 to 1.0
    f32 early_gain;          // 0.0 to 1.0
    f32 late_gain;           // 0.0 to 1.0
    f32 out_gain;            // 0.0 to 1.0
    f32 dry_gain;            // 0.0 to 1.0
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_REVERB;

s32  AXFX2ReverbGetMemSize             (AXFX_REVERB_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2ReverbInit                   (AXFX2_REVERB *reverb, AXFX_REVERB_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFX2ReverbShutdown               (AXFX2_REVERB *reverb);
BOOL AXFX2ReverbParametersPreset       (AXFX2_REVERB *reverb, REVERB_CONFIGURATION_TYPE preset);
BOOL AXFX2ReverbSettingsUpdate         (AXFX2_REVERB *reverb);
BOOL AXFX2ReverbSettingsUpdateNoReset  (AXFX2_REVERB *reverb);
void AXFX2ReverbCallback               (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_REVERB *reverb, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 reverb I3DL2
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    s32 room;                 // [-10000, 0], mB
    s32 room_hf;              // [-10000, 0], mB
    f32 decay_time;           // [0.1, 20.0], seconds
    f32 decay_hf_ratio;       // [0.1, 2.0], unitless ratio
    s32 reflections;          // [-10000, 0], mB
    f32 reflections_delay;    // [0.0, 0.3], seconds
    s32 reverb;               // [-10000, 20000], mB
    f32 reverb_delay;         // [0.0, 0.1], seconds
    f32 diffusion;            // [0.0, 100.0], percent
    f32 density;              // [0.0, 100.0], percent
    f32 hf_reference;         // [20.0, 20000.0], Hz
    f32 out_gain;             // [0.0, 1.0]
    f32 dry_gain;             // [0.0, 1.0]
    // Do not touch this!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_REVERB_I3DL2;

s32  AXFX2ReverbI3dl2GetMemSize                  (AXFX_REVERB_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2ReverbI3dl2Init                        (AXFX2_REVERB_I3DL2 *reverb_i3dl2, AXFX_REVERB_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFX2ReverbI3dl2Shutdown                    (AXFX2_REVERB_I3DL2 *reverb_i3dl2);
BOOL AXFX2ReverbI3dl2ParametersPreset            (AXFX2_REVERB_I3DL2 *reverb_i3dl2, REVERB_CONFIGURATION_TYPE preset);
void AXFX2ReverbI3dl2ParametersMigrateFromReverb (AXFX2_REVERB_I3DL2 *reverb_i3dl2, AXFX2_REVERB *reverb);
BOOL AXFX2ReverbI3dl2SettingsUpdate              (AXFX2_REVERB_I3DL2 *reverb_i3dl2);
BOOL AXFX2ReverbI3dl2SettingsUpdateNoReset       (AXFX2_REVERB_I3DL2 *reverb_i3dl2);
void AXFX2ReverbI3dl2Callback                    (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_REVERB_I3DL2 *reverb_i3dl2, AXAUXCBSTRUCT *info);


/*---------------------------------------------------------------------------*
   Sound 2 multi-channel delay effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 delay_time;        // 0.0 to max_delay_time
    f32 gain;              // 0.1 to 1.0
    f32 feedback;          // 0.1 to 1.0
    f32 low_pass_amount;   // 0.1 to 1.0
    f32 channel_spread;    // 0.1 to 1.0
    f32 out_gain;          // 0.0 to 1.0
    f32 dry_gain;          // 0.0 to 1.0
    void *priv;  // Private runtime structure
    f32 max_delay_time;    // Unbounded positive delay time in milliseconds
    u32 pad[3];
} AXFX2_DELAY;

s32  AXFX2DelayGetMemSize            (AXFX2_DELAY *delay, AXFX_DELAY_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2DelayInit                  (AXFX2_DELAY *delay, AXFX_DELAY_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFX2DelayShutdown              (AXFX2_DELAY *delay);
BOOL AXFX2DelaySettings              (AXFX2_DELAY *delay);
BOOL AXFX2DelaySettingsUpdate        (AXFX2_DELAY *delay);
BOOL AXFX2DelaySettingsUpdateNoReset (AXFX2_DELAY *delay);
void AXFX2DelayCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_DELAY *delay, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
    Sound 2 multi-channel chorus effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 delay_time;          // 1.0 to 50 ms. The average delay.
    f32 mod_depth;           // 0.1 to 1.0.  How much delay is modulated
    f32 mod_rate;            // 0.1 to 2.0 Hz.  How fast the delay is modulated.
    f32 mod_spread;          // 0.0 to 1.0. Spreads the phase of the modulation between channels.
    f32 feedback;            // 0.1 to 1.0. How much of the delay output is fed back to the input.
    f32 out_gain;            // 0.0 to 1.0. The gain of the effect output.
    f32 dry_gain;            // 0.0 to 1.0. The gain of the dry signal.  Set to non-zero when no dry signal is being added in the mixer.
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_CHORUS;

s32  AXFX2ChorusGetMemSize            (AXFX_CHORUS_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2ChorusInit                  (AXFX2_CHORUS *chorus, AXFX_CHORUS_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFX2ChorusShutdown              (AXFX2_CHORUS *chorus);
BOOL AXFX2ChorusSettingsUpdate        (AXFX2_CHORUS *chorus);
BOOL AXFX2ChorusSettingsUpdateNoReset (AXFX2_CHORUS *chorus);
void AXFX2ChorusCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_CHORUS *chorus, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 multi-channel flanger effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 delay_time; //  0.1  to   7.0 ms
    f32 mod_depth;  //  0.1  to   1.0
    f32 mod_rate;   //  0.1  to  10.0 Hz
    f32 feedback;   // -1.0  to   1.0
    f32 feedforward;//  0.0  to   1.0
    f32 out_gain;   //  0.0  to   1.0
    f32 dry_gain;   //  0.0  to   1.0
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_FLANGER;

s32  AXFX2FlangerGetMemSize            (AXFX_FLANGER_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2FlangerInit                  (AXFX2_FLANGER *flanger, AXFX_FLANGER_MODE mode, AXFX_SAMPLE_RATE sampleRate);
void AXFX2FlangerShutdown              (AXFX2_FLANGER *flanger);
BOOL AXFX2FlangerSettingsUpdate        (AXFX2_FLANGER *flanger);
BOOL AXFX2FlangerSettingsUpdateNoReset (AXFX2_FLANGER *flanger);
void AXFX2FlangerCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_FLANGER *flanger, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 multi-channel phaser effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 notch_freq;     // 70.0  to  1000.0 Hz   The center frequency of the lowest notch or peak
    f32 notch_spread;   //  0.0  to     1.0      How far apart the three notches are
    f32 feedback;       //  0.0  to     1.0      Adding feedback increases the size of the notches or peak
    f32 mod_depth;      //  0.0  to     1.0      How much the LFO modulates the notch/peak frequencies
    f32 mod_rate;       //  0.1  to     5.0 Hz   How fast the notche/peak frequencies
    f32 mod_spread;     //  0.0  to     1.0      Spreads the phase of the modulation between channels
    u32 notch_or_peak;  // 0, 1                  When this is 0 the phaser has notches.  When this is 1 there are peaks instead.
    f32 out_gain;       //  0.0  to     2.0      The output level of the effect
    f32 dry_gain;       //  0.0  to     1.0      The level of the mixed input signal
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_PHASER;

s32  AXFX2PhaserGetMemSize            (AXFX_PHASER_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2PhaserInit                  (AXFX2_PHASER *phaser, AXFX_PHASER_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2PhaserSettingsUpdate        (AXFX2_PHASER *phaser);
BOOL AXFX2PhaserSettingsUpdateNoReset (AXFX2_PHASER *phaser);
void AXFX2PhaserShutdown              (AXFX2_PHASER *phaser);
void AXFX2PhaserCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_PHASER *phaser, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 multi-channel overdrive effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 drive;          // -30.0  to  30.0 dB  The gain applied to the input signal before the distortion
    u32 distortion_type;// 0, 1, 2             The type of distorion: 0=soft clip, 1=medium curve, 2=hard clip
    f32 out_gain;       // -30.0  to  10.0 dB  Gain applied after the distortion
    f32 tone;           //  -1.0  to   1.0     Negative values emphasize low frequencies. Positive values emphasize high frequenceis
    f32 dry_gain;       //   0.0  to   2.0     The level of the original dry signal mixed in to the output
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_OVERDRIVE;

s32  AXFX2OverdriveGetMemSize     (AXFX_OVERDRIVE_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2OverdriveInit           (AXFX2_OVERDRIVE *overdrive, AXFX_OVERDRIVE_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2OverdriveSettingsUpdate (AXFX2_OVERDRIVE *overdrive);
void AXFX2OverdriveShutdown       (AXFX2_OVERDRIVE *overdrive);
void AXFX2OverdriveCallback       (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_OVERDRIVE *overdrive, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 multi-channel compressor effect
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 threshold;       // -50.0  to     0.0 dB    The level the signal must reach to start being compressed
    f32 ratio;           //   1.0  to    30.0       The amount of compression that occurs once the signal exceeds the threshold. 1.0 == no compression
    f32 attack_time;     //   1.0  to  1000.0 msec  The time it takes the gain to adjust when the signal level increases
    f32 release_time;    //   1.0  to  5000.0 msec  The time it takes the gain to adjust when the signal level decreases
    u32 makeup_gain;     // 0, 1                    Whether the overall gain is boosted to compensate for the reduced maximum output level caused by compression.  0=off, 1=on
    f32 out_gain;        // -30.0  to    10.0 dB    Gain applied after the compressor. 0dB = gain of 1.0
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_COMPRESSOR;

s32  AXFX2CompressorGetMemSize     (AXFX_COMPRESSOR_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2CompressorInit           (AXFX2_COMPRESSOR *compressor, AXFX_COMPRESSOR_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2CompressorSettingsUpdate (AXFX2_COMPRESSOR *compressor);
void AXFX2CompressorShutdown       (AXFX2_COMPRESSOR *compressor);
void AXFX2CompressorCallback       (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_COMPRESSOR *compressor, AXAUXCBSTRUCT *info);



/*---------------------------------------------------------------------------*
   Sound 2 Pitchshift
 *---------------------------------------------------------------------------*/
typedef struct
{
    // User params
    f32 pitch_shift;   // -24.0 to +24.0. The amount of pitch shift in semitones
    f32 out_gain;      // 0.0 to 2.0. The gain of the effect output.
    f32 dry_gain;      // 0.0 to 2.0. The gain of the dry signal.
    // Do not touch these!
    void *priv;  // Private runtime structure
    u32 pad[4];
} AXFX2_PITCHSHIFT;

s32  AXFX2PitchshiftGetMemSize            (AXFX_PITCHSHIFT_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2PitchshiftInit                  (AXFX2_PITCHSHIFT *pitchshift, AXFX_PITCHSHIFT_MODE mode, AXFX_SAMPLE_RATE sampleRate);
BOOL AXFX2PitchshiftSettingsUpdate        (AXFX2_PITCHSHIFT *pitchshift);
BOOL AXFX2PitchshiftSettingsUpdateNoReset (AXFX2_PITCHSHIFT *pitchshift);
void AXFX2PitchshiftShutdown              (AXFX2_PITCHSHIFT *pitchshift);
void AXFX2PitchshiftCallback              (AXFX_6CH_BUFFERUPDATE *bufferUpdate, AXFX2_PITCHSHIFT *pitchshift, AXAUXCBSTRUCT *info);


} // namespace winext
} // namespace internal
} // namespace nw

#endif // NW_WINEXT_AXFX_H_
