﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_VWRLYT_INPUTS_H_
#define NW_VWRLYT_INPUTS_H_

#include <nw/vwrlyt/vwrlyt_Config.h>
#if defined(NW_VWRLYT_ENABLE)

#include <nw/math.h>
#include <nw/dev.h>
#include <nw/dw.h>
#include <nw/lyt.h>

namespace nw
{
namespace vwrlyt
{

class Inputs;

namespace internal
{

class UnifiedPad : public nw::dev::Pad
{
public:
    NW_UT_RUNTIME_TYPEINFO(nw::dev::Pad)

    explicit UnifiedPad(Inputs* pInputs);

protected:
    //! @brief パッド更新の実装です。
    virtual void UpdateImpl();

private:
    void UpdateValue(f32* pValue, f32* pPrev, f32 newValue)
    {
        if (*pPrev != newValue)
        {
            *pPrev = newValue;
            *pValue = newValue;
        }
    }

    void UpdateValue(nw::math::VEC2* pValue, nw::math::VEC2* pPrev, const nw::math::VEC2& newValue)
    {
        if (*pPrev != newValue)
        {
            *pPrev = newValue;
            *pValue = newValue;
        }
    }

protected:
    Inputs* m_pInputs;
    f32 m_PadLeftAnalogTrigger;
    f32 m_PadRightAnalogTrigger;
    nw::math::VEC2 m_PadLeftStick;
    nw::math::VEC2 m_PadRightStick;
};

class UnifiedMouse : public nw::dev::Mouse
{
public:
    NW_UT_RUNTIME_TYPEINFO(nw::dev::Mouse)

    explicit UnifiedMouse(Inputs* pInputs);

protected:
    //! @brief マウス入力更新の実装です。
    virtual void UpdateImpl();

protected:
    Inputs* m_pInputs;
};

} // namespace internal

class IInputs
{
public:
    virtual ~IInputs() {}

    //------------------------------------------------------------------
    //! @brief Pad を取得します。
    //!
    //! @return Pad へのポインタを返します。
    //------------------------------------------------------------------
    virtual const nw::dev::Pad* GetPad() const = 0;

    //------------------------------------------------------------------
    //! @brief Mouse を取得します。
    //!
    //! @return Mouse へのポインタを返します。
    //------------------------------------------------------------------

    virtual const nw::dev::Mouse* GetMouse() const = 0;

    //------------------------------------------------------------------
    //! @brief VPad 取得します。
    //!
    //! @return VPad へのポインタを返します。
    //------------------------------------------------------------------
    virtual const nw::dev::Pad* GetVPad() const = 0;
};

class Inputs : public IInputs
{
public:
    Inputs();

    virtual ~Inputs();

    //------------------------------------------------------------------
    //! @brief メイン画面のサイズを指定します。
    //!
    //! @param[in] size メイン画面のサイズです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetMainViewportSize(const nw::lyt::Size& size)
    {
        m_MainViewportSize = size;
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief メイン画面のサイズを指定します。
    //!
    //! @param[in] width    メイン画面の幅です。
    //! @param[in] height   メイン画面の高さです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetMainViewportSize(s32 width, s32 height)
    {
        return this->SetMainViewportSize(nw::lyt::Size(f32(width), f32(height)));
    }

    //------------------------------------------------------------------
    //! @brief メイン画面のサイズを取得します。
    //!
    //! @return メイン画面のサイズを返します。
    //------------------------------------------------------------------
    const nw::lyt::Size& GetMainViewportSize() const
    {
        return m_MainViewportSize;
    }

    //------------------------------------------------------------------
    //! @brief Pad を設定します。
    //!
    //! @param[in] pPad Pad へのポインタです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetPadSrc(nw::dev::Pad* pPad)
    {
        m_pPad = pPad;
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief Pad の設定値を取得します。
    //!
    //! @return Pad へのポインタを返します。
    //------------------------------------------------------------------
    nw::dev::Pad* GetPadSrc() const
    {
        return m_pPad;
    }

    //------------------------------------------------------------------
    //! @brief VPad を設定します。
    //!
    //! @param[in] pVPad VPad へのポインタです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetVPadSrc(nw::dev::Pad* pVPad)
    {
        m_pVPad = pVPad;
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief VPad の設定値を取得します。
    //!
    //! @return VPad へのポインタを返します。
    //------------------------------------------------------------------
    nw::dev::Pad* GetVPadSrc() const
    {
        return m_pVPad;
    }

    //------------------------------------------------------------------
    //! @brief WPad を設定します。
    //!
    //! @param[in] pWPad WPad へのポインタです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetWPadSrc(nw::dev::Pad* pWPad)
    {
        m_pWPad = pWPad;
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief WPad の設定値を取得します。
    //!
    //! @return WPad へのポインタを返します。
    //------------------------------------------------------------------
    nw::dev::Pad* GetWPadSrc() const
    {
        return m_pWPad;
    }

    //------------------------------------------------------------------
    //! @brief Mouse を設定します。
    //!
    //! @param[in] pMouse Mouse へのポインタです。
    //!
    //! @return Inputs 自身への参照を返します。
    //------------------------------------------------------------------
    Inputs& SetMouseSrc(nw::dev::Mouse* pMouse)
    {
        m_pMouse = pMouse;
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief Mouse の設定値を取得します。
    //!
    //! @return Mouse へのポインタを返します。
    //------------------------------------------------------------------
    nw::dev::Mouse* GetMouseSrc() const
    {
        return m_pMouse;
    }

    //------------------------------------------------------------------
    //! @brief 情報を更新します。
    //------------------------------------------------------------------
    virtual void Update();

    //------------------------------------------------------------------
    //! @brief 入力の統合された Pad を取得します。
    //!
    //! @return 入力の統合された Pad へのポインタを返します。
    //------------------------------------------------------------------
    virtual const nw::dev::Pad* GetPad() const
    {
        return &m_UnifiedPad;
    }

    //------------------------------------------------------------------
    //! @brief 入力の統合された Pad を取得します。
    //!
    //! @return 入力の統合された Pad へのポインタを返します。
    //------------------------------------------------------------------
    virtual const nw::dev::Mouse* GetMouse() const
    {
        return &m_UnifiedMouse;
    }

    //------------------------------------------------------------------
    //! @brief VPad を取得します。
    //!
    //! @return VPad へのポインタを返します。
    //------------------------------------------------------------------
    virtual const nw::dev::Pad* GetVPad() const
    {
        return m_pVPad;
    }

protected:
    nw::dev::Pad* m_pPad;
    nw::dev::Pad* m_pVPad;
    nw::dev::Pad* m_pWPad;
    nw::dev::Mouse* m_pMouse;
    internal::UnifiedPad m_UnifiedPad;
    internal::UnifiedMouse m_UnifiedMouse;
    nw::lyt::Size m_MainViewportSize;
};

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
#endif
