﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MEMORY_ALLOCATOR_H_
#define NW_UT_MEMORY_ALLOCATOR_H_

#include <nw/types.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/mem.h>
#else
  #include <winext/cafe/mem.h>
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief   メモリアロケータです。
//---------------------------------------------------------------------------
class MemoryAllocator : public nw::ut::IAllocator
{
public:
    NW_UT_RUNTIME_TYPEINFO(nw::ut::IAllocator);

    //! @brief コンストラクタです。
    MemoryAllocator()
    : m_HeapHandle(NULL)
    {}

    //=====================
    //! @name 作成／破棄
    //@{

    //! @brief アロケータを初期化します。
    //!        内部で拡張ヒープを初期化します。
    //!
    //! @param[in] startAddress 先頭アドレスです。
    //! @param[in] size アロケートできるサイズです。
    //! @param[in] option 拡張ヒープのオプションです。
    //!
    void Initialize(void* startAddress, size_t size, u16 option = 0);

    //! @brief アロケータを破棄します。
    //!        内部で拡張ヒープの破棄を行います。
    //!        拡張ヒープの破棄はデストラクト時には行われませんので、
    //!        破棄を行う場合には明示的に Finalize を呼ぶ必要があります。
    //!
    void Finalize();

    //@}


    //! @brief 初期化時に指定した先頭アドレスを取得します。
    //!
    //! @return 先頭アドレスです。
    void* GetStartAddress() { return MEMGetHeapStartAddress(m_HeapHandle); }

    //! @brief 初期化時に指定した終端アドレスを取得します。
    //!
    //! @return 終端アドレスです。
    void* GetEndAddress() { return MEMGetHeapEndAddress(m_HeapHandle); }

    //=====================
    //! @name メモリの確保と解放
    //@{

    //! @brief メインメモリからメモリを確保します。
    //!        アライメントは 2 の累乗(1、2、4、8、16、…)で指定する必要があります。
    //!
    //! @param[in] size      確保するメモリサイズです。
    //! @param[in] alignment アライメントです。
    //!
    //! @return 確保したメモリアドレスを返します。
    virtual void* Alloc(size_t size, u32 alignment)
    {
        NW_ASSERT(size != 0);
        NW_ASSERTMSG((alignment & ~(alignment - 1)) == alignment, "alignment should be power of two");

    #if defined(NW_PLATFORM_CAFE)
        // プラットフォームが許可する最小アライメント未満は切り上げる
        if (alignment < 4)
        {
            alignment = 4;
        }
    #endif

        void* memory = MEMAllocFromExpHeapEx(m_HeapHandle, static_cast<u32>(size), alignment);

    #if defined(NW_DEBUG_TRAP_ALLOCATOR)
        if (memory != NULL)
        {
            void* unalignMemory = UnAlignMemory(memory);
            if (unalignMemory == m_BreakAllocAddress &&
                (m_BreakAllocSize == 0 || GetMemoryBlockSize(unalignMemory) == m_BreakAllocSize))
            {
                NW_DEV_LOG("Alloc(): Specified memory block allocated\n");
            #if defined(NW_DEBUG_BREAK_ALLOCATOR)
                nn::dbg::Break(nn::dbg::BREAK_REASON_USER);
            #endif
            }
        }
    #endif

        return memory;
    }

    using nw::ut::IAllocator::Alloc;

    //! @brief メインメモリにメモリを返却します。
    //!
    //! @param[in] memory    返却するメモリアドレスです。
    virtual void Free(void* memory)
    {
    #if defined(NW_DEBUG_TRAP_ALLOCATOR)
        void* unalignMemory = UnAlignMemory(memory);
        if (unalignMemory == m_BreakFreeAddress &&
            (m_BreakFreeSize == 0 || GetMemoryBlockSize(unalignMemory) == m_BreakFreeSize))
        {
            NW_DEV_LOG("Free(): Specified memory block freed\n");
        #if defined(NW_DEBUG_BREAK_ALLOCATOR)
            nn::dbg::Break(nn::dbg::BREAK_REASON_USER);
        #endif
        }
    #endif
        MEMFreeToExpHeap(m_HeapHandle, memory);
    }

    //@}
    //=====================
    //! @name デバッグ
    //@{

    //---------------------------------------------------------------------------
    //! @brief       空きメモリサイズを返します。
    //!
    //! @return      空きメモリサイズです。
    //---------------------------------------------------------------------------
    size_t GetFreeSize();

    //---------------------------------------------------------------------------
    //! @brief       割り当てられたメモリサイズを返します。
    //!
    //! @return      割り当てられたメモリサイズです。
    //---------------------------------------------------------------------------
    size_t GetTotalSize();

    //---------------------------------------------------------------------------
    //! @brief       確保されたメモリブロックのサイズを返します。
    //!
    //! @param[in]   address メモリブロックの先頭アドレスを指定します。
    //!
    //! @return      指定されたブロックのサイズです。
    //---------------------------------------------------------------------------
    size_t GetMemoryBlockSize(void* address);


    //! @brief  メインメモリの状態を表示します。
    void Dump();

    //! @brief  メモリ確保時の停止条件を設定します。
    //!
    //!         指定したアドレスとサイズのメモリ確保が行われるときにプログラムを強制的に一時停止するように設定します。
    //!         NW_DEBUG_TRAP_ALLOCATOR マクロと NW_DEBUG_BREAK_ALLOCATOR マクロを定義することでこの機能は有効になります。
    //!         NW_DEBUG_TRAP_ALLOCATOR マクロのみを定義した場合はメッセージ出力のみを行います。
    //!         サイズに 0 を指定するとアドレスのみを条件とします。
    //!
    //!         アドレス、サイズともに Dump() で表示される数値(実際に確保されたアドレスとサイズ)を指定します。
    //!
    //! @param  address ブレーク条件のアドレスです。
    //! @param  size ブレーク条件のサイズです。
    void SetBreakAlloc(int address, size_t size)
    {
#if defined(NW_DEBUG_TRAP_ALLOCATOR)
        m_BreakAllocAddress = reinterpret_cast<void *>(address);
        m_BreakAllocSize = size;
#else
        NW_UNUSED_VARIABLE(address);
        NW_UNUSED_VARIABLE(size);
#endif
    }

    //! @brief  メモリ解放時の停止条件を設定します。
    //!
    //!         指定したアドレスとサイズのメモリが解放されるときにプログラムを強制的に一時停止するように設定します。
    //!         NW_DEBUG_TRAP_ALLOCATOR マクロと NW_DEBUG_BREAK_ALLOCATOR マクロを定義することでこの機能は有効になります。
    //!         NW_DEBUG_TRAP_ALLOCATOR マクロのみを定義した場合はメッセージ出力のみを行います。
    //!         サイズに 0 を指定するとアドレスのみを条件とします。
    //!
    //!         アドレス、サイズともに Dump() で表示される数値(実際に確保されたアドレスとサイズ)を指定します。
    //!
    //! @param  address ブレーク条件のアドレスです。
    //! @param  size ブレーク条件のサイズです。
    void SetBreakFree(int address, size_t size)
    {
#if defined(NW_DEBUG_TRAP_ALLOCATOR)
        m_BreakFreeAddress = reinterpret_cast<void *>(address);
        m_BreakFreeSize = size;
#else
        NW_UNUSED_VARIABLE(address);
        NW_UNUSED_VARIABLE(size);
#endif
    }

    //@}

private:
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    typedef nw::internal::winext::MEMHeapHandle MEMHeapHandle;
#elif defined ( NW_PLATFORM_ANDROID ) || defined(NW_PLATFORM_IOS)
    typedef nw::internal::winext::MEMHeapHandle MEMHeapHandle;
#endif

    MEMHeapHandle m_HeapHandle;
};

} // namespace ut
} // namespace nw

/* NW_UT_MEMORY_ALLOCATOR_H_ */
#endif

