﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_FS_FILE_STREAM_H_
#define NW_UT_FS_FILE_STREAM_H_

#include <nw/ut/ut_FileStream.h>

#if defined(NW_PLATFORM_CAFE)
  #include <cafe/fs.h>
#else
  #include <winext/cafe/fs.h>
#endif

namespace nw   {
namespace ut   {

//---------------------------------------------------------------------------
//! @brief     FSファイルストリームクラスです。
//---------------------------------------------------------------------------
class FsFileStream : public FileStream
{
//------ publicメンバ --------------------------
public:
    NW_UT_RUNTIME_TYPEINFO( FileStream )   // ダウンキャスト用の実行時型情報を埋め込みます

    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //---------------------------------------------------------------------------
    FsFileStream();

    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //!
    //! @param[in]   fileStream  FsFileStream のポインタです。
    //---------------------------------------------------------------------------
    /* ctor */ NW_DEPRECATED_FUNCTION(explicit FsFileStream( const FsFileStream* fileStream ));

    //---------------------------------------------------------------------------
    //! @brief       ストリームをオープンします。
    //!
    //! @param[in]   client      クライアントです。
    //! @param[in]   cmdBlock    コマンドブロックです。
    //! @param[in]   filePath    ファイルパスです。
    //! @param[in]   mode        オープンモードです。
    //!
    //! @return      オープンが成功した場合には真を返し、失敗した場合は偽を返します。
    //---------------------------------------------------------------------------
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    typedef nw::internal::winext::FSClient FSClient;
    typedef nw::internal::winext::FSCmdBlock FSCmdBlock;
#elif defined ( NW_PLATFORM_ANDROID ) || defined(NW_PLATFORM_IOS)
    typedef nw::internal::winext::FSClient FSClient;
    typedef nw::internal::winext::FSCmdBlock FSCmdBlock;
#endif // defined( NW_PLATFORM_WIN32 )
    bool Open( FSClient* client, FSCmdBlock* cmdBlock, const char* filePath, const char* mode );

    //---------------------------------------------------------------------------
    //! @brief        ストリームをクローズします。
    //---------------------------------------------------------------------------
    virtual void        Close();

    //---------------------------------------------------------------------------
    //! @brief       ストリームからデータを読み込みます(同期処理)
    //!
    //! @param[in]   buf         読み込み先バッファへのポインタ
    //! @param[in]   length      読み込むデータ長
    //!
    //! @return      実際に読み込まれたデータサイズを返します。
    //---------------------------------------------------------------------------
    virtual s32         Read( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       ストリームからデータを読み込みます(非同期処理)
    //!
    //! @param[in]   buf         読み込み先バッファへのポインタ
    //! @param[in]   length      読み込むデータ長
    //! @param[in]   callback    読み込み処理完了時のコールバック
    //! @param[in]   arg         読み込み処理完了時のコールバック引数
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には 偽 を返します。
    //!              現状は常に偽を返すため、使用時はオーバーライドして実装頂く必要があります。
    //---------------------------------------------------------------------------
    virtual bool        ReadAsync(
                            void*               buf,
                            u32                 length,
                            IoStreamCallback    callback,
                            void*               arg );

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(同期版)
    //!
    //! @param[in]   buf         書き込み元バッファ
    //! @param[in]   length      書き込みデータのバイト数
    //!
    //! @return      実際に書き込まれたバイト数を返します。
    //!               書き込みに失敗した場合には負のエラーコードを返します。
    //---------------------------------------------------------------------------
    virtual s32         Write( const void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(非同期版)
    //!
    //! @param[in]   buf         書き込み元バッファへのポインタ
    //! @param[in]   length      書き込むデータ長
    //! @param[in]   callback    非同期処理完了時のコールバック
    //! @param[in]   arg         非同期処理完了時のコールバック引数
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //!              現状は常に偽を返すため、使用時はオーバーライドして実装頂く必要があります。
    //---------------------------------------------------------------------------
    virtual bool        WriteAsync(
                            const void*         buf,
                            u32                 length,
                            IoStreamCallback    callback,
                            void*               arg );

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の完了を待ちます
    //!
    //! @return      非同期処理の結果を返します。
    //!               コマンドが失敗した場合には負のエラーコードを返します。
    //!               コマンドが成功した場合にはそれ以外の値を返します。
    //---------------------------------------------------------------------------
    virtual s32         WaitAsync();

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の完了を待ちます
    //!
    //! @return      処理中の場合には真を返します。
    //!               処理中でない場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool        IsBusy() const { return FileStream::IsBusy(); }

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の可否を取得します。
    //!
    //! @return      非同期処理が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanAsync () const { return false; }

    //---------------------------------------------------------------------------
    //! @brief       Read 処理の可否を取得します。
    //!
    //! @return      Read が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanRead  () const;

    //---------------------------------------------------------------------------
    //! @brief       Write 処理の可否を取得します。
    //!
    //! @return      Write が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanWrite () const;

    //---------------------------------------------------------------------------
    //! @brief        ストリームがオープンできているかを取得します。
    //!
    //! @return       ストリームがオープンできている場合には true を返します。
    //---------------------------------------------------------------------------
    virtual bool                IsAvailable()    const { return m_Available; }

    //---------------------------------------------------------------------------
    //! @brief       ファイルサイズを取得します。
    //!
    //! @return      ファイルサイズを返します。
    //---------------------------------------------------------------------------
    virtual u32         GetSize() const;

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタを移動します。
    //!
    //! @param[in]    offset  ファイルポインタ移動値を指定します。
    //! @param[in]    origin  ファイルポインタ移動の基準点を指定します。
    //!                       FILE_STREAM_SEEK_BEGIN      - ファイルの先頭を基準
    //!                       FILE_STREAM_SEEK_CURRENT    - 現在の読み込み位置を基準
    //!                       FILE_STREAM_SEEK_END        - ファイルの終端を基準(offset <= 0で指定されるはずです)
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool        Seek( s32 offset, u32 origin );

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(同期関数)
    //---------------------------------------------------------------------------
    virtual void        Cancel();

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(非同期関数)
    //!
    //! @param[in]    callback  処理完了時のコールバック関数です。
    //! @param[out]   arg       コールバックで返されるユーザパラメータです。
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool        CancelAsync( IoStreamCallback callback, void* arg );

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタをランダムに移動できるかどうかの属性を取得します。
    //!
    //! @return       Seek が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanSeek  () const { return true; }

    //---------------------------------------------------------------------------
    //! @brief        ファイルの Read をキャンセルできるかどうかの属性を取得します。
    //!
    //! @return       キャンセルが可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanCancel() const { return false; }

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタの位置を取得します。
    //!
    //! @return       ポインタの位置を返します。
    //---------------------------------------------------------------------------
    virtual u32         Tell()  const { return m_FilePos; }


private:
    bool m_Available;
    bool m_DuplicateFlag;

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    typedef ::nw::internal::winext::FSClient FSClient;
    typedef ::nw::internal::winext::FSCmdBlock FSCmdBlock;
    typedef ::nw::internal::winext::FSFileHandle FSFileHandle;
    typedef ::nw::internal::winext::FSStat FSStat;
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    typedef ::nw::internal::winext::FSStat FSStat;
    typedef ::nw::internal::winext::FSFileHandle FSFileHandle;
#endif // defined( NW_PLATFORM_WIN32 )
    FSClient*     m_Client;
    FSCmdBlock*   m_CmdBlock;
    FSFileHandle  m_FileHandle;

    FSStat m_FileStat;
    u32 m_FilePos;
    u32 m_FileSize;
};

}   /* namespace ut   */
}   /* namespace nw   */


/* NW_UT_FS_FILE_STREAM_H_ */
#endif
