﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_FLAG_H_
#define NW_UT_FLAG_H_

#include <nw/types.h>

#define NW_FLAG_DECLARE(mflag, mbit)    \
    FLAG_##mflag##_SHIFT = (mbit),      \
    FLAG_##mflag = 0x1 << (mbit)

#define NW_FLAG_VALUE_DECLARE(mflag, mbit, msize)   \
    FLAG_##mflag##_VALUE_SHIFT = (mbit),            \
    FLAG_##mflag##_VALUE_MASK = (((0x1UL << (msize)) - 0x1) << (mbit))

//#define NW_GET_FLAG_VALUE(MFLAGS, MNAME) \
//    GetFlagValue((MFLAGS), (FLAG_##MNAME##_VALUE_SHIFT), (FLAG_##MNAME##_VALUE_MASK))

namespace nw {
namespace ut {

//----------------------------------------
//! @name フラグ操作関連
//@{

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のマスクのビットが全て真かチェックします。
//!
//! @param        flags   現在のビットフィールドです。
//! @param        mask    チェックに用いるマスクです。
//! @return       ビットフィールドの指定のマスクのビットが全て真であれば true を返します。
//---------------------------------------------------------------------------
template<typename TFlags, typename TMask>
NW_INLINE bool
CheckFlag(const TFlags& flags, const TMask& mask)
{
    //NW_STATIC_ASSERT(!ut::IsPointer(TValue));
    return (flags & static_cast<TFlags>(mask)) == static_cast<TFlags>(mask);
}

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のマスクのビットのいずれかが真かチェックします。
//!
//! @param        flags   現在のビットフィールドです。
//! @param        mask    チェックに用いるマスクです。
//! @return       ビットフィールドの指定のマスクのビットのいずれかが真であれば true を返します。
//---------------------------------------------------------------------------
template<typename TFlags, typename TMask>
NW_INLINE bool
CheckFlagOr(const TFlags& flags, const TMask& mask)
{
    //NW_STATIC_ASSERT(!ut::IsPointer(TValue));
    return (flags & static_cast<TFlags>(mask)) != 0;
}

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のビット値を真にします。
//!
//! @param        flags   現在のビットフィールドです。
//! @param        value   真にしたいビット値です。
//! @return       設定後のビットフィールドを返します。
//---------------------------------------------------------------------------
template<typename TFlags, typename TValue>
NW_INLINE TFlags
EnableFlag(const TFlags& flags, const TValue& value)
{
    return flags | static_cast<TFlags>(value);
}

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のビット値を偽にします。
//!
//! @param        flags   現在のビットフィールドです。
//! @param        value   偽にしたいビット値です。
//! @return       設定後のビットフィールドを返します。
//---------------------------------------------------------------------------
template<typename TFlags, typename TValue>
NW_INLINE TFlags
DisableFlag(const TFlags& flags, const TValue& value)
{
    return flags & ~(static_cast<TFlags>(value));
}

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のマスクのビット値を取得します。
//!
//! @param[in]    flags   現在のビットフィールドです。
//! @param[in]    shift   取得に用いるビットシフトです。
//! @param[in]    mask    取得に用いるマスクです。
//!
//! @return 指定のマスクのビット値を返します。
//---------------------------------------------------------------------------
template<typename TResult, typename TFlags, typename TMask>
NW_INLINE TResult
GetFlagValue(const TFlags& flags, int shift, const TMask& mask)
{
    return static_cast<TResult>((flags & static_cast<TFlags>(mask)) >> shift);
}

//---------------------------------------------------------------------------
//! @brief        ビットフィールドの指定のマスクのビット値を設定します。
//!
//! @param[in]    flags   現在のビットフィールドです。
//! @param[in]    shift   設定に用いるビットシフトです。
//! @param[in]    mask    設定に用いるマスクです。
//! @param[in]    value   設定したいビット値です。
//!
//! @return ビット値をセットしたフラグの値を返します。
//---------------------------------------------------------------------------
template<typename TFlags, typename TMask, typename TValue>
NW_INLINE TFlags
SetFlagValue(const TFlags& flags, int shift, const TMask& mask, const TValue& value)
{
    return (flags & ~(static_cast<TFlags>(mask))) | ((value << shift) & static_cast<TFlags>(mask));
}

//@}

} /* namespace ut */
} /* namespace nw */

#endif //  NW_UT_FLAG_H_
