﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_BINARYFILEFORMAT_H_
#define NW_UT_BINARYFILEFORMAT_H_

#include <nw/config.h>
#include <nw/types.h>
#include <nw/ut/ut_BinaryReader.h>

/* ------------------------------------------------------------------------
        マクロ定義
   ------------------------------------------------------------------------ */

//---- シグネチャ作成マクロ

#if ( NW_PLATFORM_ENDIAN == NW_ENDIAN_LITTLE )

 //---- little endian
 #define NW_UT_MAKE_SIGWORD(a,b,c,d)        \
    static_cast<nw::ut::SigWord >(          \
          (static_cast<u8>(a) <<  0)        \
        | (static_cast<u8>(b) <<  8)        \
        | (static_cast<u8>(c) << 16)        \
        | (static_cast<u8>(d) << 24)        \
    )

#else

 //---- big endian
 #define NW_UT_MAKE_SIGWORD(a,b,c,d)        \
    static_cast<nw::ut::SigWord >(          \
          (static_cast<u8>(a) << 24)        \
        | (static_cast<u8>(b) << 16)        \
        | (static_cast<u8>(c) <<  8)        \
        | (static_cast<u8>(d) <<  0)        \
    )
#endif

#define NW_UT_MAKE_VERSION_2(major, minor)                              \
    static_cast<u32>(                                                   \
        (static_cast<u8>(major) << 24) | (static_cast<u8>(minor) << 16) \
    )

#define NW_UT_MAKE_VERSION(major, minor, micro, binaryBugFix)            \
    static_cast<u32>(                                                    \
        (static_cast<u8>(major) << 24) | (static_cast<u8>(minor) << 16)  \
      | (static_cast<u8>(micro) <<  8) | (static_cast<u8>(binaryBugFix)) \
    )

#define NW_UT_VERSION_MAJOR(version)        (((version) >> 24) & 0xff)
#define NW_UT_VERSION_MINOR(version)        (((version) >> 16) & 0xff)
#define NW_UT_VERSION_MICRO(version)        (((version) >>  8) & 0xff)
#define NW_UT_VERSION_BINARYBUGFIX(version) (((version) >>  0) & 0xff)

namespace nw {
namespace ut {


/* ------------------------------------------------------------------------
        型定義
   ------------------------------------------------------------------------ */

//---- シグネチャ型
typedef u32 SigWord;        //!< @briefprivate


//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare 標準バイナリファイルヘッダ
//---------------------------------------------------------------------------
struct BinaryFileHeader
{
    SigWord signature;                  // ファイルシグネチャ   : 4 Byte
    ResU16  byteOrder;                  // バイトオーダーマーク : 2 Byte
    ResU16  headerSize;                 // ヘッダサイズ         : 2 Byte
    ResU32  version;                    // ファイルバージョン   : 4 Byte
    ResU32  fileSize;                   // ファイルサイズ       : 4 Byte
    ResU16  dataBlocks;                 // ブロック数           : 2 Byte
    ResU16  reserved;                   // 予約                 : 2 Byte
};

//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare 標準バイナリブロックヘッダ
//---------------------------------------------------------------------------
struct BinaryBlockHeader
{
    SigWord kind;                       // ブロック種別名
    ResU32  size;                       // ブロック全体サイズ
};


//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare バイナリリビジョンブロックヘッダ
//---------------------------------------------------------------------------
struct BinaryRevisionHeader
{
    SigWord signature;                  // ブロックシグニチャ : 4 Byte
    ResU32  revision;                   // リビジョン         : 4 Byte
};


//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare 標準バイナリファイルヘッダ
//---------------------------------------------------------------------------
struct RawEndianBinaryFileHeader
{
    SigWord signature;                  // ファイルシグネチャ   : 4 Byte
    u16     byteOrder;                  // バイトオーダーマーク : 2 Byte
    u16     headerSize;                 // ヘッダサイズ         : 2 Byte
    u32     version;                    // ファイルバージョン   : 4 Byte
    u32     fileSize;                   // ファイルサイズ       : 4 Byte
    u16     dataBlocks;                 // ブロック数           : 2 Byte
    u16     reserved;                   // 予約                 : 2 Byte
};

//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare 標準バイナリブロックヘッダ
//---------------------------------------------------------------------------
struct RawEndianBinaryBlockHeader
{
    SigWord kind;                       // ブロック種別名
    u32     size;                       // ブロック全体サイズ
};



//---------------------------------------------------------------------------
//! @briefprivate
//! @details       NintendoWare バイナリリビジョンブロックヘッダ
//---------------------------------------------------------------------------
struct RawEndianBinaryRevisionHeader
{
    SigWord signature;                  // ブロックシグニチャ : 4 Byte
    u32     revision;                   // リビジョン         : 4 Byte
};


/* ------------------------------------------------------------------------
        定数定義
   ------------------------------------------------------------------------ */

namespace
{
    const u16 BYTE_ORDER_MARK = 0xFEFF;
}

//----------------------------------------
//! @name バイナリファイル関連
//@{

/* ------------------------------------------------------------------------
        関数プロトタイプ
   ------------------------------------------------------------------------ */

//---------------------------------------------------------------------------
//! @brief        NintendoWare 標準バイナリファイルヘッダの正当性を
//!               チェックします。
//!
//! @param[in]    pHeader     チェック対象のファイルヘッダへのポインタ
//! @param[in]    signature   格納されているべきシグネチャ
//! @param[in]    version     格納されているべきバージョン
//! @param[in]    minBlocks   格納されているべき最小バイナリブロック数
//!
//! @return       チェックを全てパスすれば true, それ以外なら false を
//!               返します。
//---------------------------------------------------------------------------
bool IsValidBinaryFile(
                const BinaryFileHeader* pHeader,
                u32                     signature,
                u32                     version,
                u16                     minBlocks   = 1 );

//---------------------------------------------------------------------------
//! @brief        エンディアンが反転している NintendoWare 標準バイナリファイルヘッダの
//!               正当性をチェックします。
//!
//! @param[in]    pHeader     チェック対象のファイルヘッダへのポインタ
//! @param[in]    signature   格納されているべきシグネチャ
//! @param[in]    version     格納されているべきバージョン
//! @param[in]    minBlocks   格納されているべき最小バイナリブロック数
//!
//! @return       チェックを全てパスすれば true, それ以外なら false を
//!               返します。
//---------------------------------------------------------------------------
bool IsValidBinaryFile(
                       const RawEndianBinaryFileHeader* pHeader,
                       u32                     signature,
                       u32                     version,
                       u16                     minBlocks   = 1 );

//---------------------------------------------------------------------------
//! @brief        NintendoWare 標準バイナリファイルのエンディアンが逆かどうかを
//!               チェックします。
//!
//! @param[in]    pFileHeader   対象のファイルヘッダへのポインタ
//!
//! @return       エンディアンが逆なら true,さもなければ false を返します。
//---------------------------------------------------------------------------
bool IsReverseEndianBinaryFile( const BinaryFileHeader* pFileHeader );

//---------------------------------------------------------------------------
//! @brief        NintendoWare 標準バイナリファイル内の次のブロックヘッダを
//!               取得します。
//!
//! @param[out]   pFileHeader   対象のファイルヘッダへのポインタ
//! @param[out]   pBlockHeader  現在のブロックヘッダへのポインタ
//!
//! @return       pBlockHeader の次のブロックヘッダを返します。
//!               次のブロックがない場合は、NULL を返します。
//!               pBlockHeader が NULL の場合は、一番始めのブロックヘッダを返します。
//---------------------------------------------------------------------------
BinaryBlockHeader* GetNextBinaryBlockHeader(
    BinaryFileHeader*       pFileHeader,
    BinaryBlockHeader*      pBlockHeader
);

//---------------------------------------------------------------------------
//! @brief        NintendoWare 標準バイナリファイル内の次のブロックヘッダを
//!               取得します。
//!
//! @param[in]    pFileHeader     対象のファイルヘッダへのポインタ
//! @param[in]    pBlockHeader    現在のブロックヘッダへのポインタ
//!
//! @return       blockHeaderの次のブロックヘッダを返します。
//!               次のブロックがない場合は、NULLを返します。
//!               blockHeaderがNULLの場合は、一番始めのブロックヘッダを返します。
//---------------------------------------------------------------------------
NW_INLINE const BinaryBlockHeader*
GetNextBinaryBlockHeader(
    const BinaryFileHeader*     pFileHeader,
    const BinaryBlockHeader*    pBlockHeader
)
{
    return static_cast<const BinaryBlockHeader*>(
                GetNextBinaryBlockHeader(
                    const_cast<BinaryFileHeader*>(pFileHeader),
                    const_cast<BinaryBlockHeader*>(pBlockHeader)
                )
            );
}

//@}

namespace internal {

//---------------------------------------------------------------------------
//! @brief        バイナリリソースのリビジョンチェックをおこないます。
//!
//! @param[in]    resRevision バイナリリソースのリビジョンです。
//! @param[in]    libRevision ライブラリのリビジョンです。
//!
//! @return       実行可能なバイナリバージョンであれば true、
//!               そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_INLINE bool
CheckRevision( u32 resRevision, u32 libRevision )
{
    // メジャーバージョンが異なる場合は、実行不能。
    if ( NW_UT_VERSION_MAJOR(libRevision) != NW_UT_VERSION_MAJOR(resRevision) )
    {
        return false;
    }

    // リソースのマイナーバージョンがライブラリよりも新しい場合には、実行不能。
    if ( NW_UT_VERSION_MINOR(libRevision) < NW_UT_VERSION_MINOR(resRevision) )
    {
        return false;
    }

    // リソースのバイナリバグフィックスバージョンがライブラリよりも古い場合には、実行不能。
    if ( NW_UT_VERSION_BINARYBUGFIX(libRevision) > NW_UT_VERSION_BINARYBUGFIX(resRevision) )
    {
        return false;
    }

    return true;
}

} /* namespace internal */


} /* namespace ut */
} /* namespace nw */
#endif //  NW_UT_BINARYFILEFORMAT_H_
