﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_ARRAY_H_
#define NW_UT_ARRAY_H_

#include <nw/ut/ut_Iterator.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        配列を安全に使用するラッパークラスです。
//---------------------------------------------------------------------------
template <typename T>
class Array
{
public:
    typedef T                               value_type;
    typedef value_type*                     pointer;
    typedef const value_type*               const_pointer;
    typedef value_type&                     reference;
    typedef const value_type&               const_reference;
    typedef std::random_access_iterator_tag iterator_category;

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ Array() : m_Size( 0 ) , m_Array( NULL ) {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタで配列のサイズとポインタを指定します。
    //!
    //! @param[in]   size       要素の数です。
    //! @param[in]   bufferptr  設定するバッファです。
    //---------------------------------------------------------------------------
    Array( int size, T* bufferptr )
    {
        if ( size > 0 && bufferptr != NULL )
        {
            m_Size = size;
            m_Array = bufferptr;
        }
        else
        {
            NW_ERR("illegal param: size[%d] bufferptr[0x%x]", size, bufferptr);
            size = 0;
            bufferptr = NULL;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~Array() { m_Size = 0; m_Array = NULL; }

    //===========================================================================
    // イテレータ
    //===========================================================================
    class iterator;
    class const_iterator;

    class iterator : public internal::iterator<iterator_category, value_type>
    {
    public:

        typedef iterator TIt;
        typedef internal::iterator<iterator_category, value_type> TBaseIt;

        /* ctor */ explicit iterator() : m_Index( 0 ) , m_Array( NULL ) {}

        /* ctor */ explicit iterator( Array* array ) : m_Index( 0 ) , m_Array( array ) {}

        /* ctor */ iterator( Array* array, int index ) : m_Index( index ) , m_Array( array ) {}

        TIt& operator ++ () { m_Index++; return *this; }
        TIt& operator -- () { m_Index--; return *this; }

        friend bool operator ==(const TIt& x, const TIt& y)
        {
            return x.m_Index == y.m_Index && x.m_Array == y.m_Array;
        }

        friend bool operator !=(const TIt& x, const TIt& y)
        {
            return x.m_Array != y.m_Array || x.m_Index != y.m_Index;
        }

        reference operator*() { return m_Array->m_Array[m_Index]; }

        pointer operator->() { return &m_Array->m_Array[m_Index]; }

        //! @brief 要素のインデックスを取得します。
        int GetIndex() const { return m_Index; }

    private:

        int             m_Index;
        Array<T>*  m_Array;

        friend class const_iterator;  // m_Index, m_Array を使用するため。
    };

    //! @briefprivate
    typedef iterator Iterator_alias_;  // const_iterator内部で使用。

    class const_iterator
        : public internal::iterator<iterator_category, value_type>
    {
    public:

        typedef const_iterator TIt;
        typedef internal::iterator<iterator_category, value_type> TBaseIt;
        typedef const_pointer pointer;
        typedef const_reference reference;

        /* ctor */ explicit const_iterator() : m_Index( 0 ) , m_Array( NULL ) {}

        /* ctor */ explicit const_iterator( const Array* array ) : m_Index( 0 ) , m_Array( array ) {}

        /* ctor */ const_iterator( const Array* array, int index ) : m_Index( index ) , m_Array( array ) {}

        /* ctor */ explicit const_iterator( Iterator_alias_ it ) : m_Index( it.m_Index ), m_Array(it.m_Array) {}

        TIt& operator ++ () { m_Index++; return *this; }
        TIt& operator -- () { m_Index--; return *this; }

        friend bool operator ==(const TIt& x, const TIt& y)
        {
            return x.m_Index == y.m_Index && x.m_Array == y.m_Array;
        }

        friend bool operator !=(const TIt& x, const TIt& y)
        {
            return x.m_Array != y.m_Array || x.m_Index != y.m_Index;
        }

        reference operator*() { return m_Array->m_Array[m_Index]; }

        pointer operator->() { return &m_Array->m_Array[m_Index]; }

        //! @brief 要素のインデックスを取得します。
        int GetIndex() const { return m_Index; }

    private:
        int                     m_Index;
        const Array<T>*    m_Array;
    };

    //! @brief インデクサです。
    //!
    //! @param[in] index 取得したいconst要素のインデックスです。
    //! @return インデックスで指定したconst要素を返します。
    const_reference operator[] ( int index ) const
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Size)) { return m_Array[ index ]; }
        else { NW_ERR("index exceeded [%d/%d]", index, m_Size); return m_Array[ 0 ]; }
    }

    //! @brief インデクサです。
    //!
    //! @param[in] index 取得したい要素のインデックスです。
    //! @return インデックスで指定した要素を返します。
    reference operator[] ( int index )
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Size)) { return m_Array[ index ]; }
        else { NW_ERR("index exceeded [%d/%d]", index, m_Size); return m_Array[ 0 ]; }
    }

    //--------------------------------------------------------------------------
    //! @brief        先頭の要素へのイテレータを取得します。
    //!
    //! @return       先頭の要素へのイテレータを返します。
    //---------------------------------------------------------------------------
    iterator begin() { return iterator( this, 0 ); }
    const_iterator begin() const { return const_iterator( this, 0 ); }

    //------------------------------------------------------- -------------------
    //! @brief        イテレータの終端を取得します。
    //!
    //! @return       イテレータの終端を返します。
    //---------------------------------------------------------------------------
    iterator end() { return iterator( this, m_Size ); }
    const_iterator end() const { return const_iterator( this, m_Size ); }

    //! @brief 先頭の要素を取得します。
    //!
    //! @return 先頭の要素の参照を返します。
    reference front()
    {
        // m_Bufferのサイズは必ず1以上なので範囲チェックの必要はない
        return m_Array[ 0 ];
    }

    //! @brief 先頭の要素をconstとして取得します。
    //!
    //! @return 先頭の要素の参照を返します。
    const_reference front() const
    {
        // m_Bufferのサイズは必ず1以上なので範囲チェックの必要はない
        return m_Array[ 0 ];
    }

    //! @brief 最後尾の要素を取得します。
    //!
    //! @return 最後尾の要素の参照を返します。
    reference back()
    {
        // m_Bufferのサイズは必ず1以上なので範囲チェックの必要はない
        return m_Array[ m_Size - 1 ];
    }

    //! @brief 最後尾の要素をconstとして取得します。
    //!
    //! @return 最後尾の要素の参照を返します。
    const_reference back() const
    {
        // m_Bufferのサイズは必ず1以上なので範囲チェックの必要はない
        return m_Array[ m_Size - 1 ];
    }

    //! @brief 配列のサイズを取得します。
    //!
    //! @return 配列のサイズを返します。
    int size() const { return m_Size; }

    //------------------------------------------------------- -------------------
    //! @brief          使用する配列を設定します。
    //!                 bufferptrには、size以上の長さのTの配列を与えてください。
    //!
    //! @param[in]   size       要素の数です。
    //! @param[in]   arrayPtr   設定する配列です。
    //---------------------------------------------------------------------------
    void SetArray( int size, T* arrayPtr )
    {
        if ( size > 0 )
        {
            if ( arrayPtr )
            {
                m_Size = size;
                m_Array = arrayPtr;
            }
            else
            {
                NW_ERR( "arrayPtr is null" );
            }
        }
        else
        {
            NW_ERR( "size[%d] must be larger than zero", size );
        }
    }

    //--------------------------------------------------------------------------
    //! @brief        配列のバッファが設定されているか判定します。
    //!
    //! @return       設定されている場合は、true を返します。
    //---------------------------------------------------------------------------
    bool IsReady() const { return m_Array != NULL; }

    //--------------------------------------------------------------------------
    //! @brief          引数で指定したインデックスの要素へのポインタを返します。
    //!
    //! @param[in]      index   アクセスする要素のインデックスです。
    //!
    //! @return         要素へのポインタを返します。
    //---------------------------------------------------------------------------
    pointer Get( int index )
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Size))
        {
            return & m_Array[ index ];
        }
        else
        {
            NW_ERR("index exceeded [%d/%d]", index, m_Size);
            return NULL;
        }
    }

    //--------------------------------------------------------------------------
    //! @brief          引数で指定したインデックスの要素へのconstポインタを返します。
    //!
    //! @param[in]      index   アクセスする要素のインデックスです。
    //!
    //! @return         要素へのポインタを返します。
    //---------------------------------------------------------------------------
    const_pointer Get( int index ) const
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Size))
        {
            return & m_Array[ index ];
        }
        else
        {
            NW_ERR("index exceeded [%d/%d]", index, m_Size);
            return NULL;
        }
    }

    //--------------------------------------------------------------------------
    //! @brief          引数で指定したインデックスの要素へのポインタを返します。
    //!                 インデックスの範囲チェックを行いません。
    //!
    //! @param[in]      index   アクセスする要素のインデックスです。
    //!
    //! @return         要素へのポインタを返します。
    //---------------------------------------------------------------------------
    pointer UnsafeGet( int index )
    {
        NW_ASSERTMSG(static_cast<u32>(index) < static_cast<u32>(m_Size), "index exceeded [%d/%d]", index, m_Size);
        return & m_Array[ index ];
    }

    //--------------------------------------------------------------------------
    //! @brief          引数で指定したインデックスの要素へのconstポインタを返します。
    //!                 インデックスの範囲チェックを行いません。
    //!
    //! @param[in]      index   アクセスする要素のインデックスです。
    //!
    //! @return         要素へのポインタを返します。
    //---------------------------------------------------------------------------
    const_pointer UnsafeGet( int index ) const
    {
        NW_ASSERTMSG(static_cast<u32>(index) < static_cast<u32>(m_Size), "index exceeded [%d/%d]", index, m_Size);
        return & m_Array[ index ];
    }

    //! @brief 配列のサイズを取得します。
    //!
    //! @return 配列のサイズを返します。
    int GetSize() const { return m_Size; }

    //! @brief 配列のポインタを返します。
    //!
    //! @return 配列のポインタを返します。
    pointer GetArrayPtr() { return m_Array; }

    //! @brief 配列のconstポインタを返します。
    //!
    //! @return 配列のconstポインタを返します。
    const_pointer GetArrayPtr() const { return m_Array; }

    //! @brief バッファのバイト数でのサイズを返します。
    //!
    //! @return バッファのバイト数でのサイズを返します。
    u32 GetByteSize() const { return m_Size * sizeof(T); }

    //--------------------------------------------------------------------------
    //! @brief          指定したインデックスが範囲内か判定します。
    //!
    //! @param[in]      index   要素のインデックスです。
    //!
    //! @return         範囲内の場合は、true を返します。
    //---------------------------------------------------------------------------
    bool IsRangeValid( int index ) const
    {
        return ( static_cast<u32>(index) < static_cast<u32>(m_Size) );
    }

    //--------------------------------------------------------------------------
    //! @brief          引数で指定した値を全ての要素に設定します。
    //!
    //! @param[in]      value   設定する値です。
    //---------------------------------------------------------------------------
    void Fill ( const T& value )
    {
        int size = m_Size;
        for ( int i = 0; i < size; ++i )
        {
            m_Array[i] = value;
        }
    }

private:
    int     m_Size;
    T*      m_Array;
};

} // namespace ut
} // namespace nw

#endif //  NW_UT_ARRAY_H_
