﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_PRINT_H_
#define NW_UT_PRINT_H_

#include <nw/types.h>
#include <cstdarg>

namespace nw {
namespace ut {

//----------------------------------------
//! @name デバッグ出力関連
//@{

//---------------------------------------------------------------------------
//! @brief        文字列を出力します。
//!
//! @param[in]    str     nul 終端の文字列です。
//---------------------------------------------------------------------------
void PutString(const char* str);

//---------------------------------------------------------------------------
//! @brief        書式付き文字列を出力します。
//!
//! @param[in]    fmt    出力文字列フォーマット。
//---------------------------------------------------------------------------
void Printf(const char *fmt, ...);

//---------------------------------------------------------------------------
//! @brief        書式付き文字列を出力します。Printf の省メモリ版です。
//!
//! @param[in]    fmt    出力文字列フォーマット。浮動小数点系のものは指定できません。
//---------------------------------------------------------------------------
void TPrintf(const char *fmt, ...);

//---------------------------------------------------------------------------
//! @brief        引数リスト指定で書式付き文字列を出力します。
//!
//! @param[in]    fmt     出力文字列フォーマット。
//! @param[in]    vlist   可変長引数リスト
//---------------------------------------------------------------------------
void VPrintf(const char *fmt, std::va_list vlist);

//---------------------------------------------------------------------------
//! @brief        引数リスト指定で書式付き文字列を出力します。VPrintf の省メモリ版です。
//!
//! @param[in]    fmt    出力文字列フォーマット。浮動小数点系のものは指定できません。
//! @param[in]    vlist   可変長引数リスト
//---------------------------------------------------------------------------
void TVPrintf(const char *fmt, std::va_list vlist);

//---------------------------------------------------------------------------
//! @brief        ワーニングを出力します。
//!
//! @param[in]    fileName  Warning の発生したソースファイル名
//! @param[in]    line      Warning の発生した行番号
//! @param[in]    fmt       出力文字列フォーマット
//---------------------------------------------------------------------------
void Warning( const char* fileName, int line, const char* fmt, ...);

//---------------------------------------------------------------------------
//! @brief        ワーニングを出力します。Warning の省メモリ版です。
//!
//! @param[in]    fileName  Warning の発生したソースファイル名
//! @param[in]    line      Warning の発生した行番号
//! @param[in]    fmt       出力文字列フォーマット。浮動小数点系のものは指定できません。
//---------------------------------------------------------------------------
void TWarning( const char* fileName, int line, const char* fmt, ...);

//---------------------------------------------------------------------------
//! @brief        メッセージを出力して停止します。
//!
//! @param[in]    fileName  Panic の発生したソースファイル名
//! @param[in]    line      Panic の発生した行番号
//! @param[in]    fmt       出力文字列フォーマット。
//---------------------------------------------------------------------------
void Panic( const char* fileName, int line, const char* fmt, ...);

//---------------------------------------------------------------------------
//! @brief        メッセージを出力して停止します。Panic の省メモリ版です。
//!
//! @param[in]    fileName  Panic の発生したソースファイル名
//! @param[in]    line      Panic の発生した行番号
//! @param[in]    fmt       出力文字列フォーマット。浮動小数点系のものは指定できません。
//---------------------------------------------------------------------------
void TPanic( const char* fileName, int line, const char* fmt, ...);

//---------------------------------------------------------------------------
//! @brief        メッセージ付きでプログラムを異常終了します。
//!
//! @param[in]    codePos ファイル名
//! @param[in]    lineNo  行番号
//! @param[in]    format  出力文字列のフォーマット
//---------------------------------------------------------------------------
void HaltWithDetail(const char* codePos, int lineNo, const char* format, ...);

//---------------------------------------------------------------------------
//! @brief         プログラムを異常終了します。
//---------------------------------------------------------------------------
void  Halt();

//@}


} // namespace ut
} // namespace nw

#endif // NW_UT_PRINT_H_
