﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_CRITICALSECTION_H_
#define NW_UT_CRITICALSECTION_H_

#include <nw/config.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_ScopedLock.h>
#include <nw/ut/os/ut_Mutex.h>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/os.h>
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief    クリティカルセクション用の実装です。内部では Mutex を使用しています。
//---------------------------------------------------------------------------
class CriticalSection
{
private:
    static const u32 CRITICAL_SECTION_SIZE = 24;
    static const u32 CRITICAL_SECTION_ALIGN = 4;

public:
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //---------------------------------------------------------------------------
    CriticalSection()
    {
        m_Mutex.Initialize();
    }

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //---------------------------------------------------------------------------
    ~CriticalSection()
    {
        m_Mutex.Finalize();
    }

    void Enter()
    {
        m_Mutex.Lock();
    }

    bool TryEnter()
    {
        return m_Mutex.TryLock();
    }

    void Leave()
    {
        m_Mutex.Unlock();
    }

    void Lock()
    {
        this->Enter();
    }

    bool TryLock()
    {
        return this->TryEnter();
    }

    void Unlock()
    {
        this->Leave();
    }

private:
    ut::Mutex m_Mutex;
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details ScriticalSection 版のスコープドロックの特殊化版
//---------------------------------------------------------------------------
template <>
class ScopedLock< CriticalSection >
{
private:
    NW_DISALLOW_COPY_AND_ASSIGN(ScopedLock);

public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタでロックを行います。
    //!
    //! @param[out]   lockObj 使用するロックオブジェクトです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit ScopedLock(CriticalSection& lockObj) : m_LockObj(lockObj)
    {
        m_LockObj.Enter();
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタでアンロックをおこないます。
    //---------------------------------------------------------------------------
    ~ScopedLock()
    {
        m_LockObj.Leave();
    }

private:
    CriticalSection& m_LockObj;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_CRITICALSECTION_H_

