﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_SPY_DATAFILECHANNEL_H_
#define NW_SND_SPY_SPY_DATAFILECHANNEL_H_

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

#if defined(NW_PLATFORM_CAFE)
#include <cafe/fs.h>
#endif

#include <nw/snd/spy/fnd/os/sndspyfnd_CriticalSection.h>

namespace nw {
namespace snd {
namespace spy {

//! @brief SoundSpyとのHostFileIOによる通信ためのパラメータです。
//!
//! @details
//! HostFileIOを使うことでSoundSpyとの通信がより高速になり、
//! 最終波形出力などの大きなデータの送信に余裕ができます。
//!
//! SoundSpyから通知される作業用ディレクトリに通信用データファイルが２つ作成されます。
//! 通信用データファイルのサイズは上限値を超える場合があります。
//!
struct SpyDataFileChannelParam
{
#if defined(NW_PLATFORM_CAFE)
    //! 通信用ファイルの操作に使う FSClient を指定します(必須)。
    FSClient* fsClient;

    //! 通信用ファイルのサイズの上限を指定します。デフォルトは32MBです。
    u32 dataFileSizeLimit;

    //! HostFileIO のマウントパスを指定します(必須)。
    //! 指定した文字列は SpyController::Open() の中でコピーされるので、呼び出し後に破棄できます。
    const char* hfioMountPath;
#endif

    SpyDataFileChannelParam();
};

namespace internal {

//! @brief SoundSpyへのデータ通信をHostFileIOで行います。
class SpyDataFileChannel
{
public: // 型

private: // 型

public: // コンストラクタ
    SpyDataFileChannel();

public: // メソッド
    //! @brief Initialize() に必要なメモリ量を取得します。
    static size_t GetRequiredMemorySize();

    //! @brief SpyController::Initialize() から呼ばれます。
    void Initialize(void* buffer);

    //! @brief SpyController::Finalize() から呼ばれます。
    void Finalize();

    //! @brief SpyController::Open() から呼ばれます。
    bool Open(const SpyDataFileChannelParam& param);

    //! @brief SpyController::Close() から呼ばれます。
    void Close();

    //! @brief Open状態かを取得します。
    bool IsOpen() const;

    //! @brief SoundSpy から SetOutputDirPacket を受信したときに呼び出されます。
    bool BeginSession(const char* outputDir);

    //! @brief SoundSpy との通信が切断されたときに呼び出されます。
    void EndSession();

    //! @brief セッション中かを取得します。
    bool IsActive() const;

    //! @brief SoundSpy から NotifyDataReadPacket を受信したときに呼び出されます。
    bool SetCurrentReadFile(u32 fileNo);

    //! @brief SoundSpy にデータを送信します。
    //!
    //! buffer は FS_IO_BUFFER_ALIGN でアライメントしている必要があります。
    //! buffer は破壊的に利用されます。
    //! buffer は最低でも sizeof(PingPacket) の大きさがなければなりません。
    //!
    bool WriteData(void* buffer, u32 length);

private: // メソッド
    bool SetOutputDirPath(const char* outputDir);
    bool OpenFiles();
    void CloseFiles();

private: // メンバ変数
#if defined(NW_PLATFORM_CAFE)
    nw::snd::spy::internal::fnd::CriticalSection m_DataFileLock;
    FSClient* m_FsClient;
    FSCmdBlock* m_FsCmdBlock;
    FSFileHandle m_DataOutputFile[2];
    u32 m_MaxWriteLength;
    u32 m_WriteLength;
    u8 m_CurrentWriteFile;
    u8 m_CurrentReadFile;
    char m_HfioMountPath[FS_MAX_MOUNTPATH_SIZE];
    char m_OutputDirMountPath[FS_MAX_ARGPATH_SIZE + FS_MAX_ARGPATH_SIZE_PAD];
    bool m_DataOutputFileOpen;
#endif
};

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_SPY_DATAFILECHANNEL_H_
