﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_SPY_CONTROLLER_H_
#define NW_SND_SPY_SPY_CONTROLLER_H_

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_SpySession.h>
#include <nw/snd/spy/sndspy_SpyDataFileChannel.h>
#include <nw/snd/spy/sndspy_SpyDataInfo.h>
#include <nw/snd/spy/modules/sndspy_TimeModule.h>
#include <nw/snd/spy/modules/sndspy_LogModule.h>
#include <nw/snd/spy/modules/sndspy_PlotModule.h>
#include <nw/snd/spy/modules/sndspy_MarkerModule.h>
#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>
#include <nw/snd/spy/fnd/os/sndspyfnd_CriticalSection.h>
#include <nw/snd/spy/fnd/os/sndspyfnd_Event.h>
#include <nw/snd/spy/fnd/os/sndspyfnd_Thread.h>

namespace nw {
namespace snd {
namespace spy {

namespace internal {
struct DataPacket;
} // nw::snd::spy::internal

class SpyModule;

//! @briefprivate
//!
//! @brief Spy セッションと各モジュールを制御します。
//!        TODO : スレッド管理を上位レイヤーに切り出すことを検討します。
class SpyController
{
private: // 定数
    static const u32 MAX_CONNECTION_CHANGED_CALLBACK_COUNT = 16;

public: // 型
    //! @brief ポートの型です。
    typedef nw::snd::spy::internal::fnd::HioChannel::PortType      PortType;
    typedef nw::snd::spy::internal::fnd::HioChannel::ConstPortType ConstPortType;

    //! @brief 接続状態変更コールバック関数です。
    //! @param[in]  param  任意のパラメータを指定します。
    typedef void (*ConnectionChangedCallback)(void* param);

    //! @briefprivate
    //!
    //! @brief Open() のパラメータです。
    struct OpenArg
    {
        ConstPortType syncPort; //!< SYNC ポートです。
        ConstPortType dataPort; //!< DATA ポートです。
        u32 syncThreadPriority; //!< SYNC 通信スレッドのプライオリティです。
        u32 dataThreadPriority; //!< DATA 通信スレッドのプライオリティです。

        SpyDataFileChannelParam dataFileChannelParam; //!< HostFileIOを使ったデータ通信のためのパラメータです。

        OpenArg();
    };

private: // 型
    //! @briefprivate
    //!
    //! @brief コントローラの状態です。
    enum State
    {
        STATE_NOT_INITIALIZED,
        STATE_INITIALIZED,
        STATE_OPENING,
        STATE_DISCONNECTING,
        STATE_CONNECTING,
        STATE_PREPARING,
        STATE_PREPARED
    };

    //! @briefprivate
    struct CallbackInfo
    {
        ConnectionChangedCallback callback;
        void*                     param;
    };

    //! @briefprivate
    struct DataBuffer
    {
        void* address;
        u32   length;
        u32   currentPosition;

        DataBuffer() : address(NULL), length(0), currentPosition(0) { }
    };

    // Spy セッションのメッセージハンドラアダプタです。
    //! @briefprivate
    class SessionMessageHandlerAdaptor : public internal::SpySession::IMessageHandler
    {
    public:
        void Initialize(SpyController* owner)
        {
            m_Owner = owner;
        }

        virtual void OnInitializeSession()
        {
            NW_ASSERT_NOT_NULL(m_Owner);
        }

        virtual void OnFinalizeSession()
        {
            NW_ASSERT_NOT_NULL(m_Owner);
            m_Owner->m_SpyDataFileChannel.EndSession();
        }

        virtual void OnSelectDataID(u32 numFlags, u32 flags[])
        {
            NW_ASSERT_NOT_NULL(m_Owner);
            m_Owner->SelectDataIDs(numFlags, flags);
        }

        virtual void OnSetOutputDirPath(const char* outputDir)
        {
            NW_ASSERT_NOT_NULL(m_Owner);
            m_Owner->m_SpyDataFileChannel.BeginSession(outputDir);
            m_Owner->SendSetOutputDirPathReplyPacket(m_Owner->m_SpyDataFileChannel.IsActive());
        }

        virtual void OnDataRead(u32 fileNo)
        {
            NW_ASSERT_NOT_NULL(m_Owner);
            m_Owner->m_SpyDataFileChannel.SetCurrentReadFile(fileNo);
        }

        virtual const SpyDataInfo* OnQueryDataInfo()
        {
            NW_ASSERT_NOT_NULL(m_Owner);
            return m_Owner->QueryDataInfo();
        }

    private:
        SpyController* m_Owner;
    };

private: // 定数
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    static const u32 SYNC_THREAD_STACK_SIZE;
    static const u32 DATA_THREAD_STACK_SIZE;
#endif

public: // コンストラクタ
    SpyController();

public: // メソッド
    void Initialize(void* buffer, u32 bufferLength, u32 dataBufferLength);
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief         このクラスを利用するのに必要なメモリサイズを取得します。
    //!
    //! @param[in]     dataBufferLength  データバッファサイズを指定します。
    //!                                  ダブルバッファ化されるため、ここで指定した数値の2倍以上を要求します。
    //!
    //! @return        このクラスを利用するのに必要なメモリサイズを返します。
    //---------------------------------------------------------------------------
    static size_t GetRequiredMemorySize(u32 dataBufferLength);

    bool Open(const OpenArg& arg);
    void Close();
    bool WaitForClose();

    bool IsInitialized() const { return m_State > STATE_NOT_INITIALIZED; }
    bool IsOpened() const { return m_State > STATE_OPENING; }
    bool IsClosing() const { return m_State == STATE_DISCONNECTING; }
    bool IsConnecting() const { return STATE_CONNECTING <= m_State && !IsPrepared(); }
    bool IsConnected() const { return IsPrepared(); }

    //---------------------------------------------------------------------------
    //! @brief   接続状態が変化したときに呼び出されるコールバックを登録します。
    //!
    //!          このメソッドは内部でクリティカルセクションによる排他制御を行います。
    //!          状態を変化させるメソッドと並行して呼び出す際には、デッドロックに注意してください。
    //!
    //!          また、コールバック関数内で、接続状態を変化させるようなメソッドを呼び出した場合、
    //!          コールバック処理中に、新たな状態変化を通知するコールバック呼び出しが多重に呼び出されます。
    //!
    //! @return  コールバック登録に成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool RegisterConnectionChangedCallback(ConnectionChangedCallback callback, void* userparam = NULL);

    //---------------------------------------------------------------------------
    //! @brief  接続状態が変化したときに呼び出されるコールバックの登録を解除します。
    //!
    //!         このメソッドは内部でクリティカルセクションによる排他制御を行います。
    //!         状態を変化させるメソッドと並行して呼び出す際には、デッドロックに注意してください。
    //!
    //!         また、コールバック関数内で、接続状態を変化させるようなメソッドを呼び出した場合、
    //!         コールバック処理中に、新たな状態変化を通知するコールバック呼び出しが多重に呼び出されます。
    //---------------------------------------------------------------------------
    void UnregisterConnectionChangedCallback(ConnectionChangedCallback callback);

    //---------------------------------------------------------------------------
    //! @brief         現在のアプリケーションフレームを取得します。
    //!
    //! @return        現在のアプリケーションフレームを返します。
    //!
    //! @sa SetCurrentAppFrame
    //---------------------------------------------------------------------------
    u32 GetCurrentAppFrame() const
    {
        return m_TimeModule.GetCurrentAppFrame();
    }

    //---------------------------------------------------------------------------
    //! @brief         現在のアプリケーションフレームを設定します。
    //!
    //!                Spy を利用するアプリは、アプリフレームの先頭でこの関数を呼び出し、
    //!                Spy 内部のアプリケーションフレームを更新してください。
    //!
    //! @param[in]     value  現在のアプリケーションフレーム値を指定します。
    //---------------------------------------------------------------------------
    void SetCurrentAppFrame(u32 value);

    //---------------------------------------------------------------------------
    //! @brief         現在のオーディオフレームを取得します。
    //!
    //! @return        現在のオーディオフレームを返します。
    //!
    //! @sa SetCurrentAudioFrame
    //---------------------------------------------------------------------------
    u32 GetCurrentAudioFrame() const
    {
        return m_TimeModule.GetCurrentAudioFrame();
    }

    //---------------------------------------------------------------------------
    //! @brief         現在のアプリケーションフレームを設定します。
    //!
    //!                Spy を利用するアプリは、アプリフレームの先頭でこの関数を呼び出し、
    //!                Spy 内部のアプリケーションフレームを更新してください。
    //!
    //! @param[in]     value  現在のアプリケーションフレーム値を指定します。
    //---------------------------------------------------------------------------
    void SetCurrentAudioFrame(u32 value);

    //---------------------------------------------------------------------------
    //! @brief         SpyModule をインストールします。
    //!
    //! @param[in]     module  SpyModule です。
    //!
    //! @returns       インストールに成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool InstallModule(SpyModule& module);

    //---------------------------------------------------------------------------
    //! @brief         SpyModule をアンインストールします。
    //!
    //! @param[in]     module  SpyModule です。
    //!
    //---------------------------------------------------------------------------
    void UninstallModule(SpyModule& module);

    LogModule& GetLogModule()
    {
        return m_LogModule;
    }

    MarkerModule& GetMarkerModule()
    {
        return m_MarkerModule;
    }

    PlotModule& GetPlotModule()
    {
        return m_PlotModule;
    }

    bool PushData(SpyModule& module, const void* buffer, u32 length);

    void ClearDataBuffer();

    //---------------------------------------------------------------------------
    //! @brief   現在のバッファの空き容量を取得します。
    //!
    //! @return  現在のバッファの空き容量を返します。
    //---------------------------------------------------------------------------
    u32 GetFreeCurrentDataBufferLength() const
    {
        return GetCurrentDataBuffer().length - GetCurrentDataBuffer().currentPosition;
    }

private: // メソッド
    static void SessionStateChangedCallback(void* param)
    {
        NW_ASSERT_NOT_NULL(param);
        reinterpret_cast<SpyController*>(param)->OnSessionStateChanged();
    }

    void OnSessionStateChanged();
    void OnSessionConnected();
    void OnSessionDisconnected();
    void OnSessionClosed();

    void ResetSpyFrameBase();

    bool StartSyncThread(u32 priority);
    bool StartDataThread(u32 priority);

    u32 SyncThreadMain(void* param);
    u32 DataThreadMain(void* param);

    void CloseImpl();

    bool IsPreparing() const { return m_State == STATE_PREPARING; }
    bool IsPrepared() const { return m_State >= STATE_PREPARED; }

    void SetState(State value);

    void UnregisterAllConnectionChangedCallbacks();

    void NotifyConnectionChanged();

    bool PushDataPacket(SpyDataID dataID, const void* buffer, u32 length);

    internal::DataPacket* AllocateDataPacketBuffer(u32 payloadLength);
    void SendDataPacket();
    bool SendVersionPacket();
    bool SendSetOutputDirPathReplyPacket(bool active);
    void SwapBuffer();

    const DataBuffer& GetCurrentDataBuffer() const
    {
        NW_ASSERT_NOT_NULL(m_CurrentDataBuffer);
        return *m_CurrentDataBuffer;
    }

    DataBuffer& GetCurrentDataBuffer()
    {
        NW_ASSERT_NOT_NULL(m_CurrentDataBuffer);
        return *m_CurrentDataBuffer;
    }

    const DataBuffer& GetSendDataPacketBuffer() const
    {
        NW_ASSERT_NOT_NULL(m_CurrentDataBuffer);
        return m_CurrentDataBuffer == &m_DataBuffer1 ? m_DataBuffer2 : m_DataBuffer1;
    }

    DataBuffer& GetSendDataPacketBuffer()
    {
        NW_ASSERT_NOT_NULL(m_CurrentDataBuffer);
        return m_CurrentDataBuffer == &m_DataBuffer1 ? m_DataBuffer2 : m_DataBuffer1;
    }

    void SelectDataIDs(u32 numFlags, u32 flags[]);

    void InitializeDataInfo();
    void FinalizeDataInfo();
    const SpyDataInfo* QueryDataInfo();

    void DumpAllModules();
    static const char* StateToString(State value);

private: // メンバ変数
    nw::snd::spy::internal::fnd::TimeSpan m_TimestampBase;
    volatile u64 m_LastTimestamp;

    volatile State m_State;
    SpyModule* m_ModuleTop;
    SpyModule* m_ModuleLast;
    SpyModule* m_Module;

    TimeModule m_TimeModule;
    LogModule m_LogModule;
    MarkerModule m_MarkerModule;
    PlotModule m_PlotModule;

    CallbackInfo m_CallbackInfos[MAX_CONNECTION_CHANGED_CALLBACK_COUNT];

    internal::SpySession m_Session;
    SessionMessageHandlerAdaptor m_SessionMsgHandlerAdaptor;

    nw::snd::spy::internal::fnd::CriticalSection m_StateLock;
    nw::snd::spy::internal::fnd::CriticalSection m_DataBufferLock;

    DataBuffer* m_CurrentDataBuffer;
    DataBuffer  m_DataBuffer1;
    DataBuffer  m_DataBuffer2;

    nw::snd::spy::internal::fnd::Event m_PushDataEvent;

    volatile bool m_IsSyncThreadEnabled;
    volatile bool m_IsDataThreadEnabled;
    volatile bool m_IsDataSelected; //!< 1つでもデータが要求されている。

    nw::snd::spy::internal::fnd::Thread m_SyncThread;
    nw::snd::spy::internal::fnd::Thread m_DataThread;

    nw::snd::spy::internal::fnd::Thread::HandlerDelegate<SpyController> m_SyncThreadHandler;
    nw::snd::spy::internal::fnd::Thread::HandlerDelegate<SpyController> m_DataThreadHandler;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    void* m_SyncThreadStack;
    void* m_DataThreadStack;
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    PortType m_SyncPort;
    PortType m_DataPort;
#elif defined(NW_PLATFORM_CAFE)
    char m_SyncPort[nw::snd::spy::internal::fnd::HioChannel::MAX_PORT_LENGTH];
    char m_DataPort[nw::snd::spy::internal::fnd::HioChannel::MAX_PORT_LENGTH];
#endif

    internal::SpyDataFileChannel m_SpyDataFileChannel;
};

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_SPY_CONTROLLER_H_
