﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_PACKET_H_
#define NW_SND_SPY_PACKET_H_

#include "../sndspy_Config.h"
#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>
#include <nw/snd/spy/fnd/binary/sndspyfnd_PrimitiveTypes.h>
#include <nw/snd/spy/sndspy_Config.h>
#include <nw/snd/spy/sndspy_SpyDataID.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {

//----------------------------------------------------------
//! @brief パケットID
struct PacketID
{
    //! @brief パケットID の種別
    enum Type
    {
        // 無効値
        ID_INVALID = 0,

        // PC -> TARGET
        ID_INITIALIZE,
        ID_INITIALIZE_REPLY,
        ID_FINALIZE,
        ID_FINALIZE_REPLY,
        ID_SELECT_DATA_ID,
        ID_SELECT_DATA_ID_REPLY,
        ID_PING,
        ID_PONG,
        ID_SET_OUTPUT_DIR,
        ID_SET_OUTPUT_DIR_REPLY,
        ID_NOTIFY_DATA_READ,
        ID_NOTIFY_DATA_READ_REPLY,
        ID_QUERY_DATA_INFO,
        ID_QUERY_DATA_INFO_REPLY,

        // PC <- TARGET
        ID_DATA,
        ID_DATA_END,

        // 最大値
        ID_MAX = ID_DATA
    };

    //----------------------------------------------------------

    // enum は 1 バイトになったりするので、u32 で PacketId を表現することにする
    u32 value;

    //----------------------------------------------------------

    explicit PacketID(u32 value) : value(value) { }

    explicit PacketID(Type value) : value(value) { }

    operator u32()
    {
        return value;
    }

    operator u32() const
    {
        return value;
    }

    const char* ToString() const
    {
#if defined(NW_DEBUG)
        static const char* PACKET_ID_STRING[] =
        {
            "ID_INVALID",
            "ID_INITIALIZE",
            "ID_INITIALIZE_REPLY",
            "ID_FINALIZE",
            "ID_FINALIZE_REPLY",
            "ID_SELECT_DATA_ID",
            "ID_SELECT_DATA_ID_REPLY",
            "ID_PING",
            "ID_PONG",
            "ID_SET_OUTPUT_DIR",
            "ID_SET_OUTPUT_DIR_REPLY",
            "ID_NOTIFY_DATA_READ",
            "ID_NOTIFY_DATA_READ_REPLY",
            "ID_QUERY_DATA_INFO",
            "ID_QUERY_DATA_INFO_REPLY",
            "ID_DATA",
            "ID_DATA_END"
        };

        return PACKET_ID_STRING[value > ID_MAX ? ID_INVALID : value];
#else
        return NULL;
#endif
    }
};

//----------------------------------------------------------
//! @ brief パケットヘッダ
struct PacketHeader
{
    enum ResultCode
    {
        RESULT_SUCCESS = 0,
        RESULT_ERROR = -1
    };

    //----------------------------------------------------------

    u32      signature;
    PacketID id;
    u32      bodyLength;
    u32      resultCode;

    //----------------------------------------------------------

    PacketHeader()
        : signature(ProtcolConsts::Signature), id(PacketID::ID_INVALID), bodyLength(0), resultCode(RESULT_SUCCESS)
    {}

    PacketHeader(PacketID packetId, u32 bodyLength)
        : signature(ProtcolConsts::Signature), id(packetId), bodyLength(bodyLength), resultCode(RESULT_SUCCESS)
    {}

    PacketHeader(PacketID::Type packetId, u32 bodyLength)
        : signature(ProtcolConsts::Signature), id(packetId), bodyLength(bodyLength), resultCode(RESULT_SUCCESS)
    {}

    //----------------------------------------------------------

    void Dump() const
    {
#if defined(NW_DEBUG)
        NW_LOG("[%-30s](%2d) bodyLength(%4d, 0x%08x)\n", id.ToString(), u32(id), u32(bodyLength), u32(bodyLength));
#endif
    }
};

//----------------------------------------------------------
//! @ brief パケット既定クラス
struct PacketBase
{
    PacketHeader header;

    PacketBase(PacketID packetId, u32 packetLength)
        : header(packetId, packetLength - sizeof(PacketHeader))
    {}

    PacketBase(PacketID::Type packetId, u32 packetLength)
        : header(packetId, packetLength - sizeof(PacketHeader))
    {}
};

//----------------------------------------------------------
//! @ brief 初期化パケット
struct InitializePacket : public PacketBase
{
    struct Body
    {
        u32 reserved;

        //----------------------------------------------------------
        Body() :
        reserved(0)
        {
        }
    };

    Body body;

    NW_STATIC_ASSERT(sizeof(Body) == 4);

    InitializePacket()
        : PacketBase(PacketID::ID_INITIALIZE, sizeof(InitializePacket))
    {}
};

//----------------------------------------------------------
//! @ brief 初期化返信パケット
struct InitializeReplyPacket : public PacketBase
{
    InitializeReplyPacket()
        : PacketBase(PacketID::ID_INITIALIZE_REPLY, sizeof(InitializeReplyPacket))
    {}
};

//----------------------------------------------------------
//! @ brief 終了パケット
struct FinalizePacket : public PacketBase
{
    FinalizePacket()
        : PacketBase(PacketID::ID_FINALIZE, sizeof(FinalizePacket))
    {}
};

//----------------------------------------------------------
//! @ brief 終了返信パケット
struct FinalizeReplyPacket : public PacketBase
{
    FinalizeReplyPacket()
        : PacketBase(PacketID::ID_FINALIZE_REPLY, sizeof(FinalizeReplyPacket))
    {}
};

//----------------------------------------------------------
//! @ brief SpyDataID 選択パケット
struct SelectDataIDPacket : public PacketBase
{
    static const u32 MAX_NUM_FLAGS = 32;

    struct Body
    {
        //! selectionFlags メンバーの配列長です。
        u32 selectionFlagsLength;

        //! データ要求の有無を表すビットフィールドです。
        //!
        //! データID(i)の要求の有無は selectionFlags[i / 32] & (1 << (i % 32)) != 0 で判断されます。
        //!
        u32 selectionFlags[MAX_NUM_FLAGS];
    };

    NW_STATIC_ASSERT(sizeof(Body) == 132);

    Body body;

    SelectDataIDPacket()
        : PacketBase(PacketID::ID_SELECT_DATA_ID, sizeof(0)) // 可変長
    {}
};

//----------------------------------------------------------
//! @ brief SpyDataID 選択返信パケット
struct SelectDataIDReplyPacket : public PacketBase
{
    SelectDataIDReplyPacket()
        : PacketBase(PacketID::ID_SELECT_DATA_ID_REPLY, sizeof(SelectDataIDReplyPacket))
    {}
};

//----------------------------------------------------------
//! @ brief ping パケット
struct PingPacket : public PacketBase
{
    PingPacket()
        : PacketBase(PacketID::ID_PING, sizeof(PingPacket))
    {}
};

//----------------------------------------------------------
//! @ brief pong パケット
struct PongPacket : public PacketBase
{
    PongPacket()
        : PacketBase(PacketID::ID_PONG, sizeof(PongPacket))
    {}
};

//----------------------------------------------------------
//! @ brief 出力ディレクトリ指定パケット
struct SetOutputDirPacket : public PacketBase
{
    // 出力ディレクトリのパス文字列の最大長です。ヌル文字を含みます。
    static const u32 OUTPUT_DIR_PATH_SIZE = 512;

    struct Body
    {
        u32 length;
        char path[OUTPUT_DIR_PATH_SIZE];
    };

    NW_STATIC_ASSERT(sizeof(Body) == 516);

    Body body;

    SetOutputDirPacket()
        : PacketBase(PacketID::ID_SET_OUTPUT_DIR, sizeof(SetOutputDirPacket))
    {}
};

//----------------------------------------------------------
//! @ brief 出力ディレクトリ指定返信パケット
//!
//! Sync, Data両チャンネルで返信されます。
//! Dataチャンネルでの返信によりファイルによるデータ転送が有効化されます。
//!
struct SetOutputDirReplyPacket : public PacketBase
{
    SetOutputDirReplyPacket()
        : PacketBase(PacketID::ID_SET_OUTPUT_DIR_REPLY, sizeof(SetOutputDirReplyPacket))
    {}
};

//----------------------------------------------------------
//! @ brief データリード開始パケット
//!
//! SoundSpyがデータファイルの読み出しを開始したことを示します。
//! アプリのデータ書き込みがSoundSpyのデータ読み出しを追い越さないようにします。
//!
struct NotifyDataReadPacket : public PacketBase
{
    struct Body
    {
        u32 fileNo; //!< 読み込みを開始したファイル番号です(0 or 1)。
    };

    NW_STATIC_ASSERT(sizeof(Body) == 4);

    Body body;

    NotifyDataReadPacket()
        : PacketBase(PacketID::ID_NOTIFY_DATA_READ, sizeof(NotifyDataReadPacket))
    {}
};

//----------------------------------------------------------
//! @ brief データリード開始返信パケット
struct NotifyDataReadReplyPacket : public PacketBase
{
    NotifyDataReadReplyPacket()
        : PacketBase(PacketID::ID_NOTIFY_DATA_READ_REPLY, sizeof(NotifyDataReadReplyPacket))
    {}
};

//----------------------------------------------------------
//! @brief データ情報要求パケット(Sync)
//!
//! アプリが対応しているデータ情報を問い合わせます。
//!
struct QueryDataInfoPacket : public PacketBase
{
    QueryDataInfoPacket()
        : PacketBase(PacketID::ID_QUERY_DATA_INFO, sizeof(QueryDataInfoPacket))
    {}
};

//----------------------------------------------------------
//! @brief データ情報要求返信パケット(Sync)
//!
//! dataId に 0 を指定した場合には、データ情報問合せのシーケンスを終わります。
//!
struct QueryDataInfoReplyPacket : public PacketBase
{
    struct Body
    {
        u32 dataVersion; //!< データのバージョンです。
        u16 dataId; //!< セッション中にのみ有効なデータIDです。
        u16 dataNameLength; //! dataNameフィールドに格納された文字列の長さです。パディングは含みません。

        // 以下に可変長フィールドが続きます。
        // dataName[dataNameLength]; //!< データタイプを表す文字列です（ヌル終端は不要。4バイトアライメント）。
    };

    NW_STATIC_ASSERT(sizeof(Body) == 8);

    Body body;

    QueryDataInfoReplyPacket()
        : PacketBase(PacketID::ID_QUERY_DATA_INFO_REPLY, 0) // 可変長
    {}

    void SetPacketSize()
    {
        header.bodyLength = sizeof(Body) + ((body.dataNameLength + 3) & ~3);
    }
};

//----------------------------------------------------------
//! @ brief データパケット
struct DataPacket : public PacketBase
{
    struct Body
    {
        // timestamp の型に u64 を使うとバイト境界が8バイトになってしまうので２つに分割します。
        u32 timestampLo;        //!< パケットの記録時間。単位は[usec]。精度は1[msec]。usecのオーダーはフレームの区別に使います。下位4バイト。
        u32 timestampHi;        //!< パケットの記録時間。単位は[usec]。精度は1[msec]。usecのオーダーはフレームの区別に使います。上位4バイト。
        s32 dataID;             //!< データID
        u32 payloadLength;      //!< データペイロードの長さ
        u32 paddingLength;      //!< パディングの長さ

        // 以下に可変長フィールドが続く
        // payload;             //!< データペイロード

        //----------------------------------------------------------
        Body() :
        timestampLo(0),
        timestampHi(0),
        dataID(DATA_ID_INVALID),
        payloadLength(0),
        paddingLength(0)
        {
        }
    };

    NW_STATIC_ASSERT(sizeof(Body) == 20);

    Body body;

    DataPacket()
        : PacketBase(PacketID::ID_DATA, 0)  // 可変長
    {}

    static u32 GetRequiredMemorySize(u32 payloadLength)
    {
        return nw::snd::spy::internal::fnd::RoundUp(
            sizeof(DataPacket) + payloadLength,
            nw::snd::spy::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
    }

    //! @briefprivate
    //!
    //! @brief パケットの送信時間を設定します。
    //!
    //! @details
    //! SpyController が使用します。
    //!
    //! @param[in] timestamp  フレームを区別するために補正された時間情報[usec]です。
    //!
    void SetTimestamp(u64 timestamp)
    {
        body.timestampLo = static_cast<u32>(timestamp);
        body.timestampHi = static_cast<u32>(timestamp >> 32);
    }

    void SetLengths(u32 payloadLength)
    {
        body.payloadLength = payloadLength;
        body.paddingLength =
            nw::snd::spy::internal::fnd::RoundUp(payloadLength, nw::snd::spy::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT) - payloadLength;

        header.bodyLength = sizeof(Body) + payloadLength + body.paddingLength;
    }

    void* GetPayload()
    {
        return nw::snd::spy::internal::fnd::AddOffsetToPtr(this, sizeof(DataPacket));
    }
};

//----------------------------------------------------------
//! @ brief データファイル終端パケット
//!
//! 現在のファイルへのデータ書き込みが終了したことを示します。
//! 出力は次のファイルに切り替わります。
//!
struct DataEndPacket : public PacketBase
{
    DataEndPacket()
        : PacketBase(PacketID::ID_DATA_END, sizeof(DataEndPacket))
    {}
};

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_PACKET_H_
