﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_MARKER_MODULE_H_
#define NW_SND_SPY_MARKER_MODULE_H_

#include <nw/types.h>
#include <nw/snd/spy/modules/sndspy_SpyModule.h>
#include <nw/snd/spy/modules/sndspy_MarkerPacket.h>

#include <nw/snd/spy/sndspy_Config.h>

namespace nw {
namespace snd {
namespace spy {

class SpyController;
class Marker;

#ifdef NW_SND_SPY_ENABLE

//! @brief マーカーを管理する Spy モジュールです。
//!        このモジュールにアタッチすると、マーカーが有効化されます。
class MarkerModule : public SpySimpleModule
{
public:
    //! @brief コンストラクタです。
    MarkerModule();

    //! @brief 指定マーカーをアタッチします。
    //! @param[in]  marker  アタッチするマーカー
    bool AttachMarker(Marker& marker);

    //! @brief 指定マーカーをデタッチします。
    //! @param[in]  marker  デタッチするマーカー
    void DetachMarker(Marker& marker);

private:
    Marker* m_MarkerTop;
    Marker* m_MarkerLast;
};

//! @brief Spy マーカーを制御します。
class Marker
{
public:
    //! @brief コンストラクタです。
    Marker();

    //! @brief デストラクタです。
    virtual ~Marker();

    //! @brief MarkerModule へのアタッチの有無を取得します。
    bool IsAttached() const { return m_Module != NULL; }

    //! @brief マーカーに値を追加します。
    //! @param[in]  description  マーカーの説明
    //! @param[in]  r            マーカー色の r 成分
    //! @param[in]  g            マーカー色の g 成分
    //! @param[in]  b            マーカー色の b 成分
    void PushValue(const char* description, u8 r, u8 g, u8 b);

private:
    bool IsRequested() const
    {
        return IsAttached() && m_Module->IsRequested();
    }

    void InitializePacketCommonData();

    Marker* GetPrev() const { return m_Prev; }
    void    SetPrev(Marker* prev) { m_Prev = prev; }

    Marker* GetNext() const { return m_Next; }
    void    SetNext(Marker* next) { m_Next = next; }

    MarkerModule* GetModule() const { return m_Module; }

    //! @brief SpyMarkerModule との関連付けを解除します。
    void Attach(MarkerModule& module);

    //! @brief SpyMarkerModule との関連付けを解除します。
    void Detach();

private:
    MarkerModule* m_Module;

    Marker* m_Prev;
    Marker* m_Next;

    MarkerPacket::ValuePacketPayload m_ValuePacketPayload;

    friend class MarkerModule;
};

#else

//! @brief Spy マーカーを制御します。
class Marker
{
public:
    //! @brief コンストラクタです。
    Marker() { }

    //! @brief デストラクタです。
    virtual ~Marker() { }

    //! @brief MarkerModule へのアタッチの有無を取得します。
    bool IsAttached() const { return false; }

    //! @brief マーカーに値を追加します。
    //! @param[in]  description  マーカーの説明
    //! @param[in]  r            マーカー色の r 成分
    //! @param[in]  g            マーカー色の g 成分
    //! @param[in]  b            マーカー色の b 成分
    void PushValue(const char* description, u8 r, u8 g, u8 b)
    {
        NW_UNUSED_VARIABLE( description );
        NW_UNUSED_VARIABLE( r );
        NW_UNUSED_VARIABLE( g );
        NW_UNUSED_VARIABLE( b );
    }
};

#endif // NW_SND_SPY_ENABLE

} // namespace spy
} // namespace snd
} // namespace nw

#endif // NW_SND_SPY_MARKER_MODULE_H_
