﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_STREAM_SOUND_PREFETCH_FILE_H_
#define NW_SND_STREAM_SOUND_PREFETCH_FILE_H_

#include <nw/types.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_StreamSoundFile.h>
#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/ut/res/ut_ResTypes.h>

namespace nw {
namespace snd {
namespace internal {

/*
    ストリームプリフェッチファイル (*.bfstp) の構造

    bfstp
     |
     +-- FileHeader
     +-- InfoBlock // bfstm の InfoBlock と同じ
     |
     +-- PrefetchDataBlock
          |
          +-- BinaryBlockHeader header
          +-- PrefetchDataBlockBody
              |
              +-- Table<PrefetchData>------------+
                                                 |
              +----------------------------------+
              |
              +-- u32 count ---------------------+
              +-- PrefetchData item[0] ----------+
              +-- PrefetchData item[1] ----------+
                          :                      |
              +----------------------------------+
              |
              +-> PrefetchData
              |
              +-- u32 startFrame
              +-- u32 prefetchSize // PrefetchSample の data のバイト数
              +-- u32 reserved[1]
              +-- reference to PrefetchSample --+
                                                 |
              +----------------------------------+
              |
              +-> PrefetchSample
              +-- u8  data[...] // (サンプルデータ)

*/

struct StreamSoundPrefetchFile
{
    // 前方宣言
    struct PrefetchDataBlock;

    struct FileHeader : public Util::SoundFileHeader
    {
        const StreamSoundFile::InfoBlock* GetInfoBlock() const;
        const PrefetchDataBlock*          GetPrefetchDataBlock() const;
        u32                               GetPrefetchDataBlockSize() const;
    };

    // --------------------------
    // PREFETCH DATA ブロック (サンプルデータ)
    struct PrefetchSample;

    struct PrefetchData
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 startFrame;
        ut::ResU32 prefetchSize;
        ut::ResU32 reserved[1];
#else
        u32 startFrame;
        u32 prefetchSize;
        u32 reserved[1];
#endif
        Util::Reference toPrefetchSample;

        // アクセサ
        const PrefetchSample* GetPrefetchSample() const;
    };

    struct PrefetchDataBlockBody
    {
        // データ
        Util::Table<PrefetchData> prefetchDataTable;

        // アクセサ
        NW_INLINE u32 GetPrefetchDataCount() const { return prefetchDataTable.count; }
        NW_INLINE const PrefetchData* GetPrefetchData( u32 index ) const
        {
            NW_ASSERT( index < prefetchDataTable.count );
            return &prefetchDataTable.item[ index ];
        }
    };

    struct PrefetchDataBlock
    {
        ut::BinaryBlockHeader   header;
        PrefetchDataBlockBody   body;
    };

    struct PrefetchSample
    {
        // データ
        u8  data[1];

        // アクセサ
        const void* GetSampleAddress() const;
    };
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_STREAM_SOUND_PREFETCH_FILE_H_ */

