﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_StreamSoundPlayer.h
 *
 * @file snd_StreamSoundPlayer.h
 */

#ifndef NW_SND_STREAM_SOUND_PLAYER_H_
#define NW_SND_STREAM_SOUND_PLAYER_H_

#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_StreamTrack.h>
#include <nw/snd/snd_InstancePool.h>
#include <nw/snd/snd_Task.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_CachedFsFileStream.h>
#include <nw/snd/snd_StreamSoundLoader.h>
#include <nw/snd/snd_StreamSoundPrefetchFileReader.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    ストリームデータのパラメータセットです。
//!
//!           この情報は @ref StreamSoundHandle::ReadStreamDataInfo から取得できます。
//!
//! @see StreamSoundHandle::ReadStreamDataInfo
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct StreamDataInfo
{
    //---------------------------------------------------------------------------
    //! @brief    ストリームデータがループするなら true、
    //!           終端で終了するなら false となります。
    //---------------------------------------------------------------------------
    bool loopFlag;

    //---------------------------------------------------------------------------
    //! @brief    ストリームデータのサンプリングレートです。
    //---------------------------------------------------------------------------
    int sampleRate;

    //---------------------------------------------------------------------------
    //! @brief    ストリームデータがループする時のループ開始位置を、
    //!           ストリームの先頭からのサンプル数で表します。
    //---------------------------------------------------------------------------
    u32 loopStart;

    //---------------------------------------------------------------------------
    //! @brief    ストリームデータがループする時のループ終了位置を、
    //!           ストリームの先頭からのサンプル数で表します。
    //!           ループしない時は、データの終端をサンプル数で表します。
    //!           (再生される最後のサンプルの次のサンプルを指します)
    //---------------------------------------------------------------------------
    u32 loopEnd;

    //---------------------------------------------------------------------------
    //! @brief    ストリームデータがループする時のループ開始位置を、
    //!           ストリームの先頭からのサンプル数で表します。
    //!
    //!           0.3.0 より loopStart の値が、ハードウェアの制約による
    //!           補正がかかる前の値を返すように変更されました。
    //!           以前の値を取得したい場合にはこちらの値を使用してください。
    //!           ただし、このメンバの使用は非推奨で、将来的に削除予定です。
    //---------------------------------------------------------------------------
    u32 compatibleLoopStart;

    //---------------------------------------------------------------------------
    //! @brief    ストリームデータがループする時のループ終了位置を、
    //!           ストリームの先頭からのサンプル数で表します。
    //!           ループしない時は、データの終端をサンプル数で表します。
    //!
    //!           0.3.0 より loopEnd の値が、ハードウェアの制約による
    //!           補正がかかる前の値を返すように変更されました。
    //!           以前の値を取得したい場合にはこちらの値を使用してください。
    //!           ただし、このメンバの使用は非推奨で、将来的に削除予定です。
    //---------------------------------------------------------------------------
    u32 compatibleLoopEnd;
};

namespace internal {
namespace driver {

class StreamSoundPlayer;
class StreamBufferPool;

class StreamSoundPlayer : public BasicSoundPlayer, public SoundThread::PlayerCallback
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    enum SetupResult
    {
        SETUP_SUCCESS = 0,
        SETUP_ERR_CANNOT_ALLOCATE_BUFFER,
        SETUP_ERR_UNKNOWN
    };

    enum StartOffsetType
    {
        START_OFFSET_TYPE_SAMPLE,
        START_OFFSET_TYPE_MILLISEC
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
    StreamSoundPlayer();
    virtual ~StreamSoundPlayer();

    virtual void Initialize();
    virtual void Finalize();

    void SetLoaderManager(StreamSoundLoaderManager* manager)
    {
        m_pLoaderManager = manager;
    }

    struct SetupArg
    {
        StreamBufferPool* pBufferPool;
        u32 allocChannelCount;
        u16 allocTrackFlag;
        u8 fileType;    // enum StreamFileType
        bool loopFlag;
        TrackDataInfos trackInfos;
        u32 loopStart;
        u32 loopEnd;
        f32 pitch;
        u8 mainSend;
        u8 fxSend[AUX_BUS_NUM];
    };
    void Setup(const SetupArg& arg);

    struct PrepareBaseArg
    {
        driver::StreamSoundPlayer::StartOffsetType startOffsetType;
        s32 offset;
        s32 delayTime;
        StreamRegionCallback regionCallback;
        void* regionCallbackArg;
        char filePath[internal::FILE_PATH_MAX];
        FileStreamHookParam fileStreamHookParam;
    #if defined(NW_PLATFORM_CAFE)
        FSClient* fsClient;
        FsFileStream::FsCommandBlockPool* fsCommandBufferPool;
        FSPriority fsPriority;
    #endif

        PrepareBaseArg()
            : startOffsetType(START_OFFSET_TYPE_SAMPLE)
            , offset(0)
            , delayTime(0)
            , regionCallback(NULL)
            , regionCallbackArg(NULL)
            , fileStreamHookParam()
        #if defined(NW_PLATFORM_CAFE)
        #endif
        {
            std::memset(filePath, 0, sizeof(char) * internal::FILE_PATH_MAX);
        }
    };
    struct PrepareArg
    {
        PrepareBaseArg baseArg;
        void* cacheBuffer;
        size_t cacheSize;

        PrepareArg()
            : cacheBuffer(NULL)
            , cacheSize(0)
        {}
    };
    void Prepare(const PrepareArg& arg);
    struct PreparePrefetchArg
    {
        PrepareBaseArg baseArg;
        const void* strmPrefetchFile;

        PreparePrefetchArg()
            : strmPrefetchFile(NULL)
        {}
    };
    void PreparePrefetch(const PreparePrefetchArg& arg);

    virtual void Start();
    virtual void Stop();
    virtual void Pause( bool flag );

    //------------------------------------------------------------------
    bool IsFinalizing() const { return m_IsFinalizing; }
    bool IsSuspendByLoadingDelay() const { return m_LoadWaitFlag; }
    bool IsPrepared() const { return m_IsPrepared || m_IsPreparedPrefetch; }

    //------------------------------------------------------------------
    // トラックパラメータ
    void SetTrackVolume( u32 trackBitFlag, f32 volume );
    void SetTrackInitialVolume( u32 trackBitFlag, u32 volume );
    void SetTrackOutputLine( u32 trackBitFlag, u32 outputLine );
    void ResetTrackOutputLine( u32 trackBitFlag );

    void SetTrackTvVolume( u32 trackBitFlag, f32 volume );
    void SetTrackChannelTvMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& param );
    void SetTrackTvPan( u32 trackBitFlag, f32 pan );
    void SetTrackTvSurroundPan( u32 trackBitFlag, f32 span );
    void SetTrackTvMainSend( u32 trackBitFlag, f32 send );
    void SetTrackTvFxSend( u32 trackBitFlag, AuxBus bus, f32 send );

    void SetTrackDrcVolume( u32 drcIndex, u32 trackBitFlag, f32 volume );
    void SetTrackChannelDrcMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& param );
    void SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan );
    void SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span );
    void SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send );
    void SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send );

    //------------------------------------------------------------------
    // 情報取得
    bool ReadStreamDataInfo( StreamDataInfo* info ) const;
    long GetPlayLoopCount() const { return m_ActiveFlag ? m_LoopCounter : -1; }
    long GetPlaySamplePosition(bool isOriginalSamplePosition) const;
    f32 GetFilledBufferPercentage() const;

    //------------------------------------------------------------------
    // トラック取得
    StreamTrack* GetPlayerTrack( int trackNo );
    const StreamTrack* GetPlayerTrack( int trackNo ) const;

    //------------------------------------------------------------------
    // コマンド処理
    bool LoadHeader(
        bool result,
        AdpcmParam* adpcmParam[],
        u16 requestIndex
    );
    bool LoadStreamData(
        bool result,
        const LoadDataParam& loadDataParam
    );
    bool LoadStreamData(
        bool result,
        const LoadDataParam& loadDataParam,
        bool usePrefetchFlag,
        u32 currentPrefetchBlockIndex,
        u32 currentPrefetchBlockBytes
    );

    void ForceFinish() { m_FinishFlag = true; }

    //------------------------------------------------------------------
protected:
    virtual void OnUpdateFrameSoundThread() { Update(); }
    virtual void OnShutdownSoundThread() { Stop(); }

    //------------------------------------------------------------------
private:
    struct ItemData
    {
        f32 pitch;
        f32 mainSend;
        f32 fxSend[ AUX_BUS_NUM ];

        void Set( const SetupArg& arg );
    };

    struct TrackData
    {
        f32 volume;
        f32 lpfFreq;
        s32 biquadType;
        f32 biquadValue;
        f32 pan;
        f32 span;
        f32 mainSend;
        f32 fxSend[ AUX_BUS_NUM ];

        void Set( const StreamTrack* track );
    };

    struct PrefetchLoadDataParam : public LoadDataParam
    {
        u32 prefetchBlockIndex;
        u32 prefetchBlockBytes;
    };

    struct PrefetchIndexInfo
    {
        u32 lastBlockIndex;
        u32 loopStartInBlock;
        u32 loopStartBlockIndex;
        u32 loopBlockNum;

        void Initialize( const StreamDataInfoDetail& streamDataInfo );

        bool IsOverLastBlock( u32 blockIndex )
        {
            return blockIndex > lastBlockIndex;
        }

        u32 GetBlockOffsetFromLoopEnd( u32 blockIndex )
        {
            return ( blockIndex - lastBlockIndex ) % loopBlockNum;
        }

        bool IsLoopStartBlock( u32 blockOffsetFromLoopEnd )
        {
            return blockOffsetFromLoopEnd == 1;
        }

        bool IsLastBlock( u32 blockIndex, u32 blockOffsetFromLoopEnd )
        {
            return ( lastBlockIndex == 0 ) || ( blockIndex == lastBlockIndex ) || ( IsOverLastBlock( blockIndex ) && ( blockOffsetFromLoopEnd == 0 ) );
        }
    };

    void StartPlayer();
    void FinishPlayer();
    bool SetupPlayer();

    void Update();
    void UpdateBuffer();
    void UpdateVoiceParams( StreamTrack* track );

    void SetOutputParam( OutputParam* pOutputParam, const OutputParam& trackParam, const TrackData& trackData );

    void MixSettingForOutputParam( OutputParam* pOutputParam, s32 channelIndex, MixMode mixMode );
    void ApplyTvOutputParamForMultiChannel( OutputParam* pOutputParam, MultiVoice* pVoice, s32 channelIndex, MixMode mixMode );
    void ApplyDrcOutputParamForMultiChannel( OutputParam* pOutputParam, MultiVoice* pVoice, s32 channelIndex, MixMode mixMode, u32 drcIndex );

    bool AllocVoices();
    void FreeVoices();

    bool TryAllocLoader();
    void FreeLoader();

    bool AllocStreamBuffers();
    void FreeStreamBuffers();

    void UpdateLoadingBlockIndex();
    void UpdatePauseStatus();
    bool CheckDiskDriveError() const;

    bool IsBufferEmpty() const;

    bool SetupTrack( const SetupArg& arg );
    void SetPrepareBaseArg( const PrepareBaseArg& baseArg );
    void RequestLoadHeader( const PrepareArg& arg );
    bool ReadPrefetchFile( StreamSoundPrefetchFileReader& reader );
    bool ApplyStreamDataInfo( const StreamDataInfoDetail& streamDataInfo );

    u32 GetOriginalPlaySamplePosition( u32 playSamplePosition, const StreamDataInfoDetail& streamDataInfo ) const;
    u32 GetStartOffsetSamples( const StreamDataInfoDetail& streamDataInfo );
    bool IsValidStartOffset( const StreamDataInfoDetail& streamDataInfo );
    void ApplyTrackDataInfo( const StreamDataInfoDetail& streamDataInfo );

    bool LoadPrefetchBlocks(StreamSoundPrefetchFileReader& reader);
    void PreparePrefetchOnLastBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo);
    bool PreparePrefetchOnLoopStartBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, StreamSoundPrefetchFileReader& reader);
    void PreparePrefetchOnLoopBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, u32 blockOffsetFromLoopEnd);
    bool PreparePrefetchOnNormalBlock(PrefetchLoadDataParam* param, u32 blockIndex, StreamSoundPrefetchFileReader& reader);

    bool SetAdpcmInfo(
        StreamSoundPrefetchFileReader& reader,
        const StreamDataInfoDetail& streamDataInfo,
        AdpcmParam* adpcmParam,
        AdpcmContext* adpcmContext);
    bool SetAdpcmLoopInfo(
        StreamSoundPrefetchFileReader& reader,
        const StreamDataInfoDetail& streamDataInfo,
        AdpcmParam* adpcmParam,
        AdpcmContext* adpcmContext);

    static void VoiceCallbackFunc(
        MultiVoice* voice,
        MultiVoice::VoiceCallbackStatus status,
        void* arg
    );

    bool m_IsInitialized;       // Setup が呼ばれた
    bool m_IsPrepared;          // 準備完了フラグ
    bool m_IsFinalizing;
    bool m_IsPreparedPrefetch;

    bool m_PauseStatus;
    bool m_LoadWaitFlag;        // バッファが溜まるまで、一時停止解除を遅延させる
    bool m_LoadFinishFlag;
    bool m_ReportLoadingDelayFlag;      // ロード遅延メッセージ

    bool m_IsRegisterPlayerCallback;
    bool m_UseDelayCount;               // 互換性保持のため、DelayCountを使うかどうかの判定に使用します。
    u8   m_Padding[2];

    s32 m_LoopCounter;
    int m_PrepareCounter;

    StreamSoundLoaderManager* m_pLoaderManager;
    StreamSoundLoader* m_pLoader;

    // ストリームバッファ
    StreamBufferPool* m_pBufferPool;
    u32 m_BufferBlockCount;  // 波形バッファ数

    // NOTE: StreamBufferPool からロード用のバッファが与えられるが、
    //       これを、bcstm 内の 1 ブロック (現在は 8KB で固定) に分割して利用する。
    //       以下に出てくる「ブロック」とは、この 8KB で区切られたメモリ領域を指す。
    u32 m_LoadingBufferBlockIndex;        // ロード用バッファのうち、ロード中のブロックの通し番号

    u32 m_PlayingBufferBlockIndex;        // ロード用バッファのうち、再生中のブロックの通し番号
    u32 m_LastPlayFinishBufferBlockIndex; // ロード用バッファのうち、最後に再生完了したブロックの通し番号


    StartOffsetType m_StartOffsetType;
    s32 m_StartOffset;
    s32 m_DelayCount;

    u16 m_TaskRequestIndex;

    // ストリームデータ情報
    u8 m_FileType;  // enum StreamFileType
    bool m_LoopFlag;
    u8 m_Padding2[2];
    StreamDataInfoDetail m_StreamDataInfo;  // snd_StreamSoundLoader.h
    u32 m_LoopStart;
    u32 m_LoopEnd;
    ItemData m_ItemData;

    // プリフェッチデータ情報
    const void*                                     m_pStreamPrefetchFile;
    AdpcmParam                                      m_PrefetchAdpcmParam[STRM_CHANNEL_NUM];
    StreamSoundPrefetchFileReader::PrefetchDataInfo m_PrefetchDataInfo;
    s32                                             m_PrefetchOffset;

    // トラックとチャンネル
    u32 m_ChannelCount;
    u32 m_TrackCount;
    StreamChannel m_Channels[ STRM_CHANNEL_NUM ];
    StreamTrack m_Tracks[ STRM_TRACK_NUM ];

    struct WaveBufferInfo
    {
        u32 sampleBegin;
        u32 sampleLength;
        u32 loopCount; // このバッファの再生が完了したら、何回ループ回数が加算されるか
    };
    WaveBufferInfo m_WaveBufferInfo[ STRM_DATA_LOAD_TASK_MAX ];

    StreamSoundPlayer::PrepareArg m_PrepareArg;
    bool m_IsSucceedPrepare;
    StreamSoundPlayer::SetupArg m_SetupArg;

    static u16 s_TaskRequestIndexCount;
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_STREAM_SOUND_PLAYER_H_ */

