﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundHandle.h
 *
 * @file snd_SoundHandle.h
 */

#ifndef NW_SND_SOUND_HANDLE_H_
#define NW_SND_SOUND_HANDLE_H_

#include <nw/ut/ut_PreProcessor.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_Debug.h>
#include <nw/snd/snd_Global.h>      // BIQUAD_FILTER_TYPE_USER_MAX

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  再生したサウンドの操作を行うためのハンドルクラスです。
//!
//!
//!         ■ 概要
//!
//!         サウンドハンドルは、 @ref SoundArchivePlayer::PrepareSound または
//!         @ref SoundArchivePlayer::StartSound を呼び出すことにより、
//!         サウンドと関連付けられます。
//!
//!         その際、ハンドルが既に別のサウンドと関連付けられていた場合は、
//!         古いサウンドとの関連は外され、新たに再生するサウンドと関連付けられます。
//!
//!         下記の状態になると、ハンドルとサウンドの関連が外れ、
//!         その後はサウンドに対しての操作が行えないようになります。
//!
//!         - サウンドが終端に達して停止したとき
//!         - フェードフレーム 0 で @ref Stop をコールしたとき
//!         - フェードフレームを 1 以上に指定して @ref Stop をコールし、
//!           そのフレーム数だけ時間が経過して停止したとき
//!         - 明示的に SoundHandle::DetachSound を呼び出したとき
//!         - プレイヤープライオリティの判定でサウンドが停止したとき
//!
//!         したがって、@ref IsAttachedSound を毎フレーム監視することで、
//!         サウンドが終端に達して停止したかどうか、確認することができます。
//!         ただし、@ref Stop での停止や明示的な @ref DetachSound 、
//!         プレイヤープライオリティ判定でのサウンド停止の影響を考慮しておく必要があります。
//!
//!         プレイヤープライオリティの判定については、サウンドシステムマニュアルの
//!         「プライオリティの動作」の章をご参照ください。
//!
//!         @n
//!
//!         ■ サウンドハンドルの種類
//!
//!         サウンドにはシーケンスサウンド・ストリームサウンド・ウェーブサウンドの
//!         3 種類があります。
//!         SoundHandle クラスはこれらすべてのサウンドの共通の操作を扱うことができます。
//!
//!         SoundHandle クラスでは扱えない、各サウンド専用の操作を行うために、
//!         @ref SequenceSoundHandle クラス、
//!         @ref StreamSoundHandle クラス、
//!         @ref WaveSoundHandle クラスが用意されています。
//!
//!
//!         @n
//!
//!         ■ 出力先別パラメータ
//!
//!         Cafe では、[1] メイン出力 (TV 出力) [2] DRC 出力 [3] リモコン出力の
//!         3 系統の出力があります。サウンドハンドルでは、音量・ピッチ・パン
//!         などのパラメータを設定することができますが、
//!         具体的には下記のような API の影響を受けます。
//!
//!         (全体)
//!         - 音量 ... @ref SetVolume, @ref FadeIn, @ref Pause, @ref Stop
//!         - パン ... @ref SetPan
//!         - サラウンドパン ... @ref SetSurroundPan
//!         - メインセンド ... @ref SetMainSend
//!         - エフェクトセンド ... @ref SetFxSend
//!         - ピッチ ... @ref SetPitch
//!         - LPF ... @ref SetLpfFreq
//!         - Biquad フィルタ ... @ref SetBiquadFilter
//!         - 出力先 ... @ref SetOutputLine, @ref ResetOutputLine
//!
//!         (TV 出力向け)
//!         - 音量 ... 「全体」の音量に加え、@ref SetOutputVolume
//!         - パン ... 「全体」のパンに加え、@ref SetOutputPan
//!         - サラウンドパン ... 「全体」のサラウンドパンに加え、@ref SetOutputSurroundPan
//!         - メインセンド ... 「全体」のメインセンドに加え、@ref SetOutputMainSend
//!         - エフェクトセンド ... 「全体」のエフェクトセンドに加え、@ref SetOutputFxSend
//!
//!         (DRC 出力向け)
//!         - 音量 ... 「全体」の音量に加え、@ref SetOutputVolume
//!         - パン ... 「全体」のパンに加え、@ref SetOutputPan
//!         - サラウンドパン ... 「全体」のサラウンドパンに加え、@ref SetOutputSurroundPan
//!         - メインセンド ... 「全体」のメインセンドに加え、@ref SetOutputMainSend
//!         - エフェクトセンド ... 「全体」のエフェクトセンドに加え、@ref SetOutputFxSend
//!
//!         (リモコン出力向け)
//!         - 音量 ... @ref SetRemoteOutVolume
//!
//!         @n
//!
//!         各パラメータは、上記の影響の他に、下記の影響を受けます。
//!         - SoundMaker で指定したデータの影響
//!         - @ref SoundPlayer や @ref SoundActor による影響
//!         - 3D サウンド計算 (@ref Sound3DEngine) による影響
//!
//! @see SoundArchivePlayer クラス
//! @see SequenceSoundHandle クラス
//! @see StreamSoundHandle クラス
//! @see WaveSoundHandle クラス
//!
//! @date 2012/04/24 仕様変更にともない、パラメータ伝播の説明を修正
//! @date 2011/11/25 SetDrcPan 追加にともなう仕様変更のため、説明を追記
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundHandle
{
public:
    //---------------------------------------------------------------------------
    //! @brief  ポーズ状態をあらわす列挙体です。
    //!
    //!         - PAUSE_STATE_NORMAL のときに、@ref SoundHandle::Pause( true, ... )
    //!           を呼び出すと、状態は PAUSE_STATE_PAUSING になります。
    //!         - PAUSE_STATE_PAUSING のときに、
    //!           - @ref SoundHandle::Pause( true, ... ) を呼び出すと、
    //!             状態は PAUSE_STATE_PAUSING のままです。
    //!           - @ref SoundHandle::Pause( false, ... ) を呼び出すと、
    //!             状態は PAUSE_STATE_UNPAUSING になります。
    //!           - フェード処理が終了すると、状態は PAUSE_STATE_PAUSED になります。
    //!         - PAUSE_STATE_PAUSED のとき、@ref SoundHandle::Pause( false, ... )
    //!           を呼び出すと、状態は PAUSE_STATE_UNPAUSING になります。
    //!         - PAUSE_STATE_UNPAUSING のときに、
    //!           - @ref SoundHandle::Pause( true, ... ) を呼び出すと、
    //!             状態は PAUSE_STATE_PAUSING になります。
    //!           - @ref SoundHandle::Pause( false, ... ) を呼び出すと、
    //!             状態は PAUSE_STATE_UNPAUSING のままです。
    //!           - フェード処理が終了すると、状態は PAUSE_STATE_NORMAL になります。
    //!
    //! @see GetPauseState
    //!
    //! @date 2011/07/13 初版
    //---------------------------------------------------------------------------
    enum PauseState
    {
        PAUSE_STATE_NORMAL,     //!< 非ポーズ状態です。
        PAUSE_STATE_PAUSING,    //!< ポーズへの移行中をあらわします。
        PAUSE_STATE_PAUSED,     //!< ポーズ中をあらわします。
        PAUSE_STATE_UNPAUSING,  //!< ポーズ解除への移行中をあらわします。
        PAUSE_STATE_INVALID //!< 無効な状態です。ハンドルにサウンドが関連付けられていないことをあらわします。
    };

    //---------------------------------------------------------------------------
    //! @brief  ミュート状態をあらわす列挙体です。
    //!
    //!         - MUTE_STATE_NORMAL のときに、@ref SoundHandle::Mute( true, ... )
    //!           を呼び出すと、状態は MUTE_STATE_MUTING になります。
    //!         - MUTE_STATE_MUTING のときに、
    //!           - @ref SoundHandle::Mute( true, ... ) を呼び出すと、
    //!             状態は MUTE_STATE_MUTING のままです。
    //!           - @ref SoundHandle::Mute( false, ... ) を呼び出すと、
    //!             状態は MUTE_STATE_UNMUTING になります。
    //!           - フェード処理が終了すると、状態は MUTE_STATE_MUTED になります。
    //!         - MUTE_STATE_MUTED のとき、@ref SoundHandle::Mute( false, ... )
    //!           を呼び出すと、状態は MUTE_STATE_MUTING になります。
    //!         - MUTE_STATE_UNMUTING のときに、
    //!           - @ref SoundHandle::Mute( true, ... ) を呼び出すと、
    //!             状態は MUTE_STATE_MUTING になります。
    //!           - @ref SoundHandle::Mute( false, ... ) を呼び出すと、
    //!             状態は MUTE_STATE_UNMUTING のままです。
    //!           - フェード処理が終了すると、状態は MUTE_STATE_NORMAL になります。
    //!
    //! @see GetMuteState
    //!
    //! @date 2013/05/20 初版
    //---------------------------------------------------------------------------
    enum MuteState
    {
        MUTE_STATE_NORMAL,    //!< 非ミュート状態です。
        MUTE_STATE_MUTING,    //!< ミュートへの移行中をあらわします。
        MUTE_STATE_MUTED,     //!< ミュート中をあらわします。
        MUTE_STATE_UNMUTING,  //!< ミュート解除への移行中をあらわします。
        MUTE_STATE_INVALID //!< 無効な状態です。ハンドルにサウンドが関連付けられていないことをあらわします。
    };

    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //! @brief コンストラクタです。
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    SoundHandle() : m_pSound( NULL ) { }
    //! @brief デストラクタです。
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    ~SoundHandle() { DetachSound(); }
    //@}

    //----------------------------------------
    //! @name 再生/停止/一時停止/再開
    //@{
    //---------------------------------------------------------------------------
    //! @brief    再生準備が完了したサウンドを再生します。
    //!
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           サウンドの再生を開始するためには、
    //!           @ref SoundArchivePlayer::PrepareSound を呼び出した後、
    //!           再生準備が完了していなければなりません。
    //!           この関数は、再生準備が完了したサウンドの再生を開始します。
    //!           再生準備が完了していないサウンドは、
    //!           完了するまで待ってから再生を開始します。
    //!
    //! @see SoundArchivePlayer::PrepareSound
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void StartPrepared()
    {
        if ( IsAttachedSound() )
        {
            m_pSound->StartPrepared();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを停止します。
    //!
    //!           ハンドルに関連付けられたサウンドを停止します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけて
    //!           フェードアウトさせることができます。
    //!           0 を指定した場合は、フェードアウトを行いません。
    //!           ただし、シーケンスサウンドで発音中の音は、
    //!           エンベロープのリリースを発音しすべての減衰が完了した後に
    //!           サウンドが停止します。
    //!
    //!           フェードアウトの音量制御は、フェードインと共有されます。
    //!           フェードアウトにかかるフレーム数は、
    //!           最大音量から音が消えるまでにかかる変化速度を表しますので、
    //!           フェードイン中にフェードアウトを指定した時などは、
    //!           指定したフレーム数よりも短い時間で
    //!           フェードアウトが完了する可能性があります。
    //!
    //! @param[in] fadeFrames   フェードアウトにかけるフレーム数です。
    //!                         フレーム数は @ref SoundArchivePlayer::Update
    //!                         の呼び出し回数で換算されます。
    //!                         fadeFrames には 0 以上の値を指定してください。
    //!                         負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see Pause
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Stop( int fadeFrames )
    {
        if ( IsAttachedSound() )
        {
            NW_SND_LOG_APICALL_NW("SoundHandle::Stop [ID:%08x]", m_pSound->GetId());
            m_pSound->Stop( fadeFrames );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを一時停止または再開します。
    //!
    //!           ハンドルに関連付けられたサウンドを一時停止または再開します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトしながら一時停止、
    //!           またはフェードインしながら再開させることができます。
    //!           0 を指定した場合は、即座に一時停止または再開します。
    //!
    //!           一時停止・再開時のフェードは、再生開始時のフェードイン、
    //!           停止時のフェードアウトとは独立してはたらきます。
    //!           フェードにかかるフレーム数は、最大音量から音が消えるまで、
    //!           あるいは発音していない状態から最大音量に達するまでに
    //!           かかる変化速度を表しますので、
    //!           フェード中にさらにフェードを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードが完了する可能性があります。
    //!
    //!           ポーズ状態については、@ref GetPauseState および @ref PauseState
    //!           の説明をごらんください。
    //!
    //! @param[in]    flag        true なら一時停止、false なら再開します。
    //! @param[in]    fadeFrames  フェードイン・フェードアウトにかけるフレーム数です。
    //!                           フレーム数は @ref SoundArchivePlayer::Update
    //!                           の呼び出し回数で換算されます。
    //!                           fadeFrames には 0 以上の値を指定してください。
    //!                           負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see IsPause
    //! @see GetPauseState
    //! @see PauseState
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/13 ポーズ状態についての説明を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Pause( bool flag, int fadeFrames )
    {
        if ( IsAttachedSound() )
        {
            NW_SND_LOG_APICALL_NW("SoundHandle::Pause [ID:%08x flag:%s frame:%d]", m_pSound->GetId(),flag?"true":"false",fadeFrames);
            m_pSound->Pause( flag, fadeFrames );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドをミュートまたは再開します。
    //!
    //!         ハンドルに関連付けられたサウンドをミュートまたは再開します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         fadeFrames で指定したフレーム数をかけてフェードアウトしながらミュート、
    //!         またはフェードインしながら再開させることができます。
    //!         0 を指定した場合は、即座にミュートまたは再開します。
    //!
    //!         ミュート・再開時のフェードは、再生開始や一時停止のフェードイン、再生/停止開始時や一時停止/再開のフェードイン、
    //!         フェードアウトとは独立してはたらきます。
    //!         フェードにかかるフレーム数は、最大音量から音が消えるまで、
    //!         あるいは発音していない状態から最大音量に達するまでに
    //!         かかる変化速度を表しますので、
    //!         フェード中にさらにフェードを指定した時などは、
    //!         指定したフレーム数よりも短い時間でフェードが完了する可能性があります。
    //!
    //!         ミュート状態については、@ref GetMuteState および @ref MuteState
    //!         の説明をごらんください。
    //!
    //! @param[in]    flag        true ならミュート、false なら再開します。
    //! @param[in]    fadeFrames  フェードイン・フェードアウトにかけるフレーム数です。
    //!                           フレーム数は @ref SoundArchivePlayer::Update
    //!                           の呼び出し回数で換算されます。
    //!                           fadeFrames には 0 以上の値を指定してください。
    //!                           負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see IsMute
    //! @see GetMuteState
    //! @see MuteState
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2013/05/14 初版
    //---------------------------------------------------------------------------
    void Mute( bool flag, int fadeFrames )
    {
        if ( IsAttachedSound() ) m_pSound->Mute( flag, fadeFrames );
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドの再生準備が完了しているかどうかを調べます。
    //!
    //!         @ref SoundArchivePlayer::PrepareSound を呼び出した後、
    //!         そのサウンドの再生準備が完了しているかどうかを調べます。
    //!         再生準備が完了したサウンドは、
    //!         @ref SoundHandle::StartPrepared を呼び出した際に、
    //!         すぐに再生を始めることができます。
    //!
    //!         @ref SoundArchivePlayer::StartSound や HoldSound を呼び出した場合も、
    //!         内部で PrepareSound を呼び出しているため、内部呼び出しの PrepareSound
    //!         が完了した時点で、本関数は true を返します。
    //!
    //! @return   サウンドの再生準備が完了していれば true を返します。
    //!
    //! @see SoundArchivePlayer::PrepareSound
    //! @see StartPrepared
    //!
    //! @date 2012/06/11 StartSound / HoldSound 呼び出し時の挙動について補足
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPrepared() const
    {
        if ( IsAttachedSound() && m_pSound->IsPrepared() )
        {
            return true;
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドが一時停止中かどうかを調べます。
    //!
    //! @return   一時停止状態へのフェード中および、一時停止中の場合に true を返します。
    //!           これは、 @ref GetPauseState で PAUSE_STATE_PAUSING および、
    //!           PAUSE_STATE_PAUSED が返っている状態と同じです。
    //!
    //! @see GetPauseState
    //!
    //! @date 2011/07/13 返り値の説明を調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPause() const
    {
        if ( IsAttachedSound() && m_pSound->IsPause() )
        {
            return true;
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドがミュート中かどうかを調べます。
    //!
    //! @return   ミュート状態へのフェード中および、ミュート中の場合に true を返します。
    //!
    //! @date 2013/05/14 初版
    //---------------------------------------------------------------------------
    bool IsMute() const
    {
        if ( IsAttachedSound() && m_pSound->IsMute() )
        {
            return true;
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのポーズ状態を調べます。
    //!
    //! @return ポーズ状態を返します。
    //!
    //! @see IsPause
    //! @see PauseState
    //!
    //! @date 2011/07/13 初版
    //---------------------------------------------------------------------------
    PauseState GetPauseState() const
    {
        if ( ! IsAttachedSound() )
        {
            return PAUSE_STATE_INVALID;
        }

        internal::BasicSound::PauseState state = m_pSound->GetPauseState();
        return static_cast<PauseState>(state);
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのミュート状態を調べます。
    //!
    //! @return ミュート状態を返します。
    //!
    //! @see IsMute
    //! @see MuteState
    //!
    //! @date 2013/05/20 初版
    //---------------------------------------------------------------------------
    MuteState GetMuteState() const
    {
        if ( ! IsAttachedSound() )
        {
            return MUTE_STATE_INVALID;
        }

        internal::BasicSound::MuteState state = m_pSound->GetMuteState();
        return static_cast<MuteState>(state);
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを再生開始時にフェードインさせます。
    //!
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数でフェードインの指定を行うと、
    //!           @ref SoundArchivePlayer::StartSound または
    //!           @ref SoundHandle::StartPrepared を呼び出して再生を開始してから、
    //!           最初の @ref SoundArchivePlayer::Update が呼び出されたときに
    //!           フェードインが設定されます。
    //!
    //!           サウンドの再生が開始され、最初の @ref SoundArchivePlayer::Update
    //!           が呼ばれた後にこの関数を呼び出しても効果がありません。
    //!
    //!           フェードインの音量制御は、停止時のフェードアウトと共有されます。
    //!
    //! @param[in]    frames  フェードインにかけるフレーム数です。
    //!                       フレーム数は @ref SoundArchivePlayer::Update
    //!                       の呼び出し回数で換算されます。
    //!                       frames には 0 以上の値を指定してください。
    //!                       負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see StartPrepared
    //! @see SoundArchivePlayer::StartSound
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void FadeIn( int frames )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->FadeIn( frames );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    フェードイン・フェードアウト完了までの残りフレーム数を取得します。
    //!
    //!           フェード中でない場合は 0 を返します。
    //!
    //! @return   フェード完了までの残りフレーム数を返します。
    //!           フレーム数は @ref SoundArchivePlayer::Update
    //!           の呼び出し回数で換算されます。
    //!
    //! @see FadeIn
    //! @see Stop
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetRemainingFadeFrames() const
    {
        if ( IsAttachedSound() )
        {
            return m_pSound->GetRemainingFadeFrames();
        }
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    一時停止時のフェード完了までの残りフレーム数を取得します。
    //!
    //!           フェード中でない場合は 0 を返します。
    //!
    //! @return   フェード完了までの残りフレーム数を返します。
    //!           フレーム数は @ref SoundArchivePlayer::Update
    //!           の呼び出し回数で換算されます。
    //!
    //! @see Pause
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetRemainingPauseFadeFrames() const
    {
        if ( IsAttachedSound() )
        {
            return m_pSound->GetRemainingPauseFadeFrames();
        }
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの再生を開始した時から現在までの再生経過フレーム数を取得します。
    //!
    //!           一時停止時(PauseState::PAUSE_STATE_PAUSED)のフレーム数は
    //!           カウントされませんが、一時停止や再開のためのフェードの区間はカウントされます。
    //!
    //! @return   再生経過フレーム数を返します。
    //!
    //! @see Pause
    //! @see PauseState
    //!
    //! @date 2013/02/08 初版
    //---------------------------------------------------------------------------
    u32 GetPlayFrameCount() const
    {
        if ( IsAttachedSound() )
        {
            return m_pSound->GetPlayFrameCount();
        }
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    ミュート時のフェード完了までの残りフレーム数を取得します。
    //!
    //!           フェード中でない場合は 0 を返します。
    //!
    //! @return   フェード完了までの残りフレーム数を返します。
    //!           フレーム数は @ref SoundArchivePlayer::Update
    //!           の呼び出し回数で換算されます。
    //!
    //! @see Mute
    //!
    //! @date 2013/05/20 初版
    //---------------------------------------------------------------------------
    int GetRemainingMuteFadeFrames() const
    {
        if ( IsAttachedSound() )
        {
            return m_pSound->GetRemainingMuteFadeFrames();
        }
        return 0;
    }
    //@}

    //----------------------------------------
    //! @name サウンド共通パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  サウンドの音量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの音量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!         0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!         他の音量パラメータと重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //!         音量の変化は frames で指定したフレーム数をかけて行われます。
    //!         音量の変化途中にさらにこの関数を呼び出した場合は、
    //!         その時点での変化途中の音量値を基点として、
    //!         新しく指定したフレーム数をかけて音量を変化させます。
    //!
    //!         出力先別に設定したい場合は、@ref SetOutputVolume を利用してください。
    //!
    //! @param[in]    volume  変更する音量の倍率 ( 0.0 ～ ) です。
    //! @param[in]    frames  音量変化にかけるフレーム数です。
    //!                       フレーム数は @ref SoundArchivePlayer::Update
    //!                       の呼び出し回数で換算されます。
    //!                       frames には 0 以上の値を指定してください。
    //!                       負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see SetOutputVolume
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVolume( f32 volume, int frames = 0 )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetVolume( volume, frames );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのパン (左右の定位) を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのパンを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         pan は、定位の相対変化の値を設定します。
    //!         入力となる波形が、モノラル波形の場合は以下の挙動になります。
    //!         - 0.0 を指定：データで設定されたパンの値から変化しません。
    //!         - 1.0 を指定：中央に定位していた音が右端に定位するようになります。
    //!         - -1.0 を指定：中央に定位していた音が左端に定位するようになります。
    //!
    //!         入力となる波形が、ステレオ波形で、
    //!         パンモードがデュアルパンに設定されている場合には以下の挙動になります。
    //!         - 0.0 を指定：左チャンネルの波形は左端、右チャンネルの波形は右端に定位します。
    //!         - 2.0 を指定：両チャンネルの波形が右端に定位します。
    //!         - -2.0 を指定：両チャンネルの波形が左端に定位します。
    //!         - 1.0、-1.0 の場合には、どちらかのチャンネルの波形が中央に定位します。
    //!
    //!         また、パンモードがバランスパンに設定されている場合には以下の挙動になります。
    //!         - 1.0 を指定すると右チャンネルの波形のみが聞こえるようになります。
    //!         - -1.0 を指定すると左チャンネルの波形のみが聞こえるようになります。
    //!
    //!         デフォルト値は 0.0 です。
    //!
    //!         出力先別に設定したい場合は、@ref SetOutputPan を利用してください。
    //!
    //! @param[in]    pan  0.0 を基準としたパンの相対変化の値です。
    //!
    //! @see SetOutputPan
    //!
    //! @date 2014/04/25 pan の値による挙動の説明を修正
    //! @date 2012/04/24 サウンドに対する操作に仕様変更
    //! @date 2011/10/31 メイン (TV) 出力に対する操作である旨、明記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPan( f32 pan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetPan( pan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのサラウンドパン (前後の定位) を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンドパンを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         surroundPan は、定位の相対変化の値を設定します。
    //!         0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!         1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!         2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!         前方へ定位を移動させたい場合は負の値を指定してください。
    //!
    //!         デフォルト値は 0.0 です。
    //!
    //!         出力先別に設定したい場合は、@ref SetOutputSurroundPan を利用してください。
    //!
    //! @param[in] surroundPan 0.0 を基準としたサラウンドパンの相対変化の値です。
    //!
    //! @see SetOutputSurroundPan
    //!
    //! @date 2012/04/24 サウンドに対する操作に仕様変更
    //! @date 2011/10/31 メイン (TV) 出力に対する操作である旨、明記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetSurroundPan( f32 surroundPan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetSurroundPan( surroundPan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのメインセンド量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのメインセンド量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのメインセンド量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         メインセンドは、
    //!         出力に送るサウンドの音量をエフェクトセンドの後で調節するパラメータです。
    //!         主に、エフェクトのドライ・ウェット成分のバランスを調整するために使用されます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!         メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //!         出力先別に設定したい場合は、@ref SetOutputMainSend を利用してください。
    //!
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see SetFxSend
    //! @see SetOutputMainSend
    //!
    //! @date 2012/04/24 サウンドに対する操作に仕様変更
    //! @date 2011/10/31 メイン (TV) 出力に対する操作である旨、明記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetMainSend( f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetMainSend( send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドのエフェクトセンド量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのエフェクトセンド量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのエフェクトセンド量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!         最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //!         出力先別に設定したい場合は、@ref SetOutputFxSend を利用してください。
    //!
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see AuxBus
    //! @see SetMainSend
    //! @see SetOutputFxSend
    //!
    //! @date 2012/04/24 サウンドに対する操作に仕様変更
    //! @date 2011/10/31 メイン (TV) 出力に対する操作である旨、明記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetFxSend( AuxBus bus, f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetFxSend( bus, send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの音程を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの音程を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音程パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音程 pitch は、周波数の比率で指定します。
    //!           すなわち、1.0 を指定すると音程に影響を与えません。
    //!           2.0 を指定すると再生される周波数が 2 倍になり、
    //!           1 オクターブ高い音程になります。
    //!           0.5 を指定すると 1 オクターブ低い音程になります。
    //!
    //!           デフォルト値は 1.0 です。
    //!
    //! @param[in]    pitch  変更する音程の周波数比率です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPitch( f32 pitch )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetPitch( pitch );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドのローパスフィルタカットオフ値を変更します。
    //!
    //!           ハンドルに関連付けられたローパスフィルタカットオフ値を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、
    //!           他のどのローパスフィルタカットオフパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           lpfFreq は、カットオフの相対変化の値を指定します。
    //!           0.0 を指定するとカットオフの値を変更しません。
    //!           -1.0 を指定すると、フィルタがかかっていない状態から、
    //!           もっともフィルタがかかっている状態 (カットオフ周波数が下がる方向)
    //!           に変更します。
    //!
    //! @param[in] lpfFreq    0.0 を基準としたフィルタカットオフの相対変化の値です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetLpfFreq( f32 lpfFreq )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetLpfFreq( lpfFreq );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドの biquad フィルタの設定を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの biquad フィルタの設定を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         biquad フィルタは複数の箇所での設定が重ね合わされず、
    //!         以下の優先度に従って設定されます。
    //!         優先度が高い箇所でパラメータの設定がされた場合、
    //!         それより下位の設定は上書きされます。
    //!
    //!         -# サウンドハンドルでの設定
    //!         -# サウンドプレイヤーでの設定
    //!         -# アンビエントパラメータ構造体での設定
    //!         -# シーケンスデータでの設定
    //!
    //!         フィルタの種類 type は @ref BiquadFilterType の値を使用します。
    //!         プリセットで用意されているフィルタの種類の他、
    //!         ユーザーが登録したフィルタを選択することができます。
    //!
    //!         type は BIQUAD_FILTER_TYPE_MIN ～ BIQUAD_FILTER_TYPE_MAX の値を指定します。
    //!         BIQUAD_FILTER_TYPE_MIN、BIQUAD_FILTER_TYPE_MAX については @ref BiquadFilterType をご参照ください。
    //!         範囲外の値を設定すると、Debug 版 / Development 版ではアサートで停止します。
    //!         Release 版では無視されます。
    //!
    //!         フィルタのかかり具合を指定する value は、
    //!         0.0f ～ 1.0f の値で指定します。
    //!         値の意味はフィルタの係数の種類によって変化します。
    //!
    //! @param[in] type     フィルタの種類です (BIQUAD_FILTER_TYPE_MIN ～ BIQUAD_FILTER_TYPE_MAX)。
    //! @param[in] value    フィルタのかかり具合です (0.0f ～ 1.0f)。
    //!
    //! @see BiquadFilterType
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetBiquadFilter( int type, f32 value )
    {
        NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_MIN , BIQUAD_FILTER_TYPE_MAX );
        if ( IsAttachedSound() )
        {
            m_pSound->SetBiquadFilter( type, value );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    プレイヤープライオリティを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのプレイヤープライオリティを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数は、サウンドアーカイブ中のデータで指定されている
    //!           プレイヤープライオリティの値を変更します。
    //!
    //!           priority の値の範囲は 0～127 で、大きいほど優先度が高くなります。
    //!
    //! @param[in] priority   プレイヤープライオリティの値です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPlayerPriority( int priority )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetPlayerPriority( priority );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドの出力先を設定します。
    //!
    //!         ハンドルに関連付けられたサウンドの出力先を設定します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         出力先は @ref OutputLine のビットフラグの論理和で指定します。
    //!         指定した値は、それまでに設定されていた値を上書きします。
    //!         一度指定した値を無効にする場合は @ref ResetOutputLine を呼び出してください。
    //!
    //! @param[in] lineFlag     出力先のビットフラグです。
    //!
    //! @see OutputLine
    //! @see ResetOutputLine
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetOutputLine( u32 lineFlag )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputLine( lineFlag );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  サウンドの出力先の設定を無効にします。
    //!
    //!         ハンドルに関連付けられたサウンドの出力先の設定を無効にします。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数は @ref SetOutputLine で設定された出力先を無効にして、
    //!         デフォルトの出力先へ戻します。デフォルトの出力先は、
    //!         @ref SoundPlayer::SetDefaultOutputLine で設定された値です。
    //!
    //! @see SetOutputLine
    //! @see SoundPlayer::SetDefaultOutputLine
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void ResetOutputLine()
    {
        if ( IsAttachedSound() )
        {
            m_pSound->ResetOutputLine();
        }
    }
    //@}

    //----------------------------------------
    //! @name 出力先別パラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  出力先別の音量を設定します。
    //!
    //!         @ref SetVolume などで設定された値との積が、当該出力先の音量になります。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] volume   音量です。
    //!
    //! @see SetVolume
    //!
    //! @date 2012/04/24 初版
    // --------------------------------------------------------------------------
    void SetOutputVolume(OutputDevice device, f32 volume)
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputVolume(device, volume);
        }
    }

    // --------------------------------------------------------------------------
    //! @brief  出力先別のパンを設定します。
    //!
    //!         @ref SetPan で設定された値との和が、当該出力先のパンになります。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] pan      パンです。
    //!
    //! @see SetPan
    //!
    //! @date 2012/04/24 初版
    // --------------------------------------------------------------------------
    void SetOutputPan(OutputDevice device, f32 pan)
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputPan(device, pan);
        }
    }

    // --------------------------------------------------------------------------
    //! @brief  出力先別のサラウンドパンを設定します。
    //!
    //!         @ref SetSurroundPan で設定された値との和が、
    //!         当該出力先のサラウンドパンになります。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] span     サラウンドパンです。
    //!
    //! @see SetSurroundPan
    //!
    //! @date 2012/04/24 初版
    // --------------------------------------------------------------------------
    void SetOutputSurroundPan(OutputDevice device, f32 span)
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputSurroundPan(device, span);
        }
    }

    // --------------------------------------------------------------------------
    //! @brief  出力先別のメインセンドを設定します。
    //!
    //!         @ref SetMainSend で設定された値との和が、
    //!         当該出力先のメインセンド量になります。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] send     センド量です。
    //!
    //! @see SetMainSend
    //!
    //! @date 2012/04/24 初版
    // --------------------------------------------------------------------------
    void SetOutputMainSend(OutputDevice device, f32 send)
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputMainSend(device, send);
        }
    }

    // --------------------------------------------------------------------------
    //! @brief  出力先別のエフェクトセンドを設定します。
    //!
    //!         @ref SetFxSend で設定された値との和が、
    //!         当該出力先のエフェクトセンド量になります。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] bus      AUX バスです。
    //! @param[in] send     センド量です。
    //!
    //! @see SetFxSend
    //!
    //! @date 2012/04/24 初版
    // --------------------------------------------------------------------------
    void SetOutputFxSend(OutputDevice device, AuxBus bus, f32 send)
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputFxSend(device, bus, send);
        }
    }
    //@}

    //----------------------------------------
    //! @name メイン (TV) 出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  メイン (TV) 出力に対する音量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのメイン出力音量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのメイン出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         メイン出力音量は、メイン出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!
    //!         他のメイン出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //!         本関数内部で SetOutputVolume(OUTPUT_DEVICE_MAIN, volume); を呼び出しています。
    //!         この関数は過去のバージョンとの互換性のために残されています。
    //!         今後は @ref SetOutputVolume をお使いください。
    //!         次回以降のリリースで deprecated になる予定です。
    //!
    //! @param[in]    volume  変更する音量の倍率です (0.0 ～) 。
    //!
    //! @see SetOutputLine
    //! @see SetDrcOutVolume
    //! @see SetRemoteOutVolume
    //! @see SetOutputVolume
    //!
    //! @date 2012/04/24 今後 SetOutputVolume に置き換わる旨、追記
    //! @date 2011/10/31 参照に @ref SetOutputLine, @ref SetDrcOutVolume,
    //!                  @ref SetRemoteOutVolume を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetMainOutVolume( f32 volume ) )
    {
        SetOutputVolume(OUTPUT_DEVICE_MAIN, volume);
    }

    //@}

    //----------------------------------------
    //! @name DRC 出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  DRC 出力に対する音量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの、
    //!         指定した番号の DRC 出力に対する音量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどの DRC 出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         DRC 出力音量は、
    //!         指定した番号の DRC 出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!
    //!         他の DRC 出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //!         本関数内部で SetOutputVolume(OUTPUT_DEVICE_DRC, volume); を呼び出しています。
    //!         この関数は過去のバージョンとの互換性のために残されています。
    //!         今後は @ref SetOutputVolume をお使いください。
    //!         次回以降のリリースで deprecated になる予定です。
    //!
    //! @param[in] volume       変更する音量の倍率です (0.0～)。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetOutputLine
    //! @see SetMainOutVolume
    //! @see SetRemoteOutVolume
    //! @see SetOutputVolume
    //!
    //! @date 2012/04/24 今後 SetOutputVolume に置き換わる旨、追記
    //! @date 2011/10/31 参照に @ref SetOutputLine, @ref SetMainOutVolume,
    //!                  @ref SetRemoteOutVolume を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetDrcOutVolume( f32 volume, u32 drcIndex=0 ) )
    {
        (void)drcIndex;
        SetOutputVolume(OUTPUT_DEVICE_DRC, volume);
    }

    //---------------------------------------------------------------------------
    //! @brief  DRC 出力に対するパン (左右の定位) を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの DRC 出力に対するパンを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         pan は、定位の相対変化の値を設定します。
    //!         0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!         1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!         -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!
    //!         デフォルト値は 0.0 です。
    //!
    //!         本関数内部で SetOutputPan(OUTPUT_DEVICE_DRC, pan); を呼び出しています。
    //!         この関数は過去のバージョンとの互換性のために残されています。
    //!         今後は @ref SetOutputPan をお使いください。
    //!         次回以降のリリースで deprecated になる予定です。
    //!
    //! @param[in]  pan  0.0 を基準としたパンの相対変化の値です。
    //! @param[in]  drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetPan
    //! @see SetOutputPan
    //!
    //! @date 2012/04/24 今後 SetOutputPan に置き換わる旨、追記
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetDrcPan( f32 pan, u32 drcIndex=0 ) )
    {
        (void)drcIndex;
        SetOutputPan(OUTPUT_DEVICE_DRC, pan);
    }
    //@}

    //----------------------------------------
    //! @name リモコン出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  サウンドのリモコン出力音量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの、
    //!         指定したチャンネルのリモコン出力に対する音量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのリモコン出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         リモコン出力音量は、
    //!         指定したチャンネルのリモコン出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!
    //!         他のリモコン出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] remoteIndex  リモコンのチャンネル番号です。
    //! @param[in] volume       変更する音量の倍率です (0.0～)。
    //!
    //! @date 2011/10/27 remoteIndex の型を変更 (int→u32)
    //! @date 2011/08/19 初版
    //---------------------------------------------------------------------------
    void SetRemoteOutVolume( u32 remoteIndex, f32 volume )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetRemoteOutVolume( remoteIndex, volume );
        }
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドのリモコン出力に対するメインセンド量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの、
    //!         指定したチャンネルのリモコン出力に対するメインセンド量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのリモコン出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!         メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] remoteIndex  リモコンのチャンネル番号です。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    void SetRemoteMainSend( u32 remoteIndex, f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetRemoteMainSend( remoteIndex, send );
        }
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドのリモコン出力に対するメインセンド量を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの、
    //!         指定したチャンネルのリモコン出力に対するメインセンド量を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのリモコン出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!         最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] remoteIndex  リモコンのチャンネル番号です。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    void SetRemoteFxSend( u32 remoteIndex, f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetRemoteFxSend( remoteIndex, send );
        }
    }
    //@}

    //----------------------------------------
    //! @name ハンドル操作
    //@{
    //---------------------------------------------------------------------------
    //! @brief    ハンドルにサウンドが関連付けられているかどうかを調べます。
    //!
    //! @return   ハンドルに関連付けられているサウンドがあれば true を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsAttachedSound() const
    {
        if ( m_pSound != NULL )
        {
            return true;
        }
        return false;
    }

    // サウンドハンドルからサウンドを解放
    //---------------------------------------------------------------------------
    //! @brief    ハンドルからサウンドを解放します。
    //!
    //!           ハンドルから解放されたサウンドは、
    //!           その後ハンドルを通して操作できないようになります。
    //!
    //! @date 2011/07/13 誤字修正 (開放→解放)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void DetachSound();

    //---------------------------------------------------------------------------
    //! @brief    サウンドの ID を設定します。
    //!
    //!           ハンドルに関連付けられたサウンドの ID を設定します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           @ref SoundArchivePlayer でサウンドを再生すると、
    //!           再生開始時にサウンド ID が自動的に設定されます。
    //!           この関数を呼び出すと、 ID を上書きして変更します。
    //!
    //!           設定した ID を取得するためには @ref GetId を呼び出します。
    //!
    //! @param[in]    id      サウンドを識別する ID です。
    //!
    //! @see      @ref SoundArchivePlayer, @ref GetId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetId( u32 id )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetId( id );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドに設定されている ID を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドに設定されている ID を取得します。
    //!           ハンドルが無効の場合は 0xffffffff を返します。
    //!
    //!           この関数で取得できる ID は @ref SetId で設定された ID です。
    //!
    //! @return   SetId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetId() const
    {
        if ( IsAttachedSound() )
        {
            return m_pSound->GetId();
        }
        return internal::BasicSound::INVALID_ID;
    }

    //---------------------------------------------------------------------------
    //! @brief    アンビエントパラメータの値を取得します。
    //!
    //!           3D サウンドを利用している際は、
    //!           3D サウンドで計算された結果がアンビエントパラメータに格納されますので、
    //!           アンビエントパラメータを取得して 3D
    //!           サウンドによるサウンドパラメータの値を知ることができます。
    //!
    //! @return   現在のアンビエントパラメータの値を返します。
    //!           ハンドルが無効の場合は NULL を返します。
    //!
    //! @see      nw::snd::SoundParam 構造体
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const SoundParam* GetAmbientParam() const
    {
        if ( ! IsAttachedSound() )
        {
            return NULL;
        }
        return &m_pSound->GetAmbientParam();
    }

    //---------------------------------------------------------------------------
    //! @brief  ユーザーパラメータを取得します。
    //!
    //!         各サウンドインスタンスにユーザーパラメータ
    //!         (ユーザーが自由に読み書きできる領域) を持たせることができます。
    //!
    //!         SoundArchivePlayer::Initialize の userParamSizePerSound に 1
    //!         以上の値を渡していると、ユーザーパラメータを void* 型で取得できます。
    //!         0 を渡すと NULL が返ります。
    //!         つまり、0 を渡した場合はユーザーパラメータは利用できません。
    //!
    //!         取得した値は、アプリケーションプログラム側で、
    //!         適当な型にキャストするなどして利用してください。
    //!
    //!         NULL 以外の値が取得できた場合、そのアドレスには、
    //!         @ref SoundArchivePlayer::GetSoundUserParamSize
    //!         バイトのデータが保持されています。
    //!
    //!         ユーザーパラメータはサウンド再生開始時にゼロクリアされます。
    //!
    //! @return ユーザーパラメータを返します。
    //!
    //! @see SoundArchivePlayer::Initialize
    //! @see SoundArchivePlayer::GetUserParam
    //!
    //! @date 2012/07/24 初版
    //---------------------------------------------------------------------------
    void* GetUserParam() const
    {
        if ( ! IsAttachedSound() )
        {
            return NULL;
        }
        return m_pSound->GetUserParam();
    }
    //@}

    // -----------------------------------------------------------------
    // 非公開関数

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドのサラウンド振り分け方法を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンド振り分け方法を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         mixMode には MIX_MODE_PAN か MIX_MODE_MIXPARAMETER を設定します。
    //!         MIX_MODE_PAN を指定した場合は、パンとサラウンドパンの値を設定する事で
    //!         サラウンドの振り分けを行います。
    //!         @ref SetChannelMixParameter でサラウンドボリュームを設定していても無視されます。
    //!
    //!         MIX_MODE_MIXPARAMETER を指定した場合は、
    //!         サラウンドボリューム(5.1chの各スピーカーの音量)を指定する必要があります。
    //!         パンやサラウンドパンに値を設定していても無視されます。
    //!
    //! @param[in] mixMode     サラウンド振り分け方法の種類です。
    //!
    //! @see MixMode
    //! @see SetPan
    //! @see SetSurroundPan
    //! @see SetChannelMixParameter
    //! @see SetOutputChannelMixParameter
    //---------------------------------------------------------------------------
    void SetMixMode( MixMode mixMode )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetMixMode( mixMode );
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         この関数でサラウンドボリュームを指定すると、TV, DRC の両方の出力に適用されます。
    //!         出力先別に設定したい場合は、@ref SetOutputChannelMixParameter を利用してください。
    //!
    //! @param[in]    srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in]    param   サラウンドボリュームです。
    //!
    //! @see SetOutputChannelMixParameter
    //---------------------------------------------------------------------------
    void SetChannelMixParameter( u32 srcChNo, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    波形の全チャンネルに、同一のサラウンドボリュームを設定します。
    //! @param param :private
    void SetChannelMixParameter( const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            // 全チャンネルで同じ設定を使用
            for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++ )
            {
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
            }
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を出力先別で変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの
    //!         メイン (TV) もしくは DRC 出力に対するサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         TV, DRC の両方の出力にサラウンドボリュームを設定したい場合、@ref SetChannelMixParameter を利用してください。
    //!
    //! @param[in] device  出力先です。
    //! @param[in] srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in] param   サラウンドボリュームです。
    //!
    //! @see SetChannelMixParameter
    //---------------------------------------------------------------------------
    void SetOutputChannelMixParameter( OutputDevice device, u32 srcChNo, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    出力先指定で、波形の全チャンネルに同一のサラウンドボリュームを設定します。
    //! @param device :private
    //! @param param :private
    void SetOutputChannelMixParameter( OutputDevice device, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            // 全チャンネルで同じ設定を使用
            for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++)
            {
                m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
            }
        }
    }

    // ハンドルをサウンドに関連付ける
    //! @briefprivate
    //! @param sound :private
    void detail_AttachSound( internal::BasicSound* sound );
    //! @briefprivate
    //! @param sound :private
    void detail_AttachSoundAsTempHandle( internal::BasicSound* sound );

    //! @briefprivate
    //! @return :private
    internal::BasicSound* detail_GetAttachedSound()
    {
        return m_pSound;
    }

    //! @briefprivate
    //! @return :private
    const internal::BasicSound* detail_GetAttachedSound() const
    {
        return m_pSound;
    }

    // 一時ハンドルを生成する
    //! @briefprivate
    //! @param handle :private
    void detail_DuplicateHandle( SoundHandle* handle );

private:
    NW_DISALLOW_COPY_AND_ASSIGN( SoundHandle );

    internal::BasicSound* m_pSound;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_HANDLE_H_ */

