﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundDataManager.h
 *
 * @file snd_SoundDataManager.h
 */

#ifndef NW_SND_SOUND_DATA_MANAGER_H_
#define NW_SND_SOUND_DATA_MANAGER_H_

#include <nw/snd/snd_DisposeCallback.h>
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_SoundArchiveLoader.h>
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {

class SoundMemoryAllocatable;

namespace internal
{

//! @briefprivate
class SoundFileManager
{
public:
    virtual ~SoundFileManager() {}

public:
    virtual const void* GetFileAddress( SoundArchive::FileId fileId ) const = 0;
    virtual const void* GetFileWaveDataAddress( SoundArchive::FileId fileId ) const = 0;
};

} // namespace nw::snd::internal

//---------------------------------------------------------------------------
//! @brief    サウンドアーカイブのデータをロードするためのクラスです。
//!
//! @see SoundArchivePlayer クラス
//!
//! :baseclass nw::snd::internal::driver::DisposeCallback
//! :baseclass nw::snd::internal::SoundArchiveLoader
//!
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundDataManager : public internal::driver::DisposeCallback,
                         public internal::SoundArchiveLoader
{
public:

    //! @name コンストラクタ / デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundDataManager();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~SoundDataManager();
    //@}

    //----------------------------------------
    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief    初期化に必要なメモリのサイズを取得します。
    //!
    //! @param[in]    arc     このサウンドデータマネージャを登録するプレイヤーで
    //!                       使用するサウンドアーカイブです。
    //!
    //! @return   初期化に必要なメモリのサイズを返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    size_t GetRequiredMemSize( const SoundArchive* arc ) const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドデータマネージャの初期化を行います。
    //!
    //!           サウンドデータマネージャを使用する前に初期化を行う必要があります。
    //!           初期化を行うことにより、
    //!           サウンドデータマネージャがサウンドアーカイブと関連付けられ、
    //!           サウンドアーカイブプレイヤーに登録することができます。
    //!
    //!           サウンドデータマネージャが必要とするメモリのサイズは
    //!           @ref GetRequiredMemSize で取得することができます。
    //!
    //! @param[in]    arc     このサウンドデータマネージャを登録するプレイヤーで
    //!                       使用するサウンドアーカイブです。
    //! @param[out]   buffer  バッファへのポインタです。
    //!                       4 バイト境界に配置されている必要があります。
    //! @param[in]    size    バッファサイズです。
    //!
    //! @return   初期化に成功したら true を、失敗したら false を返します。
    //!
    //! @see GetRequiredMemSize
    //! @see Finalize
    //!
    //! @date 2011/09/09 buffer のアラインメントの記述を修正 (128バイト→4バイト)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool Initialize( const SoundArchive* arc, void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief    サウンドデータマネージャを破棄します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Finalize();
    //@}

    //! @briefprivate
    //! @param fileManager :private
    void detail_SetFileManager( internal::SoundFileManager* fileManager )
    {
        m_pFileManager = fileManager;
    }

    //----------------------------------------
    //! @name ロード
    //@{
    //---------------------------------------------------------------------------
    //! @brief  独自にロードしたデータを、ファイル管理テーブルに登録します。
    //!
    //!         あらかじめ別のアドレスが登録されている場合は、
    //!         上書きされます。
    //!
    //!         引数として渡すファイル ID は、
    //!         @ref SoundArchive::GetItemFileId 関数で、
    //!         アイテム ID から変換することができます。
    //!
    //! @param[in] fileId   セットするデータのファイル ID です。
    //! @param[in] address  関連するデータを確認するかを示すフラグです。
    //!
    //! @return   あらかじめ登録されていたアドレスを返します。
    //!
    //! @see SoundArchive::GetItemFileId
    //!
    //! @date 2012/05/22 存在しないデモへの言及を削除
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const void* SetFileAddress( SoundArchive::FileId fileId, const void* address );

    //---------------------------------------------------------------------------
    //! @brief  独自にロードしたグループファイルに含まれるファイルを、
    //!         ファイル管理テーブルに登録します。
    //!
    //!         グループファイル (*.bfgrp) をサウンドアーカイブ (*.bfsar)
    //!         とは別に出力するには、SoundMaker 上のグループリストで
    //!         出力方式を [ユーザー管理] にします。
    //!
    //!         あらかじめ別のアドレスが登録されている場合は、上書きされます。
    //!
    //!         不正なグループファイルを渡すと false が返ります。
    //!
    //!         詳しくは、Demo/snd/loadGroupManually デモおよび、
    //!         当該デモのドキュメントをご覧ください。
    //!
    //!         ■ LoadData との比較
    //!
    //!         これまで提供していた @ref SoundDataManager::LoadData によるデータロードと比べて、
    //!         次のようなケースでメリットがあります。
    //!
    //!         - 「クリボー」に関するサウンドデータをひとつのグループに登録し、
    //!           そのグループの出力方式を [ユーザー管理] にします。
    //!           (グループの名前が GROUP_KURIBO だとすると、GROUP_KURIBO.bfgrp
    //!            ファイルが出力されます)
    //!
    //!         - 「クリボー」のグラフィックスデータ (モデルやテクスチャなど) と
    //!           GROUP_KURIBO.bfgrp ファイルをアーカイブ化します。
    //!           (SoundMaker は汎用アーカイバを提供していませんので、
    //!            別途ユーザー側でご用意ください)
    //!
    //!         - 「クリボー」が出現するステージで、このアーカイブをロードする
    //!
    //!         こうしておけば、クリボーが出現するステージでは、
    //!         クリボーに関するサウンドデータもロード済みであることが保証でき、
    //!         「ロードを忘れて音が鳴らない」といった状況を避ける事ができます。
    //!
    //!         @ref LoadData を使う場合、あるステージにクリボーが出現するかどうか、
    //!         なんらかの手段で判断した上で、クリボーのサウンドデータをロードする必要がありました。
    //!
    //!         端的には、各ステージのオブジェクト配置変更に強い、と言えます。
    //!
    //!         ただし、LoadData の時と比べ、本関数や @ref ClearFileAddressInGroupFile
    //!         を呼び出す必要がありますので、その点ご注意ください。
    //!
    //!         また、プレイヤーヒープを用いるサウンドが、
    //!         [ユーザー管理] のグループにのみ属している場合は、
    //!         プレイヤーヒープを使ったサウンドの再生ができませんので、ご注意ください。
    //!         (グループファイルをロードし、本関数で SoundDataManager
    //!         に登録すると再生することが可能です。プレイヤーヒープは使われません。)
    //!
    //!         プレイヤーヒープを使ってサウンドを再生したい場合は、
    //!         該当サウンドデータが下記のどちらかになるようにグループ構成を調整してください。
    //!
    //!         - [ユーザー管理] グループに含めない
    //!         - [ユーザー管理] グループに含めるが、[ユーザー管理]
    //!           でないグループ ([リンク] や [埋め込み]) にも含める
    //!
    //!
    //! @return 登録に成功すると true、不正なグループファイルを渡すと false を返します。
    //!
    //! @param[in] groupFile    ロード済みグループファイルのアドレスです。
    //! @param[in] size         上記ファイルのサイズです。
    //!
    //! @see ClearFileAddressInGroupFile
    //!
    //! @date 2012/05/22 初版
    //---------------------------------------------------------------------------
    bool SetFileAddressInGroupFile( const void* groupFile, size_t size );

    //---------------------------------------------------------------------------
    //! @brief  @ref SetFileAddressInGroupFile で指定したファイルを登録解除します。
    //!
    //!         サウンドスレッドをブロックする点にご注意ください。
    //!
    //!         本関数は、@ref SetFileAddressInGroupFile
    //!         で指定したグループファイルをメモリ上から解放する前に、
    //!         必ず呼び出す必要があります。
    //!
    //!         連続した領域にグループファイルを複数ロードしているケースで、
    //!         その複数グループファイルをメモリ上から解放したい場合は、
    //!         本関数を 1 回にまとめて呼び出すことが可能です。
    //!         その場合、groupFile に先頭アドレスを、size
    //!         に先頭アドレス～終端アドレスの差を渡します。
    //!
    //! @param[in] groupFile    登録解除するグループファイルのアドレスです。
    //! @param[in] size         上記ファイルのサイズです。
    //!
    //! @see SetFileAddressInGroupFile
    //!
    //! @date 2012/05/22 初版
    //---------------------------------------------------------------------------
    void ClearFileAddressInGroupFile( const void* groupFile, size_t size );
    //@}

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    const void* detail_GetFileAddress( SoundArchive::FileId fileId ) const;

    //! @briefprivate
    //! @param address :private
    //! @return :private
    u32 detail_GetFileIdFromTable( const void* address ) const;

    //---------------------------------------------------------------------------
    //! @brief  独自にロードしたデータを、ファイル管理テーブルから破棄します。
    //!
    //!         サウンドスレッドをブロックする点にご注意ください。
    //!
    //!         本関数は、独自にロードしたデータをメモリ上から解放する前に、
    //!         必ず呼び出す必要があります。
    //!
    //!         連続した領域に独自にロードしたデータを複数ロードしているケースで、
    //!         そのデータをメモリ上から解放したい場合は、
    //!         本関数を 1 回にまとめて呼び出すことが可能です。
    //!         その場合にはその連続領域の先頭アドレスを渡し、size
    //!         には先頭アドレス～終端アドレスの差を渡します。
    //!
    //! @param[in] address      破棄するデータの先頭アドレスです。
    //! @param[in] size         上記データのサイズです。
    //!
    //! @see SetFileAddress
    //!
    //! @date 2014/05/21 初版
    //---------------------------------------------------------------------------
    void InvalidateSoundData( const void* address, size_t size );

protected:
    //! @briefprivate
    //! @param start :private
    //! @param end :private
    virtual void InvalidateData( const void* start, const void* end );

    //! @briefprivate
    //! @param fileId :private
    //! @param address :private
    //! @return :private
    virtual const void* SetFileAddressToTable(
            SoundArchive::FileId fileId, const void* address );
    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* GetFileAddressFromTable( SoundArchive::FileId fileId ) const;

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* GetFileAddressImpl( SoundArchive::FileId fileId ) const;
        // テーブルを含め、アクセス可能な箇所にデータが読み込まれているかを確認する

private:
    //! @briefprivate
    struct FileAddress
    {
        const void* address;
    };

    //! @briefprivate
    typedef internal::Util::Table<FileAddress> FileTable;

    //! @briefprivate
    bool CreateFileAddressTable( const SoundArchive* arc, void** buffer, void* endp );

    //! @briefprivate
    FileTable* m_pFileTable;

    //! @briefprivate
    internal::SoundFileManager* m_pFileManager;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_DATA_MANAGER_H_ */

