﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_ARCHIVE_FILES_HOOK_H_
#define NW_SND_SOUND_ARCHIVE_FILES_HOOK_H_

#include <nw/types.h>

namespace nw {

namespace ut {

class FileStream;

}

namespace snd {
namespace internal {

//! @brief  サウンドアーカイブに関連するファイルアクセスをフックします。
class SoundArchiveFilesHook
{
public:
    static const char ITEM_TYPE_STREAM_SOUND[];
    static const char ITEM_TYPE_WAVE_SOUND[];
    static const char ITEM_TYPE_SEQUENCE_SOUND[];
    static const char FILE_TYPE_STREAM_BINARY[];
    static const char FILE_TYPE_WAVE_SOUND_BINARY[];
    static const char FILE_TYPE_SEQUENCE_BINARY[];
    static const char FILE_TYPE_BANK_BINARY[];
    static const char FILE_TYPE_WAVE_ARCHIVE_BINARY[];
    static const char FILE_TYPE_STREAM_PREFETCH_BINARY[];

protected:
    SoundArchiveFilesHook() : m_IsEnabled(true) { }
    virtual ~SoundArchiveFilesHook() { }

public:
    //---------------------------------------------------------------------------
    //! @brief         フックが有効かどうかを調べます。
    //!
    //! @return        有効な場合は true、無効な場合は false を返します。
    //---------------------------------------------------------------------------
    bool GetIsEnable() const { return m_IsEnabled; }

    //---------------------------------------------------------------------------
    //! @brief         フックの有効/無効を設定します。
    //!
    //! @param[in]     value     有効性を指定します。
    //---------------------------------------------------------------------------
    void SetIsEnable(bool value) { m_IsEnabled = value; }

    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルがフック対象かどうかを調べます。
    //!
    //! @param[in]     itemLabel     アイテムラベルを指定します。
    //!
    //! @return        対象の場合は true、対象でない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsTargetItem(const char* itemLabel)
    {
        return m_IsEnabled ? IsTargetItemImpl(itemLabel) : false;
    }

    //---------------------------------------------------------------------------
    //! @brief         フックの有無が変化しないようにロックします。
    //---------------------------------------------------------------------------
    void Lock() { LockImpl(); }

    //---------------------------------------------------------------------------
    //! @brief         フックのロックを解除します。
    //---------------------------------------------------------------------------
    void Unlock() { UnlockImpl(); }

    //---------------------------------------------------------------------------
    //! @brief         指定バッファを使って指定アイテムラベルに関連するファイルを開きます。
    //!
    //! @param[in]     buffer            ファイルを開くのに利用するバッファを指定します。
    //! @param[in]     bufferLength      buffer の長さを指定します。
    //! @param[in]     cacheBuffer       ファイルロード用のキャッシュバッファを指定します。
    //! @param[in]     cacheBufferLength cacheBuffer の長さを指定します。
    //! @param[in]     itemLabel         アイテムラベルを指定します。
    //! @param[in]     fileType          取得するファイルタイプを指定します。
    //!
    //! @return        ut::FileStream を返します。
    //---------------------------------------------------------------------------
    ut::FileStream* OpenFile(void* buffer, u32 bufferLength, void* cacheBuffer, u32 cacheBufferLength, const char* itemLabel, const char* fileType)
    {
        return m_IsEnabled ? OpenFileImpl(buffer, bufferLength, cacheBuffer, cacheBufferLength, itemLabel, fileType) : NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルに関連するファイルデータへのアドレスを取得します。
    //!
    //! @param[in]     itemLabel    アイテムラベルを指定します。
    //! @param[in]     itemType     アイテム種別を指定します。
    //! @param[in]     fileType     取得するファイルタイプを指定します。
    //! @param[in]     fileIndex    取得するファイルのインデックスがあれば指定します。
    //!
    //! @return        ファイルデータへのアドレスを返します。
    //---------------------------------------------------------------------------
    const void* GetFileAddress(const char* itemLabel, const char* itemType, const char* fileType, u32 fileIndex = 0)
    {
        return m_IsEnabled ? GetFileAddressImpl(itemLabel, itemType, fileType, fileIndex) : NULL;
    }

protected:
    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルがフック対象かどうかを調べます。
    //!
    //! @param[in]     itemLabel     アイテムラベルを指定します。
    //!
    //! @return        対象の場合は true、対象でない場合は false を返します。
    //---------------------------------------------------------------------------
    virtual bool IsTargetItemImpl(const char* itemLabel)
    {
        (void)itemLabel;
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief         フックの有無が変化しないようにロックします。
    //---------------------------------------------------------------------------
    virtual void LockImpl() { }

    //---------------------------------------------------------------------------
    //! @brief         フックのロックを解除します。
    //---------------------------------------------------------------------------
    virtual void UnlockImpl() { }

    //---------------------------------------------------------------------------
    //! @brief         指定バッファを使って指定アイテムラベルに関連するファイルを開きます。
    //!
    //! @param[in]     buffer            ファイルを開くのに利用するバッファを指定します。
    //! @param[in]     bufferLength      buffer の長さを指定します。
    //! @param[in]     cacheBuffer       ファイルロード用のキャッシュバッファを指定します。
    //! @param[in]     cacheBufferLength cacheBuffer の長さを指定します。
    //! @param[in]     itemLabel         アイテムラベルを指定します。
    //! @param[in]     fileType          取得するファイルタイプを指定します。
    //!
    //! @return        ut::FileStream を返します。
    //---------------------------------------------------------------------------
    virtual ut::FileStream* OpenFileImpl(void* buffer, u32 bufferLength, void* cacheBuffer, u32 cacheBufferLength, const char* itemLabel, const char* fileType)
    {
        (void)buffer;
        (void)bufferLength;
        (void)cacheBuffer;
        (void)cacheBufferLength;
        (void)itemLabel;
        (void)fileType;
        return NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルに関連するファイルデータへのアドレスを取得します。
    //!
    //! @param[in]     itemLabel    アイテムラベルを指定します。
    //! @param[in]     itemType     アイテム種別を指定します。
    //! @param[in]     fileType     取得するファイルタイプを指定します。
    //! @param[in]     fileIndex    取得するファイルのインデックスがあれば指定します。
    //!
    //! @return        ファイルデータへのアドレスを返します。
    //---------------------------------------------------------------------------
    virtual const void* GetFileAddressImpl(const char* itemLabel, const char* itemType, const char* fileType, u32 fileIndex = 0)
    {
        (void)itemLabel;
        (void)itemType;
        (void)fileType;
        (void)fileIndex;
        return NULL;
    }

private:
    bool m_IsEnabled;
    u8   m_Padding[3];
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_SOUND_ARCHIVE_FILES_HOOK_H_ */

