﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundActor.h
 *
 * @file snd_SoundActor.h
 */

#ifndef NW_SND_SOUND_ACTOR_H_
#define NW_SND_SOUND_ACTOR_H_

#include <nw/snd/snd_SoundStartable.h>
#include <nw/snd/snd_ExternalSoundPlayer.h>
#include <nw/snd/snd_BasicSound.h>          // SoundActorParam

namespace nw {
namespace snd {

class SoundHandle;
class SoundArchivePlayer;

//---------------------------------------------------------------------------
//! @brief    複数のサウンドをまとめて再生管理するためのクラスです。
//!
//!           サウンドアクターは、
//!           ゲーム内でキャラクターやモデル等に関連付けて使用されることが想定されています。
//!           例えばキャラクタの移動にあわせて、
//!           そのキャラクタに関連付けられたサウンドアクターの音の定位を変更する、
//!           といった使い方が可能です。
//!
//!           サウンドアクターに設定するパラメータは、
//!           そのサウンドアクターで再生するすべてのサウンドに対して効果があります。
//!
//!           サウンドアクターは内部に 4 つのアクタープレイヤーを持っています。
//!           このアクタープレイヤーごとに同時に再生できるサウンド数を制限することができます。
//!           設定したサウンド数を超えた場合には、
//!           各サウンドに設定されたプレイヤープライオリティを比較して、
//!           もっとも優先度が低いサウンドが停止します。
//!           同時に再生可能なサウンド数の初期値は下記のように設定されています。
//!
//!             - アクタープレイヤー 0 番は無制限
//!             - その他は 1
//!
//!           サウンドがどのアクタープレイヤーで再生されるかは、
//!           サウンドデータに設定されています。
//!           この情報は再生時に @ref nw::snd::SoundStartable::StartInfo を用いて、
//!           上書きすることができます。
//!
//!         @n
//!
//!         ■ 出力先別パラメータ
//!
//!         Cafe では、[1] メイン出力 (TV 出力) [2] DRC 出力 [3] リモコン出力の
//!         3 系統の出力があります。
//!         SoundActor では、この SoundActor 経由で再生するサウンドに関して、
//!         下記の影響を与えることができます。
//!
//!         (全体)
//!         - ピッチ ... @ref SetPitch
//!
//!         (TV 出力向け)
//!         - 音量 ... @ref SetVolume, @ref StopAllSound, @ref PauseAllSound, @ref SetMainOutVolume
//!         - パン ... @ref SetPan
//!
//!         (DRC 出力向け)
//!         - 音量 ... @ref SetVolume, @ref StopAllSound, @ref PauseAllSound, @ref SetDrcOutVolume
//!         - パン ... @ref SetDrcPan
//!
//!         @n
//!
//!         各パラメータは、上記の影響の他に、下記の影響を受けます。
//!         - SoundMaker で指定したデータの影響
//!         - @ref SoundHandle や @ref SoundPlayer による影響
//!         - 3D サウンド計算 (@ref Sound3DEngine) による影響
//!
//! @see SoundStartable クラス
//!
//! @date 2011/11/25 出力先別パラメータの説明を追加
//! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundActor : public SoundStartable
{
    typedef internal::ExternalSoundPlayer ActorPlayer;

public:
    //---------------------------------------------------------------------------
    //! @brief    アクタープレイヤーの保持数です。
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static const int ACTOR_PLAYER_COUNT = 4;

    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           初期化時には、プレイヤーの音量が 1.0、ピッチが 1.0、パンが 0.0、
    //!           同時に再生可能なサウンド数が 1 に設定されます。
    //!
    //!           引数付きのコンストラクタの場合は、
    //!           @ref Initialize を呼ぶ必要はありません
    //!           (呼んでも問題ありません)。
    //!
    //! @param[in] soundArchivePlayer 再生に使用するサウンドアーカイブプレイヤーです。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    explicit SoundActor( SoundArchivePlayer& soundArchivePlayer );

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           初期化時には、プレイヤーの音量が 1.0、ピッチが 1.0、パンが 0.0、
    //!           同時に再生可能なサウンド数が 1 に設定されます。
    //!
    //!           引数なしのコンストラクタは、@ref Initialize を呼ぶ必要があります。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundActor();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //!           内部で、@ref Finalize を呼んでいます。
    //!
    //! @see Finalize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~SoundActor();
    //@}

    //----------------------------------------
    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアクターを初期化します。
    //!
    //!           サウンドアクターを使用する前に、初期化を行う必要があります。
    //!           ただし、引数付きコンストラクタを呼んでいる場合は、
    //!           呼ぶ必要はありません (呼んでも問題ありません)。
    //!
    //!           Sound3DActor の初期化には、本関数ではなく、
    //!           @ref SoundArchivePlayer と @ref Sound3DManager を引数に取る、
    //!           @ref Sound3DActor::Initialize を呼ぶ必要があります。
    //!
    //!           同じ SoundActor インスタンスを再利用するには、
    //!           あらかじめ @ref Finalize 関数を呼んでから、
    //!           再度、本関数を呼び出してください。
    //!
    //! @param[in] soundArchivePlayer 再生に使用するサウンドアーカイブプレイヤーです。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see SoundActor()
    //! @see Finalize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Initialize( SoundArchivePlayer& soundArchivePlayer );

    //---------------------------------------------------------------------------
    //! @brief    サウンドアクターを破棄します。
    //!
    //!           デストラクタの中で呼ばれますが、
    //!           明示的に呼んでも問題ありません。
    //!
    //!           同じ SoundActor インスタンスを再利用するには、
    //!           本関数を呼んだ後、再度 @ref Initialize 関数を呼び出してください。
    //!
    //! @see ~SoundActor
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Finalize();
    //@}

    //----------------------------------------
    //! @name 再生
    //@{
public:
    //---------------------------------------------------------------------------
    //! @brief    アクターで再生中のすべてのサウンドを停止します。
    //!
    //!           各サウンドに対して、ハンドルクラスを通して停止させたときと同様の処理を行います。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトさせることができます。
    //!           0 を指定した場合は、フェードアウトを行いません。
    //!           ただし、シーケンスサウンドで発音中の音は、
    //!           エンベロープのリリースを発音しすべての減衰が完了した後にサウンドが停止します。
    //!
    //!           フェードアウトの音量制御は、フェードインと共有されます。
    //!           フェードアウトにかかるフレーム数は、
    //!           最大音量から音が消えるまでにかかる変化速度を表しますので、
    //!           フェードイン中にフェードアウトを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードアウトが完了する可能性があります。
    //!
    //! @param[in] fadeFrames   フェードアウトにかけるフレーム数です。
    //!                         フレーム数は @ref SoundArchivePlayer::Update
    //!                         の呼び出し回数で換算されます。
    //!                         fadeFrames には 0 以上の値を指定してください。
    //!                         負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void StopAllSound( int fadeFrames );

    //---------------------------------------------------------------------------
    //! @brief    アクターで再生中のすべてのサウンドを一時停止・再開します。
    //!
    //!           各サウンドに対して、ハンドルクラスを通して一時停止・
    //!           再開させたときと同様の処理を行います。
    //!
    //!           一時停止・再開時のフェードは、再生開始時のフェードイン、
    //!           停止時のフェードアウトとは独立してはたらきます。
    //!
    //! @param[in] flag   true なら一時停止、false なら再開します。
    //! @param[in] fadeFrames   フェードイン・フェードアウトにかけるフレーム数です。
    //!                         フレーム数は @ref SoundArchivePlayer::Update
    //!                         の呼び出し回数で換算されます。
    //!                         fadeFrames には 0 以上の値を指定してください。
    //!                         負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void PauseAllSound( bool flag, int fadeFrames );

protected:
    //---------------------------------------------------------------------------
    //! @brief  サウンドを再生する際に呼び出される仮想関数です。
    //!
    //!         この関数をオーバーライドすることにより、
    //!         サウンドを再生するときの処理をカスタマイズすることができます。
    //!
    //!         setupArg は @ref SoundActor::SetupSound を呼び出す際に、
    //!         そのまま引数として渡します。
    //!
    //!         例えば、継承したクラスで以下のように記述すると、
    //!         任意の ID のサウンドを再生することができます。
    //!
    //! @code
    //! SoundStartable::StartResult MyActor::SetupSound(
    //!     SoundHandle* handle,
    //!     u32 soundId,
    //!     const StartInfo* startInfo,
    //!     void* setupArg
    //! )
    //! {
    //!     u32 newId;
    //!     // ここでnewIdに別の音を設定する
    //!
    //!     // SetupSoundを呼び出して音を再生
    //!     return SoundActor::SetupSound(
    //!         handle,
    //!         newId,
    //!         startInfo,
    //!         setupArg // この引数はそのまま渡します
    //!     );
    //! }
    //! @endcode
    //!
    //! @param[in] handle     再生するサウンドと関連付けられるハンドルです。
    //! @param[in] soundId    再生するサウンドの ID です。
    //! @param[in] startInfo  詳細な再生パラメータです。
    //! @param[in] setupArg   セットアップ時に使用されるパラメータです。
    //!
    //! @return   再生処理の結果を @ref SoundStartable::StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see SoundStartable::StartResult
    //! @see SoundStartable::StartInfo
    //!
    //! @date 2011/11/25 サンプルコードを追加
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual SoundStartable::StartResult SetupSound(
        SoundHandle* handle,
        u32 soundId,
        const SoundStartable::StartInfo* startInfo,
        void* setupArg
    );
    //@}

public:
    //----------------------------------------
    //! @name サウンドパラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    アクターの音量を変更します。
    //!
    //!           アクターの音量は、アクターで再生するすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 から 1.0 の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //! @param[in] volume 変更する音量の倍率 ( 0.0 ～ 1.0 ) です。
    //!
    //! @see GetVolume
    //! @see SetMainOutVolume
    //! @see SetDrcOutVolume
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVolume( f32 volume ) { m_ActorParam.volume = volume; }

    //---------------------------------------------------------------------------
    //! @brief    アクターに設定されている音量を取得します。
    //!
    //! @return   現在の音量を返します。
    //!
    //! @see SetVolume
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetVolume() const { return m_ActorParam.volume; }

    //---------------------------------------------------------------------------
    //! @brief    アクターの音程を変更します。
    //!
    //!           アクターの音程は、アクターで再生するすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、他のどの音程パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音程 pitch は、周波数の比率で指定します。
    //!           すなわち、1.0 を指定すると音程に影響を与えません。
    //!           2.0 を指定すると再生される周波数が 2 倍になり、1 オクターブ高い音程になります。
    //!           0.5 を指定すると 1 オクターブ低い音程になります。
    //!           デフォルト値は 1.0 です。
    //!
    //! @param[in] pitch 変更する音程の周波数比率です。
    //!
    //! @see GetPitch
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPitch( f32 pitch ) { m_ActorParam.pitch = pitch; }

    //---------------------------------------------------------------------------
    //! @brief    アクターに設定されている音程を取得します。
    //!
    //! @return   現在の音程を返します。
    //!
    //! @see SetPitch
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetPitch() const { return m_ActorParam.pitch; }

    //---------------------------------------------------------------------------
    //! @brief  アクターのローパスフィルタカットオフ値を変更します。
    //!
    //!         アクターのローパスフィルタは、
    //!         アクターで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、他のどのローパスフィルタカットオフパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         lpfFreq は、カットオフの相対変化の値を指定します。
    //!         0.0 を指定するとカットオフの値を変更しません。
    //!         -1.0 を指定すると、フィルタがかかっていない状態から、
    //!         もっともフィルタがかかっている状態（カットオフ周波数が下がる方向）に変更します。
    //!
    //! @param[in] lpf    0.0 を基準としたフィルタカットオフの相対変化の値です。
    //!
    //! @see GetLpfFreq
    //!
    //! @date 2014/05/26 初版
    //---------------------------------------------------------------------------
    void SetLpfFreq( f32 lpf ) { m_ActorParam.lpf = lpf; }

    //---------------------------------------------------------------------------
    //! @brief    アクターに設定されているローパスフィルタカットオフ値を取得します。
    //!
    //! @return   現在のローパスフィルタカットオフ値を返します。
    //!
    //! @see SetLpfFreq
    //!
    //! @date 2014/05/26 初版
    //---------------------------------------------------------------------------
    f32 GetLpfFreq() const { return m_ActorParam.lpf; }
    //@}

    //----------------------------------------
    //! @name メイン (TV) 出力用サウンドパラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するアクターの音量を変更します。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 から 1.0 の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //! @param[in] volume 変更する音量の倍率 ( 0.0 ～ 1.0 ) です。
    //!
    //! @see GetMainOutVolume
    //! @see SetVolume
    //! @see SetDrcOutVolume
    //!
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetMainOutVolume( f32 volume ) ) { m_ActorParam.tvVolume = volume; }

    //---------------------------------------------------------------------------
    //! @brief  アクターに設定されているメイン (TV) 出力に対する音量を取得します。
    //!
    //! @return 現在のメイン (TV) 出力に対する音量を返します。
    //!
    //! @see SetMainOutVolume
    //!
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetMainOutVolume() const ) { return m_ActorParam.tvVolume; }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するアクターのパン (左右の定位) を変更します。
    //!
    //!           アクターのパンは、アクターで再生するすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] pan 0.0 を基準としたパンの相対変化の値です。
    //!
    //! @see GetPan
    //! @see SetDrcPan
    //!
    //! @date 2011/11/25 仕様変更にともない、メイン (TV) 出力に対するパンの設定であることを追記
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetPan( f32 pan ) ) { m_ActorParam.tvPan = pan; }

    //---------------------------------------------------------------------------
    //! @brief    アクターに設定されているメイン (TV) 出力に対するパンを取得します。
    //!
    //! @return   現在のメイン (TV) 出力に対するパンを返します。
    //!
    //! @see SetPan
    //!
    //! @date 2011/11/25 仕様変更にともない、メイン (TV) 出力に対するパンの設定であることを追記
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetPan() const ) { return m_ActorParam.tvPan; }
    //@}

    //----------------------------------------
    //! @name DRC 出力用サウンドパラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するアクターの音量を変更します。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 から 1.0 の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //! @param[in] volume 変更する音量の倍率 ( 0.0 ～ 1.0 ) です。
    //! @param[in] drcIndex    DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see GetDrcOutVolume
    //! @see SetVolume
    //! @see SetMainOutVolume
    //!
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetDrcOutVolume( f32 volume, u32 drcIndex=0 ) )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        m_ActorParam.drcVolume[drcIndex] = volume;
    }

    //---------------------------------------------------------------------------
    //! @brief  アクターに設定されている DRC 出力に対する音量を取得します。
    //!
    //! @return 現在の DRC 出力に対する音量を返します。
    //!
    //! @param[in] drcIndex    DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetMainOutVolume
    //!
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetDrcOutVolume( u32 drcIndex=0 ) const )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_ActorParam.drcVolume[drcIndex];
    }

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するアクターのパン (左右の定位) を変更します。
    //!
    //!           アクターのパンは、アクターで再生するすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] pan 0.0 を基準としたパンの相対変化の値です。
    //! @param[in] drcIndex    DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see GetDrcPan
    //! @see SetPan
    //!
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetDrcPan( f32 pan, u32 drcIndex=0 ) )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        m_ActorParam.drcPan[drcIndex] = pan;
    }

    //---------------------------------------------------------------------------
    //! @brief    アクターに設定されている DRC 出力に対するパンを取得します。
    //!
    //! @return   現在の DRC 出力に対するパンを返します。
    //!
    //! @param[in] drcIndex    DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetPan
    //!
    //! @date 2012/09/05 誤字修正 (DRCC → DRC)
    //! @date 2011/11/25 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetDrcPan( u32 drcIndex=0 ) const )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_ActorParam.drcPan[drcIndex];
    }
    //@}

    //----------------------------------------
    //! @name 出力先別サウンドパラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    指定した出力先に対するアクターの音量を変更します。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 から 1.0 の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //! @param[in] device 出力先のデバイスです。
    //! @param[in] volume 変更する音量の倍率 ( 0.0 ～ 1.0 ) です。
    //!
    //! @see GetOutputVolume
    //! @see SetVolume
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    void SetOutputVolume( OutputDevice device, f32 volume )
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch (device)
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            m_ActorParam.tvVolume = volume;
            break;
        case nw::snd::OUTPUT_DEVICE_DRC:
            m_ActorParam.drcVolume[0] = volume;
            break;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  指定した出力先の、アクターに設定されている音量を取得します。
    //!
    //! @param[in] device 出力先のデバイスです。
    //!
    //! @return 指定した出力先に対する音量を返します。
    //!         出力先のデバイスが存在しない場合、0.0f を返します。
    //!
    //! @see SetOutputVolume
    //! @see GetVolume
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    f32 GetOutputVolume( OutputDevice device ) const
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch (device)
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            return m_ActorParam.tvVolume;
        case nw::snd::OUTPUT_DEVICE_DRC:
            return m_ActorParam.drcVolume[0];
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
        return 0.0f;
    }

    //---------------------------------------------------------------------------
    //! @brief    指定した出力先に対するアクターのパン (左右の定位) を変更します。
    //!
    //!           アクターのパンは、アクターで再生するすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] device 出力先のデバイスです。
    //! @param[in] pan    0.0 を基準としたパンの相対変化の値です。
    //!
    //! @see GetOutputPan
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    void SetOutputPan( OutputDevice device, f32 pan )
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch (device)
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            m_ActorParam.tvPan = pan;
            break;
        case nw::snd::OUTPUT_DEVICE_DRC:
            m_ActorParam.drcPan[0] = pan;
            break;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    指定した出力先の、アクターに設定されているパンを取得します。
    //!
    //! @param[in] device 出力先のデバイスです。
    //!
    //! @return   指定した出力先に対する、現在のパンを返します。
    //!           出力先のデバイスが存在しない場合、0.0f を返します。
    //!
    //! @see SetOutputPan
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    f32 GetOutputPan( OutputDevice device ) const
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch (device)
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            return m_ActorParam.tvPan;
        case nw::snd::OUTPUT_DEVICE_DRC:
            return m_ActorParam.drcPan[0];
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
        return 0.0f;
    }
    //@}

    //----------------------------------------
    //! @name サウンド数
    //@{
    //---------------------------------------------------------------------------
    //! @brief    アクタープレイヤーで現在再生中のサウンドの個数を取得します。
    //!
    //! @param[in] actorPlayerId アクタープレイヤー番号です。
    //!
    //! @return   アクタープレイヤーで再生中のサウンド数を返します。
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetPlayingSoundCount( int actorPlayerId ) const;

    //---------------------------------------------------------------------------
    //! @brief    同時に再生可能なサウンド数を設定します。
    //!
    //!           サウンドアクターは、アクター内に 4 個のアクタープレイヤーを持っています。
    //!           このアクタープレイヤーごとに、同時に再生可能なサウンド数制限を設定することができます。
    //!           この設定は、サウンドプレイヤーで設定されている同時再生サウンド数とは別に、
    //!           独立してはたらきます。
    //!
    //!           設定したサウンド数を超えるサウンドを再生しようとすると、
    //!           各サウンドに設定されたプレイヤープライオリティを比較して、
    //!           もっとも優先度が低いサウンドが停止します。
    //!
    //!           初期値はアクタープレイヤー 0 番は無制限で、その他は 1 に設定されています。
    //!
    //! @param[in] actorPlayerId  アクタープレイヤー番号です。
    //! @param[in] count          同時に再生可能なサウンド数です。
    //!
    //! @see GetPlayableSoundCount
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPlayableSoundCount( int actorPlayerId, int count );

    //---------------------------------------------------------------------------
    //! @brief    アクタープレイヤーで同時に再生可能なサウンド数を取得します。
    //!
    //! @param[in] actorPlayerId  アクタープレイヤー番号です。
    //!
    //! @return   アクタープレイヤーで同時に再生可能なサウンド数を返します。
    //!
    //! @see SetPlayableSoundCount
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetPlayableSoundCount( int actorPlayerId ) const;
    //@}

    //----------------------------------------
    //! @name その他
    //@{
    //---------------------------------------------------------------------------
    //! @brief    アクターで再生中のすべてのサウンドに対して処理を行います。
    //!
    //!           @ref ForEachSound のFunctionの参照渡し版となります。
    //!
    //! @tparam Function 関数ポインタ、または関数オブジェクトの型です。
    //!
    //! @param[in] function   関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse    処理順を逆にする場合は true を指定します。
    //!
    //! @date 2012/02/10 初版
    //---------------------------------------------------------------------------
    template< class Function >
    void ForEachSoundRef( Function& function, bool reverse = false );

    //---------------------------------------------------------------------------
    //! @brief    アクターで再生中のすべてのサウンドに対して処理を行います。
    //!
    //!           アクターで再生中のすべてのサウンドに対して、
    //!
    //! @code
    //! function( nw::snd::SoundHandle& handle )
    //! @endcode
    //!
    //!           を呼び出します。
    //!
    //!           function には、再生中のサウンドに関連付けられたサウンドハンドル
    //!           handle が渡されます。これは一時的なハンドルですので、
    //!           ハンドルを後で使用することはできません。
    //!
    //!           関数 function は、サウンドの再生が古い順に呼び出されます。
    //!           reverse に true を指定すると、サウンドの再生が新しい順に呼び出されます。
    //!
    //!           function には関数ポインタ、または関数オブジェクトを渡します。
    //!
    //! @tparam Function 関数ポインタ、または関数オブジェクトの型です。
    //!
    //! @param[in] function   関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse    処理順を逆にする場合は true を指定します。
    //!
    //! @return   引数に指定された関数ポインタ、または関数オブジェクトです。
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    template< class Function >
    Function ForEachSound( Function function, bool reverse = false );
    //@}

    //-----------------------------------------------------------------------------
    //  internal functions

    //! @briefprivate
    //! @param actorPlayerId :private
    //! @return :private
    ActorPlayer* detail_GetActorPlayer( int actorPlayerId ) {
        if ( actorPlayerId < 0 || ACTOR_PLAYER_COUNT <= actorPlayerId ) return NULL;
        return &m_ActorPlayer[actorPlayerId];
    }

    //! @briefprivate
    //! @return :private
    const internal::SoundActorParam& detail_GetActorParam() const { return m_ActorParam; }

    //! @briefprivate
    //! @param handle :private
    //! @param soundId :private
    //! @param startInfo :private
    //! @param ambientInfo :private
    //! @param setupArg :private
    //! @return :private
    virtual SoundStartable::StartResult detail_SetupSoundWithAmbientInfo(
        SoundHandle* handle,
        u32 soundId,
        const SoundStartable::StartInfo* startInfo,
        internal::BasicSound::AmbientInfo* ambientInfo,
        void* setupArg
    );

private:
    virtual SoundStartable::StartResult detail_SetupSound(
        SoundHandle* handle,
        u32 soundId,
        bool holdFlag,
        const SoundStartable::StartInfo* startInfo
    );
    virtual SoundArchive::ItemId detail_GetItemId( const char* pString );

    struct SetupInfo
    {
        bool holdFlag;
    };

    SoundArchivePlayer* m_pSoundArchivePlayer;
    ActorPlayer m_ActorPlayer[ ACTOR_PLAYER_COUNT ];
    internal::SoundActorParam m_ActorParam;

    bool m_IsInitialized;
    bool m_IsFinalized;
};

template< class Function >
inline void SoundActor::ForEachSoundRef( Function& function, bool reverse )
{
    for ( int actorPlayerIndex = 0; actorPlayerIndex < ACTOR_PLAYER_COUNT; actorPlayerIndex++ )
    {
        m_ActorPlayer[ actorPlayerIndex ].ForEachSoundRef<Function>( function, reverse );
    }
}

template< class Function >
inline Function SoundActor::ForEachSound( Function function, bool reverse )
{
    for ( int actorPlayerIndex = 0; actorPlayerIndex < ACTOR_PLAYER_COUNT; actorPlayerIndex++ )
    {
        m_ActorPlayer[ actorPlayerIndex ].ForEachSound<Function>( function, reverse );
    }

    return function;
}

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_ACTOR_H_ */

