﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SequenceSoundHandle.h
 *
 * @file snd_SequenceSoundHandle.h
 */

#ifndef NW_SND_SEQUENCE_SOUND_HANDLE_H_
#define NW_SND_SEQUENCE_SOUND_HANDLE_H_

#include <nw/assert.h>
#include <nw/snd/snd_SequenceSound.h>
#include <nw/ut/ut_PreProcessor.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    再生したシーケンスサウンドの操作を行うためのハンドルクラスです。
//!
//!           サウンドは再生時に汎用サウンドハンドル
//!           @ref SoundHandle と関連付けられますが、
//!           シーケンスサウンドに固有の処理は汎用サウンドハンドルには実装されていません。
//!
//!           シーケンスサウンドに固有の処理を行いたい場合は、
//!           汎用サウンドハンドルを引数にとるコンストラクタを用いて、
//!           シーケンスサウンドハンドルを生成し、使用してください。
//!
//!           ハンドルの使い方は、汎用サウンドハンドル @ref SoundHandle と同じです。
//!
//! @see      SoundHandle クラス
//!
//! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SequenceSoundHandle
{
public:
    //! @ref SetTrackBankIndex で index として指定できる最小値です。
    static const int BANK_INDEX_MIN = internal::SequenceSound::BANK_INDEX_MIN;
    //! @ref SetTrackBankIndex で index として指定できる最大値です。
    static const int BANK_INDEX_MAX = internal::SequenceSound::BANK_INDEX_MAX;
    //! @ref SetTrackTranspose で transpose として指定できる最小値です。
    static const s8 TRANSPOSE_MIN = internal::SequenceSound::TRANSPOSE_MIN;
    //! @ref SetTrackTranspose で transpose として指定できる最大値です。
    static const s8 TRANSPOSE_MAX = internal::SequenceSound::TRANSPOSE_MAX;
    //! @ref SetTrackVelocityRange で range として指定できる最小値です。
    static const u8 VELOCITY_RANGE_MIN = internal::SequenceSound::VELOCITY_RANGE_MIN;
    //! @ref SetTrackVelocityRange で range として指定できる最大値です。
    static const u8 VELOCITY_RANGE_MAX = internal::SequenceSound::VELOCITY_RANGE_MAX;

    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           引数の無いコンストラクタは、
    //!           どのサウンドとも関連付けられていないハンドルを構築します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SequenceSoundHandle() : m_pSound( NULL ) { }

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           引数付きコンストラクタは、引数 handle
    //!           で指定したサウンドハンドルが参照しているサウンドを用いて、
    //!           シーケンスサウンドハンドルを構築します。
    //!           シーケンスサウンド以外のサウンドがサウンドハンドルに関連付けられているときは、
    //!           引数の無いコンストラクタと同様に構築します。
    //!
    //!           引数付きコンストラクタで構築したときに、
    //!           サウンドが既に別のシーケンスサウンドハンドルと関連付けられている場合は、
    //!           その関連付けが切られる点に注意してください。
    //!           ただし、引数に渡したサウンドハンドルとの関連付けは切れません。
    //!
    //! @param[in] handle シーケンスサウンドと関連付けられたサウンドハンドルです。
    //!
    //! @see      SoundHandle クラス
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    explicit SequenceSoundHandle( SoundHandle* handle );

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    ~SequenceSoundHandle() { DetachSound(); }
    //@}

    // -----------------------------------------------------------------
    //! @name 再生/停止/一時停止/再開
    //@{

    //---------------------------------------------------------------------------
    //! @brief    再生準備の完了を待ってサウンドを再生します。
    //!
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           サウンドの再生を開始するためには、@ref SoundArchivePlayer::PrepareSound
    //!           を呼び出した後、再生準備が完了していなければなりません。
    //!           この関数は、再生準備が完了したサウンドの再生を開始します。
    //!           再生準備が完了していないサウンドは、完了するまで待ってから再生を開始します。
    //!
    //! @see  nw::snd::SoundArchivePlayer::PrepareSound
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void StartPrepared()
    {
        if ( IsAttachedSound() ) m_pSound->StartPrepared();
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを停止します。
    //!
    //!           ハンドルに関連付けられたサウンドを停止します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトさせることができます。
    //!           0 を指定した場合はフェードアウトを行わず、
    //!           エンベロープのリリースを発音しすべての減衰が完了した後にサウンドが停止します。
    //!
    //!           フェードアウトの音量制御は、フェードインと共有されます。
    //!           フェードアウトにかかるフレーム数は、
    //!           最大音量から音が消えるまでにかかる変化速度を表しますので、
    //!           フェードイン中にフェードアウトを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードアウトが完了する可能性があります。
    //!
    //! @param[in] fadeFrames   フェードアウトにかけるフレーム数です。
    //!                         フレーム数は @ref SoundArchivePlayer::Update
    //!                         の呼び出し回数で換算されます。
    //!                         fadeFrames には 0 以上の値を指定してください。
    //!                         負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Stop( int fadeFrames )
    {
        if ( IsAttachedSound() ) m_pSound->Stop( fadeFrames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを一時停止または再開します。
    //!
    //!           ハンドルに関連付けられたサウンドを一時停止または再開します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトしながら一時停止、
    //!           またはフェードインしながら再開させることができます。
    //!           0 を指定した場合は、即座に一時停止または再開します。
    //!
    //!           一時停止・再開時のフェードは、再生開始時のフェードイン、
    //!           停止時のフェードアウトとは独立してはたらきます。
    //!           フェードにかかるフレーム数は、最大音量から音が消えるまで、
    //!           あるいは、
    //!           発音していない状態から最大音量に達するまでにかかる変化速度を表しますので、
    //!           フェード中にさらにフェードを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードが完了する可能性があります。
    //!
    //! @param[in]    flag        true なら一時停止、false なら再開します。
    //! @param[in]    fadeFrames    フェードイン・フェードアウトにかけるフレーム数です。
    //!                             フレーム数は @ref SoundArchivePlayer::Update
    //!                             の呼び出し回数で換算されます。
    //!                             fadeFrames には 0 以上の値を指定してください。
    //!                             負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see  IsPause
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Pause( bool flag, int fadeFrames )
    {
        if ( IsAttachedSound() ) m_pSound->Pause( flag, fadeFrames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの再生準備が完了しているかどうかを調べます。
    //!
    //!           @ref SoundArchivePlayer::PrepareSound を呼び出した後、
    //!           そのサウンドの再生準備が完了しているかどうかを調べます。
    //!           再生準備が完了したサウンドは、@ref StartPrepared を呼び出した際に、
    //!           すぐに再生を始めることができます。
    //!
    //! @return   サウンドの再生準備が完了していれば true を返します。
    //!
    //! @see  SoundArchivePlayer::PrepareSound
    //! @see  StartPrepared
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPrepared() const { return IsAttachedSound() && m_pSound->IsPrepared(); }

    //---------------------------------------------------------------------------
    //! @brief    サウンドが一時停止中かどうかを調べます。
    //!
    //! @return   サウンドが一時停止状態であれば true を返します。
    //!
    //! @see Pause
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPause() const { return IsAttachedSound() && m_pSound->IsPause(); }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを再生開始時にフェードインさせます。
    //!
    //!           この関数でフェードインの指定を行うと、
    //!           @ref SoundArchivePlayer::StartSound または
    //!           @ref StartPrepared を呼び出して再生を開始してから、
    //!           最初の @ref SoundArchivePlayer::Update
    //!           が呼び出されたときにフェードインが設定されます。
    //!
    //!           サウンドの再生が開始され、最初の @ref SoundArchivePlayer::Update
    //!           が呼ばれた後に、この関数を呼び出しても効果がありません。
    //!
    //!           フェードインの音量制御は、停止時のフェードアウトと共有されます。
    //!
    //! @param[in] frames   フェードインにかけるフレーム数です。
    //!                     フレーム数は @ref SoundArchivePlayer::Update
    //!                     の呼び出し回数で換算されます。
    //!                     frames には 0 以上の値を指定してください。
    //!                     負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see  SoundArchivePlayer::StartSound
    //! @see  StartPrepared
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void FadeIn( int frames )
    {
        if ( IsAttachedSound() ) m_pSound->FadeIn( frames );
    }
    //@}

    // -----------------------------------------------------------------
    //! @name   パラメータ変更
    //@{

    //---------------------------------------------------------------------------
    //! @brief    サウンドの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //!           音量の変化は frames で指定したフレーム数をかけて行われます。
    //!           音量の変化途中にさらにこの関数を呼び出した場合は、
    //!           その時点での変化途中の音量値を基点として、
    //!           新しく指定したフレーム数をかけて音量を変化させます。
    //!
    //! @param[in] volume 変更する音量の倍率 (0.0～) です。
    //! @param[in] frames   音量変化にかけるフレーム数です。
    //!                     フレーム数は @ref SoundArchivePlayer::Update
    //!                     の呼び出し回数で換算されます。
    //!                     frames には 0 以上の値を指定してください。
    //!                     負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVolume( f32 volume, int frames = 0 )
    {
        if ( IsAttachedSound() ) m_pSound->SetVolume( volume, frames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの音程を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの音程を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音程パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音程 pitch は、周波数の比率で指定します。
    //!           すなわち、1.0 を指定すると音程に影響を与えません。
    //!           2.0 を指定すると再生される周波数が 2 倍になり、
    //!           1 オクターブ高い音程になります。
    //!           0.5 を指定すると 1 オクターブ低い音程になります。
    //!           デフォルト値は 1.0 です。
    //!
    //! @param[in] pitch  変更する音程の周波数比率です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPitch( f32 pitch )
    {
        if ( IsAttachedSound() ) m_pSound->SetPitch( pitch );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドのローパスフィルタカットオフ値を変更します。
    //!
    //!           ハンドルに関連付けられたローパスフィルタカットオフ値を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、
    //!           他のどのローパスフィルタカットオフパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           lpfFreq は、カットオフの相対変化の値を指定します。
    //!           0.0 を指定するとカットオフの値を変更しません。
    //!           -1.0 を指定すると、フィルタがかかっていない状態から、
    //!           もっともフィルタがかかっている状態 (カットオフ周波数が下がる方向)
    //!           に変更します。
    //!
    //! @param[in] lpfFreq    0.0 を基準としたフィルタカットオフの相対変化の値です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetLpfFreq( f32 lpfFreq )
    {
        if ( IsAttachedSound() ) m_pSound->SetLpfFreq( lpfFreq );
    }

    //---------------------------------------------------------------------------
    //! @brief    プレイヤープライオリティを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのプレイヤープライオリティを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数は、サウンドアーカイブ中のデータで指定されている
    //!           プレイヤープライオリティの値を変更します。
    //!
    //!           priority の値の範囲は 0～127 で、大きいほど優先度が高くなります。
    //!
    //! @param[in] priority   プレイヤープライオリティの値です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPlayerPriority( int priority )
    {
        if ( IsAttachedSound() ) m_pSound->SetPlayerPriority( priority );
    }

    //---------------------------------------------------------------------------
    //! @brief    発音プライオリティを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの発音プライオリティを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数は、
    //!           サウンドアーカイブ中のデータで指定されている、
    //!           発音プライオリティの値を変更します。
    //!
    //!           priority の値の範囲は 0～127 で、大きいほど優先度が高くなります。
    //!
    //! @param[in] priority   発音プライオリティの値です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetChannelPriority( int priority )
    {
        if ( IsAttachedSound() ) m_pSound->SetChannelPriority( priority );
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのテンポの倍率を変更します。
    //!
    //!           ハンドルに関連付けられたシーケンスサウンドのテンポを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           テンポ tempoRatio は、シーケンスデータのテンポに対する倍率で指定します。
    //!           すなわち、1.0 を指定するとテンポを変更しません。
    //!
    //! @param[in] tempoRatio 変更するテンポの倍率です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTempoRatio( f32 tempoRatio )
    {
        if ( IsAttachedSound() ) m_pSound->SetTempoRatio( tempoRatio );
    }
    //@}


    //----------------------------------------
    //! @name メイン (TV) 出力用パラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するパン (左右の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] pan    0.0 を基準としたパンの相対変化の値です。
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPan( f32 pan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetPan( pan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  メイン (TV) 出力に対するサラウンドパン (前後の定位) を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのメイン出力に対するサラウンドパンを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         surroundPan は、定位の相対変化の値を設定します。
    //!         0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!         1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!         2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!         前方へ定位を移動させたい場合は負の値を指定してください。
    //!         デフォルト値は 0.0 です。
    //!
    //! @param[in] surroundPan    0.0 を基準としたサラウンドパンの相対変化の値です。
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetSurroundPan( f32 surroundPan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetSurroundPan( surroundPan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するメインセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するメインセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのメインセンド量とも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           メインセンドは、
    //!           出力に送るサウンドの音量をエフェクトセンドの後で調節するパラメータです。
    //!           主に、エフェクトのドライ・ウェット成分のバランスを調整するために使用されます。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!           メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see SetFxSend
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetMainSend( f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetMainSend( send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するエフェクトセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するエフェクトセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのエフェクトセンド量とも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!           最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see AuxBus
    //! @see SetMainSend
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetFxSend( AuxBus bus, f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetFxSend( bus, send );
        }
    }
    //@}

    // -----------------------------------------------------------------
    //! @name シーケンス変数
    //@{

    //---------------------------------------------------------------------------
    //! @brief    シーケンスローカル変数を読み込みます。
    //!
    //!           読み込みに成功すると var で指定したアドレスへ変数の値が書き込まれます。
    //!
    //!           サウンドハンドルが無効の場合は、読み込みに失敗して false を返します。
    //!
    //! @param[in] varNo  変数番号 ( 0 ～ 15 ) です。
    //! @param[out] var   読み込む値を格納する変数へのポインタです。
    //!
    //! @return   読み込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadVariable( int varNo, signed short* var ) const
    {
        return IsAttachedSound() && m_pSound->ReadVariable( varNo, var );
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスグローバル変数を読み込みます。
    //!
    //!           読み込みに成功すると var で指定したアドレスへ変数の値が書き込まれます。
    //!
    //! @param[in] varNo  変数番号 ( 0 ～ 15 ) です。
    //! @param[out] var   読み込む値を格納する変数へのポインタです。
    //!
    //! @return   読み込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool ReadGlobalVariable( int varNo, signed short* var )
    {
        return internal::SequenceSound::ReadGlobalVariable( varNo, var );
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンストラック変数を読み込みます。
    //!
    //!           読み込みに成功すると var で指定したアドレスへ変数の値が書き込まれます。
    //!
    //!           サウンドハンドルが無効の場合は、読み込みに失敗して false を返します。
    //!
    //! @param[in] trackNo    トラック番号 ( 0 ～ 15 ) です。
    //! @param[in] varNo      変数番号 ( 0 ～ 15 ) です。
    //! @param[out] var       読み込む値を格納する変数へのポインタです。
    //!
    //! @return   読み込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadTrackVariable( int trackNo, int varNo, signed short* var ) const
    {
        return IsAttachedSound() && m_pSound->ReadTrackVariable( trackNo, varNo, var );
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスローカル変数を書き込みます。
    //!
    //!           サウンドハンドルが無効の場合は、書き込みに失敗して false を返します。
    //!
    //! @param[in] varNo  変数番号 ( 0 ～ 15 ) です。
    //! @param[in] var    書き込む変数値です。
    //!
    //! @return   書き込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool WriteVariable( int varNo, signed short var )
    {
        if ( ! IsAttachedSound() ) return false;
        m_pSound->WriteVariable( varNo, var );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスグローバル変数を書き込みます。
    //!
    //! @param[in] varNo  変数番号（ 0 ～ 15 ) です。
    //! @param[in] var    書き込む変数値です。
    //!
    //! @return   書き込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool WriteGlobalVariable( int varNo, signed short var )
    {
        internal::SequenceSound::WriteGlobalVariable( varNo, var );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンストラック変数を書き込みます。
    //!
    //!           サウンドハンドルが無効の場合は、書き込みに失敗して false を返します。
    //!
    //! @param[in] trackNo    トラック番号 ( 0 ～ 15 ) です。
    //! @param[in] varNo      変数番号 ( 0 ～ 15 ) です。
    //! @param[in] var        書き込む変数値です。
    //!
    //! @return   書き込みに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool WriteTrackVariable( int trackNo, int varNo, signed short var )
    {
        if ( ! IsAttachedSound() ) return false;
        m_pSound->WriteTrackVariable( trackNo, varNo, var );
        return true;
    }
    //@}

    // -----------------------------------------------------------------
    //! @name トラックパラメータ設定
    //@{

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックをミュートします。
    //!           または、ミュート状態を解除します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックのミュート状態を変更します。
    //!           サウンドハンドルが無効の場合は、何もしません。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、ミュートの設定を変更します。
    //!           例えば、トラック 2 とトラック 5 のミュート設定を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           トラックをミュートすると、
    //!           それ以降は指定されたトラックでノートオンが行われなくなります。
    //!
    //!           発音中の音が停止するかどうかは @ref SeqMute での指定によります。
    //!           ミュートを解除すると発音が行われるようになりますが、
    //!           すぐに音が鳴り出すわけではありません。
    //!           次のノートオンから鳴り始めることに注意してください。
    //!
    //!           ミュート状態によく似た、トラックを無音にするサイレント状態
    //!           (@ref SetTrackSilence) があります。
    //!           ミュート状態とサイレント状態はそれぞれ個別に設定できます。
    //!           両方を設定した場合は、両方を解除するまでトラックは発音しなくなります。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] mute           ミュート状態です。
    //!
    //! @see SeqMute
    //! @see SetTrackSilence
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTrackMute( u32 trackBitFlag, SeqMute mute )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackMute( trackBitFlag, mute );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックをミュートします。
    //!           または、ミュート状態を解除します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックのミュート状態を変更します。
    //!           サウンドハンドルが無効の場合は、何もしません。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、ミュートの設定を変更します。
    //!           例えば、トラック 2 とトラック 5 のミュート設定を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           トラックをミュートすると、
    //!           それ以降は指定されたトラックでノートオンが行われなくなります。
    //!
    //!           muteFlag を true にするとシーケンストラックをミュート状態にします。
    //!           同時に、そのトラックで発音中の音を直ちに停止します
    //!           (@ref SeqMute の MUTE_STOP と同じです)。
    //!
    //!           muteFlag を false にすると、ミュートを解除します。
    //!
    //!           ミュートを解除すると発音が行われるようになりますが、
    //!           すぐに音が鳴り出すわけではありません。
    //!           次のノートオンから鳴り始めることに注意してください。
    //!
    //!           ミュート状態によく似た、トラックを無音にするサイレント状態
    //!           (@ref SetTrackSilence) があります。
    //!           ミュート状態とサイレント状態はそれぞれ個別に設定できます。
    //!           両方を設定した場合は、両方を解除するまでトラックは発音しなくなります。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] muteFlag       ミュートするときは true、
    //!                           ミュートを解除するときは false を指定します。
    //!
    //! @see SeqMute
    //! @see SetTrackSilence
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTrackMute( u32 trackBitFlag, bool muteFlag )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackMute( trackBitFlag, muteFlag );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックをサイレント状態にします。
    //!           または、サイレント状態を解除します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックのサイレント状態を変更します。
    //!           サウンドハンドルが無効の場合は、何もしません。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サイレント状態を変更します。
    //!           例えば、トラック 2 とトラック 5 のサイレント状態を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           サイレント状態はミュート状態 (@ref SetTrackMute) と似ています。
    //!           ミュート状態のトラックではノートオン処理が行われないので、
    //!           そのトラックが発音しないようなります。
    //!           一方、サイレンス状態のトラックではノートオン処理は行われますが、
    //!           音量をゼロにすることによって発音しないようになります。
    //!           そのため、サイレンス状態を解除したときに、
    //!           ノートを途中から即座に再生することが可能になります。
    //!
    //!           fadeTimes は、この関数を呼び出したときに発音中の音に対してのみ、
    //!           効果があることに注意してください。
    //!           そのためサイレント状態に移行するフェード期間中に、
    //!           新たに発音した音は、鳴り始めから無音状態になります。
    //!
    //!           サイレント状態では、
    //!           ノートオンされたボイスが無音の状態で消費されてしまうので注意してください。
    //!
    //!           ミュート状態とサイレント状態はそれぞれ個別に設定できます。
    //!           両方を設定した場合は、両方を解除するまでトラックは発音しなくなります。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] silenceFlag    サイレント状態にするときは true、
    //!                           サイレント状態を解除する時は false を指定します。
    //! @param[in] fadeTimes      サイレント状態を変更する時にかかるフェードの時間 (ミリ秒) です。
    //!
    //! @see SetTrackMute
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTrackSilence( u32 trackBitFlag, bool silenceFlag, int fadeTimes )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackSilence( trackBitFlag, silenceFlag, fadeTimes );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、音量を変更します。
    //!           例えば、トラック 2 とトラック 5 の音量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] volume         変更する音量の倍率です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTrackVolume( u32 trackBitFlag, f32 volume )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackVolume( trackBitFlag, volume );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックの音程を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの音程を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音程パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、音程を変更します。
    //!           例えば、トラック 2 とトラック 5 の音程を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           音程 pitch は、周波数の比率で指定します。
    //!           すなわち、1.0 を指定すると音程に影響を与えません。
    //!           2.0 を指定すると再生される周波数が 2 倍になり、
    //!           1 オクターブ高い音程になります。
    //!           0.5 を指定すると 1 オクターブ低い音程になります。
    //!           デフォルト値は 1.0 です。
    //!
    //! @param trackBitFlag   トラックごとのビットフラグです。
    //! @param pitch          変更する音程の周波数比率です。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetTrackPitch( u32 trackBitFlag, f32 pitch )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackPitch( trackBitFlag, pitch );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックのバンクを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックのバンクを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           後着優先で有効になります。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、バンクを変更します。
    //!           例えば、トラック 2 とトラック 5 のバンクを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           バンク番号 bankIndex は、0 ～ 3 の数値で指定します。
    //!           0 ～ 3 は SoundMaker の「マルチバンクの設定」
    //!           で表示されるダイアログ内のリストの行番号に該当します。
    //!           それ以外の番号を指定すると、
    //!           (Debug 版、Development 版) アサートで停止します。
    //!           (Release 版) false を返します。
    //!
    //!           シーケンス内に bank_select コマンドを入れていた場合、
    //!           本関数での設定と bank_select コマンドは、
    //!           後から設定されたものが有効になります。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] bankIndex      変更するバンクの番号です。
    //!
    //! @return   bankIndex が BANK_INDEX_MIN ～ BANK_INDEX_MAX
    //!           の範囲内の場合は true を返します。範囲外の場合は false を返します
    //!           (Release 版のみ。Debug 版、Development 版はアサートで停止します)。
    //!
    //! @date 2011/10/19 BANK_INDEX_MIN, BANK_INDEX_MAX 追加にともない、説明を修正
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool SetTrackBankIndex( u32 trackBitFlag, int bankIndex )
    {
        NW_ASSERT_MINMAX( bankIndex, BANK_INDEX_MIN, BANK_INDEX_MAX );
        if ( bankIndex < BANK_INDEX_MIN || bankIndex > BANK_INDEX_MAX )
        {
            return false;
        }

        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackBankIndex( trackBitFlag, bankIndex );
        }
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックのトランスポーズの値を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           トランスポーズの値を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのトランスポーズの値とも独立して動作し、
    //!           後着優先で有効になります。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、トランスポーズの値を変更します。
    //!           例えば、トラック 2 とトラック 5 のトランスポーズの値を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           トランスポーズの値として、@ref TRANSPOSE_MIN ～ @ref TRANSPOSE_MAX
    //!           の値を設定することができます。
    //!           それ以外の値を指定すると、
    //!           (Debug 版、Development 版) アサートで停止します。
    //!           (Release 版) false を返します。
    //!
    //!           シーケンス内に transpose コマンドを入れていた場合、
    //!           本関数での設定と transpose コマンドは、
    //!           後から設定されたものが有効になります。
    //!
    //! @param[in] trackBitFlag トラックごとのビットフラグです。
    //! @param[in] transpose    トランスポーズの値です。
    //!
    //! @return   transpose が @ref TRANSPOSE_MIN ～ @ref TRANSPOSE_MAX
    //!           の範囲内の場合は true を返します。範囲外の場合は false を返します
    //!           (Release 版のみ。Debug 版、Development 版はアサートで停止します)。
    //!
    //! @see TRANSPOSE_MIN
    //! @see TRANSPOSE_MAX
    //!
    //! @date 2011/10/19 初版
    //---------------------------------------------------------------------------
    bool SetTrackTranspose( u32 trackBitFlag, s8 transpose )
    {
        NW_ASSERT_MINMAX( transpose, TRANSPOSE_MIN, TRANSPOSE_MAX );
        if ( transpose < TRANSPOSE_MIN || transpose > TRANSPOSE_MAX )
        {
            return false;
        }

        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackTranspose( trackBitFlag, transpose );
        }
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドのトラックのベロシティレンジの値を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           ベロシティレンジの値を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのベロシティレンジの値とも独立して動作し、
    //!           後着優先で有効になります。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、ベロシティレンジの値を変更します。
    //!           例えば、トラック 2 とトラック 5 のベロシティレンジを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           ベロシティレンジの値として、@ref VELOCITY_RANGE_MIN ～ @ref VELOCITY_RANGE_MAX
    //!           の値を設定することができます。
    //!           それ以外の値を指定すると、
    //!           (Debug 版、Development 版) アサートで停止します。
    //!           (Release 版) false を返します。
    //!
    //!           シーケンス内に velocity_range コマンドを入れていた場合、
    //!           本関数での設定と velocity_range コマンドは、
    //!           後から設定されたものが有効になります。
    //!
    //! @param[in] trackBitFlag トラックごとのビットフラグです。
    //! @param[in] range        ベロシティレンジの値です。
    //!
    //! @return   velocity_range が @ref VELOCITY_RANGE_MIN ～ @ref VELOCITY_RANGE_MAX
    //!           の範囲内の場合は true を返します。範囲外の場合は false を返します
    //!           (Release 版のみ。Debug 版、Development 版はアサートで停止します)。
    //!
    //! @see VELOCITY_RANGE_MIN
    //! @see VELOCITY_RANGE_MAX
    //!
    //! @date 2011/10/19 初版
    //---------------------------------------------------------------------------
    bool SetTrackVelocityRange( u32 trackBitFlag, u8 range )
    {
        NW_ASSERT_MAX( range, VELOCITY_RANGE_MAX );
        if ( range > VELOCITY_RANGE_MAX )
        {
            return false;
        }

        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackVelocityRange( trackBitFlag, range );
        }
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  シーケンスサウンドのトラックの出力先を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのシーケンストラックの出力先を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         出力先の設定は複数の箇所での設定が重ね合わされず、
    //!         以下の優先度に従って設定されます。
    //!         優先度が高い箇所でパラメータ設定がされた場合、
    //!         それより下位の設定は上書きされます。
    //!
    //!         -# 本関数での設定
    //!         -# サウンドハンドルでの設定 (@ref SoundHandle::SetOutputLine)
    //!         -# サウンドプレイヤーでの設定 (@ref SoundPlayer::SetDefaultOutputLine)
    //!
    //!         trackBitFlag で、設定するトラックを指定します。
    //!         下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!         ビットが立っているトラックすべてに対して、出力先を変更します。
    //!         例えば、トラック 2 とトラック 5 の出力先を変更したい場合は
    //!         (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!         出力先は @ref OutputLine のビットフラグの論理和で指定します。
    //!         指定した値は、それまで設定されていた値を上書きします。
    //!         初期状態では、@ref nw::snd::OUTPUT_LINE_MAIN のみが指定されています。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] lineFlag       出力先のビットフラグです。
    //!
    //! @see ResetTrackOutputLine
    //! @see SoundHandle::SetOutputLine
    //! @see SoundPlayer::SetDefaultOutputLine
    //!
    //! @date 2011/10/25 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputLine( u32 trackBitFlag, u32 lineFlag )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackOutputLine( trackBitFlag, lineFlag );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  シーケンスサウンドのトラックの出力先の設定を無効にします。
    //!
    //!         ハンドルに関連付けられたサウンドのシーケンストラックの出力先の設定を無効にします。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         この関数は @ref SetTrackOutputLine で設定された出力先を無効にして、
    //!         デフォルトの出力先へ戻します。デフォルトの出力先は、
    //!         @ref SoundPlayer::SetDefaultOutputLine で設定された値か、
    //!         @ref SoundHandle::SetOutputLine で設定された値です
    //!         (後者の設定が優先されます)。
    //!
    //!         trackBitFlag で、設定するトラックを指定します。
    //!         下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!         ビットが立っているトラックすべてに対して、出力先の設定を無効にします。
    //!         例えば、トラック 2 とトラック 5 の出力先の設定を無効にしたい場合は
    //!         (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //!
    //! @see SetTrackOutputLine
    //! @see SoundHandle::SetOutputLine
    //! @see SoundPlayer::SetDefaultOutputLine
    //!
    //! @date 2011/10/25 初版
    //---------------------------------------------------------------------------
    void ResetTrackOutputLine( u32 trackBitFlag )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->ResetTrackOutputLine( trackBitFlag );
        }
    }
    //@}

    //! @name 出力先別トラックパラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    出力先別のトラックの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           メイン (TV) もしくは DRC 出力に対するシーケンストラックの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、音量を変更します。
    //!           例えば、トラック 2 とトラック 5 の音量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] volume         変更する音量の倍率です (0.0 ～) 。
    //!
    //! @date 2012/12/18 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputVolume( OutputDevice device, u32 trackBitFlag, f32 volume )
    {
        if ( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackMainOutVolume( trackBitFlag, volume );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcOutVolume( 0, trackBitFlag, volume );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    出力先別のトラックのパン (左右の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           メイン (TV) もしくは DRC 出力に対するシーケンストラックのパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、パンを変更します。
    //!           例えば、トラック 2 とトラック 5 のパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] pan            0.0 を基準としたパンの相対変化の値です。
    //!
    //! @date 2012/12/18 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputPan( OutputDevice device, u32 trackBitFlag, f32 pan )
    {
        if ( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackPan( trackBitFlag, pan );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcPan( 0, trackBitFlag, pan );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    出力先別のトラックのサラウンドパン (前後の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン (TV) もしくは DRC 出力に対するサラウンドパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サラウンドパンを変更します。
    //!           例えば、トラック 2 とトラック 5 のサラウンドパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           surroundPan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!           2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!           前方へ定位を移動させたい場合は負の値を指定してください。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] surroundPan    0.0 を基準としたサラウンドパンの相対変化の値です。
    //!
    //! @date 2012/12/18 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputSurroundPan( OutputDevice device, u32 trackBitFlag, f32 surroundPan )
    {
        if ( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackSurroundPan( trackBitFlag, surroundPan );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcSurroundPan( 0, trackBitFlag, surroundPan );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    出力先別のトラックのメインセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン (TV) もしくは DRC 出力に対するメインセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、メインセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のメインセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!           メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @date 2012/12/18 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputMainSend( OutputDevice device, u32 trackBitFlag, f32 send )
    {
        if ( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackMainSend( trackBitFlag, send );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcMainSend( 0, trackBitFlag, send );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    出力先別のトラックのエフェクトセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン (TV) もしくは DRC 出力に対するエフェクトセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、エフェクトセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のエフェクトセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!           最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] device   出力先です。
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @date 2012/12/18 初版
    //---------------------------------------------------------------------------
    void SetTrackOutputFxSend( OutputDevice device, u32 trackBitFlag, AuxBus bus, f32 send )
    {
        if ( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackFxSend( trackBitFlag, bus, send );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcFxSend( 0, trackBitFlag, bus, send );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    //@}

    //! @name メイン (TV) 出力用トラックパラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するトラックの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           メイン出力に対するシーケンストラックの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、音量を変更します。
    //!           例えば、トラック 2 とトラック 5 の音量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] volume         変更する音量の倍率です (0.0 ～) 。
    //!
    //! @see SetTrackDrcOutVolume
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackMainOutVolume( u32 trackBitFlag, f32 volume ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackMainOutVolume( trackBitFlag, volume );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するトラックのパン (左右の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           メイン出力に対するシーケンストラックのパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、パンを変更します。
    //!           例えば、トラック 2 とトラック 5 のパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] pan            0.0 を基準としたパンの相対変化の値です。
    //!
    //! @see SetTrackDrcPan
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackPan( u32 trackBitFlag, f32 pan ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackPan( trackBitFlag, pan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するトラックのサラウンドパン (前後の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン出力に対するサラウンドパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サラウンドパンを変更します。
    //!           例えば、トラック 2 とトラック 5 のサラウンドパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           surroundPan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!           2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!           前方へ定位を移動させたい場合は負の値を指定してください。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] surroundPan    0.0 を基準としたサラウンドパンの相対変化の値です。
    //!
    //! @see SetTrackDrcSurroundPan
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackSurroundPan( u32 trackBitFlag, f32 surroundPan ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackSurroundPan( trackBitFlag, surroundPan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するトラックのメインセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン出力に対するメインセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、メインセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のメインセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!           メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see SetTrackDrcMainSend
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackMainSend( u32 trackBitFlag, f32 send ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackMainSend( trackBitFlag, send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するトラックのエフェクトセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン出力に対するエフェクトセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、エフェクトセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のエフェクトセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!           最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see SetTrackDrcFxSend
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackFxSend( u32 trackBitFlag, AuxBus bus, f32 send ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackFxSend( trackBitFlag, bus, send );
        }
    }
    //@}

    //! @name DRC 出力用トラックパラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するトラックの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           DRC 出力に対するシーケンストラックの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、音量を変更します。
    //!           例えば、トラック 2 とトラック 5 の音量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] volume         変更する音量の倍率です (0.0 ～) 。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetTrackMainOutVolume
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackDrcOutVolume( u32 trackBitFlag, f32 volume, u32 drcIndex=0 ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackDrcOutVolume( drcIndex, trackBitFlag, volume );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するトラックのパン (左右の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの
    //!           DRC 出力に対するシーケンストラックのパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、パンを変更します。
    //!           例えば、トラック 2 とトラック 5 のパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] pan            0.0 を基準としたパンの相対変化の値です。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetTrackPan
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackDrcPan( u32 trackBitFlag, f32 pan, u32 drcIndex=0 ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackDrcPan( drcIndex, trackBitFlag, pan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するトラックのサラウンドパン (前後の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           DRC 出力に対するサラウンドパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サラウンドパンを変更します。
    //!           例えば、トラック 2 とトラック 5 のサラウンドパンを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           surroundPan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!           2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!           前方へ定位を移動させたい場合は負の値を指定してください。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] surroundPan    0.0 を基準としたサラウンドパンの相対変化の値です。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetTrackSurroundPan
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackDrcSurroundPan( u32 trackBitFlag, f32 surroundPan, u32 drcIndex=0 ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackDrcSurroundPan( drcIndex, trackBitFlag, surroundPan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するトラックのメインセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           DRC 出力に対するメインセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、メインセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のメインセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!           メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetTrackMainSend
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackDrcMainSend( u32 trackBitFlag, f32 send, u32 drcIndex=0 ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackDrcMainSend( drcIndex, trackBitFlag, send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    DRC 出力に対するトラックのエフェクトセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           DRC 出力に対するエフェクトセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、エフェクトセンド量を変更します。
    //!           例えば、トラック 2 とトラック 5 のエフェクトセンド量を変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!           最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] trackBitFlag   トラックごとのビットフラグです。
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see SetTrackFxSend
    //!
    //! @date 2011/10/31 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetTrackDrcFxSend( u32 trackBitFlag, AuxBus bus, f32 send, u32 drcIndex=0 ) )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetTrackDrcFxSend( drcIndex, trackBitFlag, bus, send );
        }
    }
    //@}

    // -----------------------------------------------------------------
    //! @name ハンドル操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief    ハンドルにサウンドが関連付けられているかどうかを調べます。
    //!
    //! @return   ハンドルに関連付けられているサウンドがあれば true を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsAttachedSound() const { return m_pSound != NULL; }

    //---------------------------------------------------------------------------
    //! @brief    ハンドルからサウンドを解放します。
    //!
    //!           ハンドルから開放されたサウンドは、
    //!           その後ハンドルを通して操作できないようになります。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void DetachSound();
    //@}

    // -----------------------------------------------------------------
    //! @name 情報設定 / 取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    サウンドの ID を設定します。
    //!
    //!           ハンドルに関連付けられたサウンドの ID を設定します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           @ref SoundArchivePlayer でサウンドを再生すると、
    //!           再生開始時にサウンド ID が自動的に設定されます。
    //!           この関数を呼び出すと、ID を上書きして変更します。
    //!
    //!           設定した ID を取得するためには @ref GetId を呼び出します。
    //!
    //! @param[in] id サウンドを識別する ID です。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see GetId
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetId( u32 id )
    {
        if ( IsAttachedSound() ) m_pSound->SetId( id );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの ID を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドに設定されている ID を取得します。
    //!           ハンドルが無効の場合は 0xffffffff を返します。
    //!
    //!           この関数で取得できる ID は @ref SetId で設定された ID です。
    //!
    //! @return   サウンドに設定されている ID を返します。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetId() const
    {
        if ( IsAttachedSound() ) return m_pSound->GetId();
        return internal::BasicSound::INVALID_ID;
    }

    //---------------------------------------------------------------------------
    //! @brief    アンビエントパラメータの値を取得します。
    //!
    //!           3D サウンドを利用している際は、
    //!           3D サウンドで計算された結果がアンビエントパラメータに格納されますので、
    //!           アンビエントパラメータを取得して 3D
    //!           サウンドによるサウンドパラメータの値を知ることができます。
    //!
    //! @return   現在のアンビエントパラメータの値を返します。
    //!           ハンドルが無効の場合は NULL を返します。
    //!
    //! @see      nw::snd::SoundParam 構造体
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const SoundParam* GetAmbientParam() const
    {
        if ( ! IsAttachedSound() )
        {
            return NULL;
        }
        return &m_pSound->GetAmbientParam();
    }

    //---------------------------------------------------------------------------
    //! @brief  再生開始からのティック数を取得します。
    //!
    //!         サウンドハンドルが無効の場合は、0 を返します。
    //!
    //!         ティック数のカウンタが 32 ビットを越えると、
    //!         関数が返す値は 0 に戻ります。
    //!
    //!         ティックとは時間を表現する単位で、タイムベースを基準としています
    //!         (タイムベースについては、
    //!         シーケンスデータマニュアルの「タイムベース」の章をご覧ください)。
    //!         たとえば、タイムベースが 48 の場合は、四分音符の長さが 48 ティックに相当します。
    //!
    //! @return   再生開始からのティック数を返します。
    //!
    //! @date 2012/05/07 ティックについて説明を追加
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    unsigned long GetTick() const { return IsAttachedSound() ? m_pSound->GetTick() : 0; }
    //@}


    // -----------------------------------------------------------------
    // 非公開関数

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドのサラウンド振り分け方法を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンド振り分け方法を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         mixMode には MIX_MODE_PAN か MIX_MODE_MIXPARAMETER を設定します。
    //!         MIX_MODE_PAN を指定した場合は、パンとサラウンドパンの値を設定する事で
    //!         サラウンドの振り分けを行います。
    //!         @ref SetChannelMixParameter でサラウンドボリュームを設定していても無視されます。
    //!
    //!         MIX_MODE_MIXPARAMETER を指定した場合は、
    //!         サラウンドボリューム(5.1chの各スピーカーの音量)を指定する必要があります。
    //!         パンやサラウンドパンに値を設定していても無視されます。
    //!
    //! @param[in] mixMode     サラウンド振り分け方法の種類です。
    //!
    //! @see MixMode
    //! @see SetPan
    //! @see SetSurroundPan
    //! @see SetChannelMixParameter
    //! @see SetOutputChannelMixParameter
    //! @see SetTrackChannelMixParameter
    //! @see SetTrackChannelOutputMixParameter
    //---------------------------------------------------------------------------
    void SetMixMode( MixMode mixMode )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetMixMode( mixMode );
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         この関数でサラウンドボリュームを指定すると、TV, DRC の両方の出力に適用されます。
    //!         出力先別に設定したい場合は、@ref SetOutputChannelMixParameter を利用してください。
    //!
    //!         @ref SetTrackChannelMixParameter で設定した値はこれとは別に扱われ、
    //!         最終的なサラウンドボリュームは、掛け合わせた結果になります。
    //!
    //! @param[in]    srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in]    param   サラウンドボリュームです。
    //!
    //! @see SetOutputChannelMixParameter
    //! @see SetTrackChannelMixParameter
    //! @see SetTrackChannelOutputMixParameter
    //---------------------------------------------------------------------------
    void SetChannelMixParameter( u32 srcChNo, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    波形の全チャンネルに、同一のサラウンドボリュームを設定します。
    //! @param param :private
    void SetChannelMixParameter( const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            // 全チャンネルで同じ設定を使用
            for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++ )
            {
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
            }
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を出力先別で変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの
    //!         メイン (TV) もしくは DRC 出力に対するサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         TV, DRC の両方の出力にサラウンドボリュームを設定したい場合、@ref SetChannelMixParameter を利用してください。
    //!
    //!         @ref SetTrackChannelMixParameter で設定した値はこれとは別に扱われ、
    //!         最終的なサラウンドボリュームは、掛け合わせた結果になります。
    //!
    //! @param[in] device  出力先です。
    //! @param[in] srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in] param   サラウンドボリュームです。
    //!
    //! @see SetChannelMixParameter
    //! @see SetTrackChannelMixParameter
    //! @see SetTrackChannelOutputMixParameter
    //---------------------------------------------------------------------------
    void SetOutputChannelMixParameter( OutputDevice device, u32 srcChNo, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    出力先指定で、波形の全チャンネルに同一のサラウンドボリュームを設定します。
    //! @param device :private
    //! @param param :private
    void SetOutputChannelMixParameter( OutputDevice device, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            // 全チャンネルで同じ設定を使用
            for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++)
            {
                m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief    波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)をトラック毎に変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックのサラウンドボリュームを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サラウンドボリュームを変更します。
    //!           例えば、トラック 2 とトラック 5 のサラウンドボリュームを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           param は、サラウンドボリュームの値を設定します。
    //!           全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!           デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!           ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!           ミックスモードにパンが指定されている場合は
    //!           この値を指定しても無視されます。
    //!
    //!           この関数でサラウンドボリュームを指定すると、TV, DRC の両方の出力に適用されます。
    //!           出力先別に設定したい場合は、@ref SetTrackChannelOutputMixParameter を利用してください。
    //!
    //!           @ref SetChannelMixParameter で設定した値はこれとは別に扱われ、
    //!           最終的なサラウンドボリュームは、掛け合わせた結果になります。
    //!
    //! @param[in] trackBitFlag トラックごとのビットフラグです。
    //! @param[in] srcChNo      サラウンドボリュームを指定するチャンネルです。
    //! @param[in] param        サラウンドボリュームです。
    //!
    //! @see SetChannelMixParameter
    //! @see SetOutputChannelMixParameter
    //! @see SetTrackChannelOutputMixParameter
    //---------------------------------------------------------------------------
    void SetTrackChannelMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetTrackChannelMixParameter( trackBitFlag, srcChNo, param );
            m_pSound->SetTrackDrcChannelMixParameter( 0, trackBitFlag, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    波形の全トラック、全チャンネルに同一のサラウンドボリュームを設定します。
    //! @param param :private
    void SetTrackChannelMixParameter( const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            // 全トラック、全チャンネルで同じ設定を使用
            for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++)
            {
                m_pSound->SetTrackChannelMixParameter( internal::driver::SequenceSoundPlayer::ALL_TRACK_BIT_FLAG, srcChNo, param );
                m_pSound->SetTrackDrcChannelMixParameter( 0, internal::driver::SequenceSoundPlayer::ALL_TRACK_BIT_FLAG, srcChNo, param );
            }
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief    波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)をトラック毎に出力先別で変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのシーケンストラックの
    //!           メイン (TV) もしくは DRC 出力に対するサラウンドボリュームを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           trackBitFlag で、設定するトラックを指定します。
    //!           下位ビットから順にトラック 0、トラック 1、トラック 2 … を表し、
    //!           ビットが立っているトラックすべてに対して、サラウンドボリュームを変更します。
    //!           例えば、トラック 2 とトラック 5 のサラウンドボリュームを変更したい場合は
    //!           (1 << 2) | (1 << 5) すなわち、0x0024 とします。
    //!
    //!           param は、サラウンドボリュームの値を設定します。
    //!           全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!           デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!           ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!           ミックスモードにパンが指定されている場合は
    //!           この値を指定しても無視されます。
    //!
    //!           TV, DRC の両方の出力にサラウンドボリュームを設定したい場合、@ref SetTrackChannelMixParameter を利用してください。
    //!
    //!           @ref SetChannelMixParameter で設定した値はこれとは別に扱われ、
    //!           最終的なサラウンドボリュームは、掛け合わせた結果になります。
    //!
    //! @param[in] device       出力先です。
    //! @param[in] trackBitFlag トラックごとのビットフラグです。
    //! @param[in] srcChNo      サラウンドボリュームを指定するチャンネルです。
    //! @param[in] param        サラウンドボリューム量です。
    //!
    //! @see SetChannelMixParameter
    //! @see SetOutputChannelMixParameter
    //! @see SetTrackChannelMixParameter
    //---------------------------------------------------------------------------
    void SetTrackChannelOutputMixParameter( OutputDevice device, u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                m_pSound->SetTrackChannelMixParameter( trackBitFlag, srcChNo, param );
                break;
            case OUTPUT_DEVICE_DRC:
                m_pSound->SetTrackDrcChannelMixParameter( 0, trackBitFlag, srcChNo, param );
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }
    //! @briefprivate
    //! @brief    出力先指定で、波形の全トラック、全チャンネルに同一のサラウンドボリュームを設定します。
    //! @param device :private
    //! @param param :private
    void SetTrackChannelOutputMixParameter( OutputDevice device, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            // 全トラック、全チャンネルで同じ設定を使用
            switch (device)
            {
            case OUTPUT_DEVICE_MAIN:
                for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++)
                {
                    m_pSound->SetTrackChannelMixParameter( internal::driver::SequenceSoundPlayer::ALL_TRACK_BIT_FLAG, srcChNo, param );
                }
                break;
            case OUTPUT_DEVICE_DRC:
                for ( u32 srcChNo = 0; srcChNo < WAVE_CHANNEL_MAX; srcChNo++)
                {
                    m_pSound->SetTrackDrcChannelMixParameter( 0, internal::driver::SequenceSoundPlayer::ALL_TRACK_BIT_FLAG, srcChNo, param );
                }
                break;
            default:
                NW_ASSERTMSG( false, "Invalid device\n" );
                break;
            }
        }
    }

    // ハンドルをサウンドに関連付ける
    //! @briefprivate
    //! @param sound :private
    void detail_AttachSoundAsTempHandle( internal::SequenceSound* sound );

    //! @briefprivate
    //! @return :private
    internal::SequenceSound* detail_GetAttachedSound() { return m_pSound; }

    //! @briefprivate
    //! @return :private
    const internal::SequenceSound* detail_GetAttachedSound() const { return m_pSound; }

private:
    NW_DISALLOW_COPY_AND_ASSIGN( SequenceSoundHandle );

    internal::SequenceSound* m_pSound;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SEQUENCE_SOUND_HANDLE_H_ */

