﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_NWVOICE_H_
#define NW_SND_NWVOICE_H_

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
#include <cafe/os.h>
#include <cafe/ax.h> // for AX_FINAL_MIX_CB_STRUCT
#elif defined( NW_PLATFORM_WIN32 )
#include <winext/cafe/os.h>
#include <winext/cafe/ax.h> // for AX_FINAL_MIX_CB_STRUCT
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
#include <winext/cafe/ax.h> // for AX_FINAL_MIX_CB_STRUCT
#endif

#include <nw/ut.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_FinalMixCallback.h>

namespace nw  { namespace snd {

namespace internal
{

class NwVoiceManager;
class Voice;

struct MixWaveBuffer
{
    f32* frontLeft;     //!< フロント左チャンネルの波形です。
    f32* frontRight;    //!< フロント右チャンネルの波形です。
    f32* rearLeft;      //!< リア左チャンネルの波形です。
    f32* rearRight;     //!< リア右チャンネルの波形です。
    f32* frontCenter;   //!< フロントセンターチャンネルの波形です。
    f32* lfe;           //!< LFE（低域効果音）チャンネルの波形です。

    u32 sampleCount;    //!< バッファの最大サンプル数です。
    u32 channelCount;   //!< バッファの有効チャンネル数です。
};

struct NwVoiceSynthesizeBuffer
{
    static const u32 SAMPLE_PER_FRAME = 144;
    static const int CHANNEL_COUNT_TV = 6;
    static const int CHANNEL_COUNT_DRC = 4;
    static const int DRC_COUNT = 2;

    enum
    {
        FLAG_TV_L   = (1<<0),
        FLAG_TV_R   = (1<<1),
        FLAG_TV_SL  = (1<<2),
        FLAG_TV_SR  = (1<<3),
        FLAG_TV_FC  = (1<<4),
        FLAG_TV_LFE = (1<<5),

        FLAG_DRC0_L  = (1<<6),
        FLAG_DRC0_R  = (1<<7),
        FLAG_DRC0_SL = (1<<8),
        FLAG_DRC0_SR = (1<<9),

        FLAG_DRC1_L  = (1<<10),
        FLAG_DRC1_R  = (1<<11),
        FLAG_DRC1_SL = (1<<12),
        FLAG_DRC1_SR = (1<<13),
    };

    u32 flag;
    u32 sampleCount;

    MixWaveBuffer tvMix;
    MixWaveBuffer drc0Mix;
    MixWaveBuffer drc1Mix;

    ut::Event eventTv;
    ut::Event eventDrc0;
    ut::Event eventDrc1;

    static u32 GetSynthesizeBufferSize() { return SAMPLE_PER_FRAME * ( CHANNEL_COUNT_TV + CHANNEL_COUNT_DRC * DRC_COUNT ) * sizeof(f32); }

    NwVoiceSynthesizeBuffer();
    void Initialize( void* synthesizeBuffer, u32 synthesizeBufferSize );
    void Finalize();
};

class NwVoice
{
    friend class NwVoiceManager;

public:
    typedef void (*DisposeCallbackFunc)( NwVoice* voice, void* arg );

    static const u32 SAMPLE_RATE = 48000;
    static const u32 SAMPLE_PER_FRAME = SAMPLE_RATE * AX_MS_PER_FRAME / 1000;

    NwVoice();

    void Free();
    void FreeAllWaveBuffer();

    bool IsAvailable() const { return m_IsAppendedToList; }

    void AppendWaveBuffer( WaveBuffer* waveBuffer );

    void SetPriority(u32 priority);
    u32 GetPlayPosition() const { return m_PlayPosition; }

    void SetState( VoiceState state ) { m_State = state; }

    void SetSampleFormat(SampleFormat format) { m_SampleFormat = format; }
    void SetSampleRate(u32 sampleRate) { m_SampleRate = sampleRate; }
    void SetAdpcmParam(const AdpcmParam& pParam) { m_AdpcmParam = pParam; }

    void SetVoiceParam( const VoiceParam& voiceParam ) { m_VoiceParam = voiceParam; }

    void UpdateVoiceInfo( VoiceInfo* voiceInfo ) const;

private:
    static const int HISTORY_WAVE_SAMPLES = 4;
    static const int DECODE_WAVE_SAMPLES = 2048;

    struct VeParam
    {
        u16 vol;
        s16 volDelta;
    };
    void Initialize( DisposeCallbackFunc func, void* arg );
    void Finalize();

    void UpdateState( const OutputMode* outputMode );

    void SetupVoice( const VoiceParam& voiceParam, const OutputMode* outputMode );
    void UpdateVoice( const VoiceParam& voiceParam, const OutputMode* outputMode );

    void SetupTvMix( const VoiceParam& voiceParam, OutputMode outputMode );
    void SetupDrcMix( const VoiceParam& voiceParam, OutputMode outputMode );

    void UpdateTvMix( const VoiceParam& voiceParam, OutputMode outputMode );
    void UpdateDrcMix( const VoiceParam& voiceParam, OutputMode outputMode );

    void UpdateVe( const VoiceParam& voiceParam, bool initialFlag );

    void CalcSetupTvMix( VeParam mix[], const OutputMix& src, bool surroundFlag );
    bool CalcUpdateTvMix( VeParam mix[], const OutputMix& src, bool surroundFlag );

    void CalcSetupDrcMix( VeParam mix0[], VeParam mix1[], const OutputMix& src, bool surroundFlag, bool frontBypass );
    bool CalcUpdateDrcMix( VeParam mix0[], VeParam mix1[], const OutputMix& src, bool surroundFlag, bool frontBypass );

    u32 DecodePcm16( s16* buffer, u32 samples );
    u32 DecodePcm8( s16* buffer, u32 samples );
    u32 DecodeAdpcm( s16* buffer, u32 samples );

    void Synthesize( NwVoiceSynthesizeBuffer* buffer, f32* workBuffer, u32 samples, f32 sampleRate );
    bool MixSample( const f32* srcData, f32* mixBus, u32 samples, const VeParam& ve );

    bool UpdateNextWaveBuffer();

    //-------------------------
    // static関数
    //-------------------------
    static bool UpdateDelta( s32 target, VeParam* ve );

    static const u32 BASE_PITCH = 0x00010000;

    enum SrcType
    {
        SRC_TYPE_2_4, //  有効周波数成分1/2のみ残す
        SRC_TYPE_3_4, //  有効周波数成分3/4のみ残す
        SRC_TYPE_4_4, //  有効周波数成分カットしない
        SRC_TYPE_COUNT
    };
    static const int CHANNEL_COUNT_TV = 6;
    static const int CHANNEL_COUNT_DRC = 4;
    static const int DRC_COUNT = 2;

    static const int SRC_COEF_TAP_COUNT = 4;
    static const int SRC_COEF_ELEMENT_COUNT = 128;
    static const s16 s_SrcCoef[SRC_TYPE_COUNT][SRC_COEF_TAP_COUNT * SRC_COEF_ELEMENT_COUNT];

    static s16 s_DecodeWaveBuffer[ DECODE_WAVE_SAMPLES + HISTORY_WAVE_SAMPLES ];

    //-------------------------
    // DisposeCallback
    //-------------------------
    DisposeCallbackFunc m_DisposeCallbackFunc;
    void* m_DisposeCallbackArg;

    //-------------------------
    // Voice
    //-------------------------
    NwVoiceManager* m_pVoiceManager;

    //-------------------------
    // WaveBuffer
    //-------------------------
    WaveBuffer*         m_WaveBufferListBegin;
    WaveBuffer*         m_WaveBufferListEnd;

    //-------------------------
    // ステート保持用変数
    //-------------------------
    VeParam m_Ve;
    VeParam m_MixTv[CHANNEL_COUNT_TV];
    VeParam m_MixDrc[DRC_COUNT][CHANNEL_COUNT_DRC];

    AdpcmContext m_AdpcmContext;

    u32 m_CurrentAddressFraction;
    s16 m_LastSamples[HISTORY_WAVE_SAMPLES];
    u32 m_PlayPosition;

    //-------------------------
    // サンプルパラメータ
    //-------------------------
    SampleFormat        m_SampleFormat;
    u32                 m_SampleRate;
    AdpcmParam          m_AdpcmParam;

    //-------------------------
    // ボイスパラメータ
    //-------------------------
    VoiceState m_State;
    VoiceParam m_VoiceParam;

    //-------------------------
    // プライオリティ
    //-------------------------
    u32                 m_Priority;

    //-------------------------
    // 再生パラメータ
    //-------------------------
    u32                 m_CurrentOffset;    // AXPBOFFSET::currentOffset を保持（ポーズ用）

    u32                 m_LoopCount; // バッファ更新判定用

    bool                m_PauseFlag; // AXボイスの停止状態が、本当は一時停止状態なのかどうか

    //-------------------------
    // 状態変数
    //-------------------------
    bool                m_IsAppendedToList; // s_VoiceListに登録されているかどうか

public:
    ut::LinkListNode    m_VoiceListNode;
    Voice* m_pVoice;    // for develop
};

class NwVoiceManager
{
public:
    static const u32 VOICE_COUNT_MAX = 96;

    void Initialize( );
    void Finalize();

    NwVoice* AllocVoice( u32 priority, NwVoice::DisposeCallbackFunc func, void* arg );
    void FreeVoice( NwVoice* voice );

    void UpdatePriorityOrder( NwVoice* voice );

    void detail_UpdateAllVoices();
    void detail_Synthesize( NwVoiceSynthesizeBuffer* buffer, u32 samples, f32 sampleRate );

    u32 GetActiveVoiceCount() const;
    const NwVoice* detail_GetVoiceById( int ch ); // 開発用

private:
    void AppendVoiceList( NwVoice* voice );
    void EraseVoiceList( NwVoice* voice );

    typedef ut::LinkList< NwVoice, offsetof(NwVoice,m_VoiceListNode)> NwVoiceList;
    NwVoiceList m_VoiceList;
    NwVoiceList m_FreeVoiceList;
    NwVoice m_NwVoiceArray[VOICE_COUNT_MAX];
};

class NwVoiceRenderer : public FinalMixCallback
{
public:
    static const int SYNTHESIZE_BUFFER_COUNT_MAX = 16;

    virtual ~NwVoiceRenderer() {}

    void Initialize( void* synthesizeBuffer, u32 synthesizeBufferSize );
    void Finalize();

    void UpdateAllVoices();
    u32 Synthesize();

    enum SampleRateType
    {
        SAMPLE_RATE_48000,
        SAMPLE_RATE_32000
    };
    void SetSampleRate( SampleRateType type );

protected:
    virtual void OnFinalMix(nw::snd::OutputDevice device, const nw::snd::FinalMixData* data);

private:
    void OnMainOutProcess(const FinalMixData* data);
    void OnDrc0OutProcess(const FinalMixData* data);
    void OnDrc1OutProcess(const FinalMixData* data);


private:
    void MixSamples( const f32* srcp, s32* destp, u32 samples );

    NwVoiceManager m_VoiceManager;

    u32 m_SampleCount;
    f32 m_SampleRate;

    ut::MessageQueue m_SynthesizeMsgQueueTv;
    ut::MessageQueue m_SynthesizeMsgQueueDrc0;
    ut::MessageQueue m_SynthesizeMsgQueueDrc1;
    ut::MessageQueue::BufferType m_SynthesizeMsgBufferTv[ SYNTHESIZE_BUFFER_COUNT_MAX ];
    ut::MessageQueue::BufferType m_SynthesizeMsgBufferDrc0[ SYNTHESIZE_BUFFER_COUNT_MAX ];
    ut::MessageQueue::BufferType m_SynthesizeMsgBufferDrc1[ SYNTHESIZE_BUFFER_COUNT_MAX ];
    NwVoiceSynthesizeBuffer m_SynthesizeBuffer[ SYNTHESIZE_BUFFER_COUNT_MAX ];
    u32 m_SynthesizeBufferCount;

};

}}} // namespace nw::snd

#endif //NW_SND_NWVOICE_H_
