﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_MemorySoundArchive.h
 *
 * @file snd_MemorySoundArchive.h
 */

#ifndef NW_SND_MEMORY_SOUND_ARCHIVE_H_
#define NW_SND_MEMORY_SOUND_ARCHIVE_H_

#include <nw/snd/snd_SoundArchiveFileReader.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief        メモリ上にあるサウンドアーカイブを扱うクラスです。
//!
//!               MemorySoundArchive クラスでは、
//!               サウンドアーカイブから外部参照されているデータを扱うことができません。
//!               外部参照データの再生は必ず失敗します。
//!
//!               MemorySoundArchive では既にメモリ上にサウンドアーカイブが展開されているため、
//!               波形アーカイブの個別ロード機能を使用するメリットはありません。
//!               逆に、サウンドアーカイブが必要とするメモリが
//!               個別ロード機能の管理領域の分だけ大きくなってしまうため、ご注意ください。
//!
//! @date 2013/07/16 個別ロード機能を使用した場合の注意点について追記
//! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class MemorySoundArchive : public SoundArchive
{
private:
    class MemoryFileStream;

public:
    //----------------------------------------
    //! @name コンストラクタ / デストラクタ
    //@{
    //----------------------------------------
    //! @brief コンストラクタです。
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //----------------------------------------
    MemorySoundArchive();
    //----------------------------------------
    //! @brief デストラクタです。
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //----------------------------------------
    virtual ~MemorySoundArchive();
    //@}

    //----------------------------------------
    //! @name 初期化・終了処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief    メモリ上のサウンドアーカイブデータを初期化します。
    //!
    //! @param[in]    soundArchiveData    サウンドアーカイブデータのアドレス。
    //!
    //! @return 初期化に成功すれば true を、失敗すれば false を返します。
    //!
    //! @see Finalize
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool Initialize( const void* soundArchiveData );

    //---------------------------------------------------------------------------
    //! @brief    メモリ上のサウンドアーカイブデータを破棄します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Finalize();
    //@}

    //! @briefprivate
    //! @return :private
    virtual size_t detail_GetRequiredStreamBufferSize() const;

    //! @briefprivate
    //! @param fileId :private
    virtual const void* detail_GetFileAddress( FileId fileId ) const;

protected:

    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param begin :private
    //! @param length :private
    //! @return :private
    virtual ut::FileStream* OpenStream( void* buffer, int size, u32 begin, u32 length ) const;

    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param extFilePath :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    virtual ut::FileStream* OpenExtStream(
            void* buffer,
            int size,
            const char* extFilePath,
            void* cacheBuffer,
            size_t cacheSize ) const;

private:
    const void* m_pData;
    internal::SoundArchiveFileReader m_FileReader;
};

//! @briefprivate
class MemorySoundArchive::MemoryFileStream : public ut::FileStream
{
public:
    MemoryFileStream( const void* buffer, u32 size );

    virtual bool CanSeek() const { return true; }
    virtual bool CanCancel() const { return false; }
    virtual bool CanAsync() const { return false; }
    virtual bool CanRead() const { return true; }
    virtual bool CanWrite() const { return false; }
    virtual bool IsAvailable() const { return m_pBuffer != NULL; }

    virtual void Close();
    virtual s32  Read( void* buf, u32 length );
    virtual bool Seek( s32 offset, u32 origin );
    virtual u32  Tell()    const { return m_Position; }
    virtual u32  GetSize() const { return m_Size; }

private:
    const void* m_pBuffer;
    u32 m_Size;
    u32 m_Position;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_MEMORY_SOUND_ARCHIVE_H_ */

