﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_FsSoundArchive.h
 *
 * @file snd_FsSoundArchive.h
 */

#ifndef NW_SND_FS_SOUND_ARCHIVE_H_
#define NW_SND_FS_SOUND_ARCHIVE_H_

#if defined(NW_PLATFORM_ANDROID)
#include <android/asset_manager.h>
#endif

#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_SoundArchiveFileReader.h>
#include <nw/snd/snd_FsFileStream.h>
#include <nw/snd/snd_CachedFsFileStream.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  FS 上にあるサウンドアーカイブを扱うクラスです。
//! @see SoundArchive クラス
//!
//! @date 2012/04/19 初版
//---------------------------------------------------------------------------
class FsSoundArchive : public SoundArchive
{
private:
    class FsFileStream;
    class CachedFsFileStream;

public:
    //! @brief  各種ロード先のバッファのアラインメントサイズです。
    //!
    //! @see LoadHeader
    //! @see LoadLabelStringData
    //!
    //! @date 2012/04/24 初版
    static const int BUFFER_ALIGN_SIZE = internal::FsFileStream::BUFFER_ALIGN_SIZE;

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief  ストリームサウンドのファイル入出力に用いる FS コマンドの優先度の初期値です。
    //!
    //! @date 2013/01/11 初版
    //---------------------------------------------------------------------------
    static const u8 FS_STREAM_PRIORITY_DEFAULT = 8;
#endif

    //----------------------------------------
    //! @name コンストラクタ / デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    FsSoundArchive();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    virtual ~FsSoundArchive();
    //@}

    //----------------------------------------
    //! @name オープン / クローズ
    //@{
#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief  オープン時に必要なメモリサイズを取得します。
    //!
    //! @return オープン時に必要なメモリサイズを返します。
    //!
    //! @see Open
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    size_t GetRequiredMemSize() const;

    //---------------------------------------------------------------------------
    //! @brief  FS 上のサウンドアーカイブファイルを開きます。
    //!
    //!         開いたサウンドアーカイブを使用するためには、@ref LoadHeader
    //!         を呼び出してサウンドアーカイブのヘッダを読み込まなければなりません。
    //!
    //!         buffer には、@ref GetRequiredMemSize で取得したサイズのメモリを
    //!         渡してください。
    //!
    //!         bufferSize には、@ref GetRequiredMemSize
    //!         で取得したサイズを渡してください。
    //!
    //!         ここで渡したバッファは、@ref Close
    //!         を呼び出すまで解放してはいけません。
    //!
    //! @param[in]    client     FS クライアントです。
    //! @param[in]    filePath   ファイルの FS ファイルシステム上のパスです。
    //! @param[in]    buffer     バッファへのポインタです。
    //! @param[in]    bufferSize バッファサイズです。
    //! @param[in]    fsPriority サウンドアーカイブでの FS コマンドに使用する優先度です。
    //! @param[in]    fsStreamPriority ストリームサウンドのファイル入出力に使用する FS コマンドの優先度です。
    //!
    //! @return   ファイルを開くのに成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see GetRequiredMemSize
    //! @see Close
    //! @see LoadHeader
    //!
    //! @date 2012/04/19 初版
    //! @date 2013/01/15 FS コマンドの優先度を設定する引数を追加
    //---------------------------------------------------------------------------
    bool Open( FSClient* client, const char* filePath, void* buffer, size_t bufferSize, u8 fsPriority = FS_PRIORITY_DEFAULT, u8 fsStreamPriority = FS_STREAM_PRIORITY_DEFAULT );
#else
    //---------------------------------------------------------------------------
    //! @brief  FS 上のサウンドアーカイブファイルを開きます。
    //!
    //! @param[in]    filePath   ファイルの FS ファイルシステム上のパスです。
    //!
    //! @return   ファイルを開くのに成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see Close
    //! @see LoadHeader
    //---------------------------------------------------------------------------
    bool Open(const char* filePath);
#endif

    //! @brief FS 上のサウンドアーカイブファイルのオープンに使用するパラメータです。
    struct OpenArg
    {
#if defined(NW_PLATFORM_CAFE)
        FSClient* client;     //!< FS クライアントです。
        void* buffer;         //!< バッファへのポインタです。
        size_t bufferSize;    //!< バッファサイズです。
        u8 fsPriority;        //!< サウンドアーカイブでの FS コマンドに使用する優先度です。
        u8 fsStreamPriority;  //!< ストリームサウンドのファイル入出力に使用する FS コマンドの優先度です。

        OpenArg() :
            client(NULL), buffer(NULL), bufferSize(0), fsPriority(FS_PRIORITY_DEFAULT), fsStreamPriority(FS_STREAM_PRIORITY_DEFAULT)
        {}
#else
        OpenArg() {}
#endif
    };

    //---------------------------------------------------------------------------
    //! @brief  FS 上のサウンドアーカイブファイルを開きます。
    //!
    //! @param[in]    filePath   ファイルの FS ファイルシステム上のパスです。
    //! @param[in]    arg        サウンドアーカイブファイルのオープンに使用するパラメータです。
    //!
    //! @return   ファイルを開くのに成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @date 2013/01/15 初版
    //---------------------------------------------------------------------------
    bool Open(const char* filePath, const OpenArg* arg)
    {
#if defined(NW_PLATFORM_CAFE)
        return Open(arg->client, filePath, arg->buffer, arg->bufferSize, arg->fsPriority, arg->fsStreamPriority);
#else
        NW_UNUSED_VARIABLE(arg);
        return Open(filePath);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief    FS 上のサウンドアーカイブファイルを閉じます。
    //!
    //! @see Open
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    void Close();
    //@}

    //----------------------------------------
    //! @name ヘッダ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブのヘッダのサイズを取得します。
    //!
    //! @return   ヘッダのサイズを返します。
    //!
    //! @see LoadHeader
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    size_t GetHeaderSize() const { return m_ArchiveReader.GetInfoBlockSize(); }

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブのヘッダをロードします。
    //!
    //!           サウンドアーカイブからの情報の取得やサウンドデータのロードを行う前に、
    //!           この関数を呼びだしてヘッダをロードしておく必要があります。
    //!
    //!           ヘッダのロードに必要なメモリのサイズは、
    //!           @ref GetHeaderSize で取得することができます。
    //!
    //! @param[out]   buffer    ヘッダをロードするバッファへのポインタです。
    //!                         BUFFER_ALIGN_SIZE バイト境界に配置されている必要があります。
    //! @param[in]    size      バッファのサイズです。
    //!
    //! @return   ヘッダのロードに成功したら true を、失敗したら false を返します。
    //!
    //! @see Open
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    bool LoadHeader( void* buffer, unsigned long size );
    //@}

    //----------------------------------------
    //! @name ラベル文字列データ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のラベル文字列データのサイズを取得します。
    //!
    //! @return   ラベル文字列データのサイズを返します。
    //!
    //! @see LoadLabelStringData
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    size_t GetLabelStringDataSize() const { return m_ArchiveReader.GetStringBlockSize(); }

    //---------------------------------------------------------------------------
    //! @brief  サウンドアーカイブ中のラベル文字列データをロードします。
    //!
    //!         サウンドアーカイブに含まれるリソースにラベル文字列でアクセスするためには、
    //!         ラベル文字列データをロードしておく必要があります。
    //!
    //!         ラベル文字列データのロードに必要なメモリのサイズは、
    //!         @ref GetLabelStringDataSize で取得することができます。
    //!
    //!         サウンドアーカイブファイル (.bcsar) に文字列テーブルが含まれていないと、
    //!         失敗します。その場合は、SoundMaker の [プロジェクト設定] -
    //!         [サウンドアーカイブ] タブ - [文字列テーブルを出力する]
    //!         にチェックがついているかどうかご確認ください。
    //!
    //! @param[out]   buffer    ラベル文字列をロードするバッファへのポインタです。
    //!                         BUFFER_ALIGN_SIZE バイト境界に配置されている必要があります。
    //! @param[in]    size      バッファのサイズです。
    //!
    //! @return   ラベル文字列のロードに成功したら true を、失敗したら false を返します。
    //!
    //! @see Open
    //!
    //! @date 2012/04/19 初版
    //---------------------------------------------------------------------------
    bool LoadLabelStringData( void* buffer, unsigned long size );
    //@}

    //! @briefprivate
    //! @return :private
    virtual size_t detail_GetRequiredStreamBufferSize() const;

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* detail_GetFileAddress( FileId fileId ) const
    {
        NW_UNUSED_VARIABLE( fileId );
        return NULL;    // NOTE: MemorySoundArchive なら非 NULL 値を返しうる
    }

    //! @briefprivate
    enum FileAccessMode
    {
        FILE_ACCESS_ALWAYS,         //!< Close が呼ばれるまで bfsar をオープンし続けます
        FILE_ACCESS_IN_FUNCTION     //!< 関数単位 (LoadHeader など) で bfsar をオープン→クローズします。
    };

    //! @briefprivate
    //! @brief  サウンドアーカイブファイルへのアクセスモードを設定します。
    //!
    //!         FILE_ACCESS_ALWAYS はこれまでと同じ挙動で、
    //!         Open ～ Close の間中、ずっとサウンドアーカイブファイルをオープンし続けます。
    //!
    //!         FILE_ACCESS_IN_FUNCTION は、Open / LoadHeader /　LoadLabelStringData
    //!         および SoundDataManager::LoadData の間だけ、サウンドアーカイブファイルを
    //!         オープンします。
    //!         それ以外のタイミングではクローズしていますので、HostFile IO デバイス上の
    //!         サウンドアーカイブファイルを Open している場合は、
    //!         PC 上のサウンドアーカイブファイルを変更することが可能です。
    //!
    //!         FILE_ACCESS_IN_FUNCTION の場合、上記で挙げた各 API で、
    //!         サウンドアーカイブファイルのオープン・クローズを繰り返すため
    //!         ある程度のオーバーヘッドが予想されます。
    //!         このオーバーヘッドを緩和するには、FileAccessBegin および FileAccessEnd
    //!         をご利用ください。
    //!
    //!         FileAccessBegin ～ FileAccessEnd の間は、サウンドアーカイブファイルが
    //!         オープンされっぱなしになるため、
    //!         PC 上のサウンドアーカイブファイルを変更することはできませんが、
    //!         オープン・クローズを繰り返す時と比べ、オーバーヘッドは緩和されます。
    //!
    //!         本関数は、Open の前に呼び出す必要があります。
    //!         Open 以降に呼び出した場合の動作は未保証です。
    //!
    //! @param[in] mode アクセスモードです。
    void SetFileAccessMode( FileAccessMode mode ) { m_FileAccessMode = static_cast<u8>(mode); }

    //! @briefprivate
    //! @return :private
    FileAccessMode GetFileAccessMode() const { return static_cast<FileAccessMode>(m_FileAccessMode); }

    //! @briefprivate
    virtual void FileAccessBegin() const;

    //! @briefprivate
    virtual void FileAccessEnd() const;

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブでの FS コマンドに使用する優先度を取得します。
    //!
    //! @return   サウンドアーカイブでの FS コマンドに使用する優先度です。
    //!
    //! @date 2013/01/15 初版
    //---------------------------------------------------------------------------
    virtual u8 GetFsPriority() const { return m_FsPriority; }

    //---------------------------------------------------------------------------
    //! @brief    ストリームサウンドのファイル入出力に用いる FS コマンドの優先度を取得します。
    //!
    //! @return   ストリームサウンドのファイル入出力に用いる FS コマンドの優先度です。
    //!
    //! @date 2013/01/15 初版
    //---------------------------------------------------------------------------
    virtual u8 GetFsStreamPriority() const { return m_FsStreamPriority; }
#endif
#if defined(NW_PLATFORM_ANDROID)
    //---------------------------------------------------------------------------
    //! @brief        Android の Assets 領域へアクセスするための AssetManager を設定します。
    //!
    //! @param[in]    pAssetManager  AssetManager の ハンドル
    //---------------------------------------------------------------------------
    static void SetAssetManager(AAssetManager *pAssetManager) { nw::snd::internal::FsFileStream::SetAssetManager(pAssetManager); }
    static const char* const ANDROID_ASSETS_PATH_PREFIX;
#endif

protected:
    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param begin :private
    //! @param length :private
    //! @return :private
    virtual ut::FileStream* OpenStream(
            void* buffer,
            int size,
            u32 begin,
            u32 length ) const;

    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param extFilePath :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    virtual ut::FileStream* OpenExtStream(
            void* buffer,
            int size,
            const char* extFilePath,
            void* cacheBuffer,
            size_t cacheSize ) const;

private:
    bool LoadFileHeader();

    internal::SoundArchiveFileReader m_ArchiveReader;
    mutable nw::snd::internal::FsFileStream m_FileStream;
    bool m_IsOpened;
    u8 m_FileAccessMode;    // FileAccessMode が入る
    u8 padding[2];
    mutable u32 m_FileAccessCount;
    char m_SoundArchiveFullPath[ FILE_PATH_MAX ];

#if defined(NW_PLATFORM_CAFE)
    u8 m_FsPriority;
    u8 m_FsStreamPriority;
#endif
};

//! @briefprivate
class FsSoundArchive::FsFileStream : public internal::FsFileStream
{
public:
#if defined(NW_PLATFORM_CAFE)
    FsFileStream( FSClient* client, const char* path, u32 offset, u32 size, FsCommandBlockPool* pool );
#else
    FsFileStream( const char* path, u32 offset, u32 size );
#endif
    FsFileStream( const internal::FsFileStream* fileStream, u32 offset, u32 size );

    virtual s32  Read( void* buf, u32 length );
    virtual bool Seek( s32 offset, u32 origin );
    virtual u32  Tell()    const { return internal::FsFileStream::Tell() - m_Offset; }
    virtual u32  GetSize() const { return m_Size; }

private:
    s32 m_Offset;
    u32 m_Size;
};

//! @briefprivate
class FsSoundArchive::CachedFsFileStream : public internal::CachedFsFileStream
{
public:
#if defined(NW_PLATFORM_CAFE)
    CachedFsFileStream( FSClient* client, const char* path, FsCommandBlockPool* pool );
#else
    explicit CachedFsFileStream( const char* path );
#endif
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_FS_SOUND_ARCHIVE_H_ */

