﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FRAME_HEAP_H_
#define NW_SND_FRAME_HEAP_H_

#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_FrameHeap.h>
#include <nw/snd/snd_SoundDataManager.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        class definition
   ======================================================================== */

class FrameHeap
{
    /* ------------------------------------------------------------------------
            type definition
       ------------------------------------------------------------------------ */
public:
    typedef void (*DisposeCallback)( void* mem, unsigned long size, void* userArg );

    struct Block
    {
      public:
        Block( void* buffer, u32 size, DisposeCallback callback, void* callbackArg )
        : m_pBuffer( buffer ),
          m_Size( size ),
          m_Callback( callback ),
          m_pCallbackArg( callbackArg )
        {}

        ~Block()
        {
            if ( m_Callback != NULL )
            {
                m_Callback( m_pBuffer, m_Size, m_pCallbackArg );
            }
        }

        void* GetBufferAddr(){ return m_pBuffer; }
        const void* GetBufferAddr() const { return m_pBuffer; }
        u32 GetBufferSize() const { return m_Size; }
        DisposeCallback GetDisposeCallback() const { return m_Callback; }
            /* const DisposeCallback を返そうとすると、RVCT で、
               "type qualifier on return type is meaningless" の警告 */
        const void* GetDisposeCallbackArg() const { return m_pCallbackArg; }

      public:
        ut::LinkListNode m_Link;
      private:
        void* m_pBuffer;
        u32 m_Size;
        DisposeCallback m_Callback;
        void* m_pCallbackArg;
    };

    typedef ut::LinkList< Block, offsetof(Block,m_Link)> BlockList;

    class Section
    {
      public:
        Section();
        ~Section();
        void AppendBlock( Block* block );
        const BlockList& GetBlockList() const { return m_BlockList; }
        BlockList& GetBlockList() { return m_BlockList; }

        void* GetAddr() { return this; }
        void SetUseCallback( bool use ) { m_UseCallback = use; }

        void Dump( nw::snd::SoundDataManager& mgr, nw::snd::SoundArchive& arc ) const;

      public:
        ut::LinkListNode m_Link;
      private:
        BlockList m_BlockList;
        bool m_UseCallback;
    };

    typedef ut::LinkList< Section, offsetof(Section,m_Link)> SectionList;

    /* ------------------------------------------------------------------------
            constant declaration
       ------------------------------------------------------------------------ */
private:
    static const int HEAP_ALIGN = 256; // TODO

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    FrameHeap();
    ~FrameHeap();
    bool  Create( void* startAddress, u32 size );
    void  Destroy();

    void* Alloc(
        u32 size,
        FrameHeap::DisposeCallback callback,
        void* callbackArg
    );
    void  Clear();

    int   SaveState();
    void  LoadState( int level );
    int   GetCurrentLevel() const;

    u32   GetSize() const;
    u32   GetFreeSize() const;

    bool  IsValid() const { return m_pHeap != NULL; }

    // TODO: 不要？
    // const SectionList& GetSectionList() const { return m_SectionList; }

    void Dump( nw::snd::SoundDataManager& mgr, nw::snd::SoundArchive& arc ) const;
    bool ProcessCallback( int level );

private:
    ut::FrameHeap* m_pHeap;
    SectionList m_SectionList;

    bool NewSection();
    void ClearSection();
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_FRAME_HEAP_H_ */

