﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_AXVOICE_H_
#define NW_SND_AXVOICE_H_

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/os.h>
  #include <cafe/ax.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe/os.h>
  #include <winext/cafe/ax.h>
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
  #include <winext/cafe/os.h>
  #include <winext/cafe/ax.h>
#elif defined( NW_USE_NINTENDO_SDK )
  // TODO: nn_audio
  #include <winext/cafe/ax.h>
#endif

#include <nw/ut.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Config.h>

namespace nw  { namespace snd {

namespace internal
{

class Voice;

class AxVoice
{
public:
    static const int VOICE_COUNT_MAX = AX_MAX_VOICES;

    typedef void (*DisposeCallbackFunc)( AxVoice* voice, void* arg );

    AxVoice();

    static AxVoice* AllocVoice( u32 priority, DisposeCallbackFunc func, void* arg );

    void Free();
    void FreeAllWaveBuffer();

    bool IsAvailable() const { return m_pVpb != NULL; }

    void AppendWaveBuffer( WaveBuffer* waveBuffer );

    void SetPriority(u32 priority);
    u32 GetPlayPosition() const { return m_PlayPosition; }

    void SetState( VoiceState state ) { m_State = state; }

    void SetSampleFormat(SampleFormat format) { m_SampleFormat = format; }
    void SetSampleRate(u32 sampleRate) { m_SampleRate = sampleRate; }
    void SetAdpcmParam(const AdpcmParam& pParam) { m_AdpcmParam = pParam; }

    void SetVoiceParam( const VoiceParam& voiceParam ) { m_VoiceParam = voiceParam; }
    void SetRenderer( VoiceRendererType type ) { m_Renderer = type; }

    static void detail_UpdateAllVoices(); // SoundThread::HwCallbackProcから呼び出される
    static void detail_UpdatePriorityList();
    static const AxVoice* detail_GetVoiceById( int ch ); // 開発用

    const Voice* detail_GetVoice() const { return m_pVoice; }

    void UpdateVoiceInfo( VoiceInfo* voiceInfo ) const;

    VoiceState GetState() const { return m_State; }
    const VoiceParam& GetVoiceParam() const { return m_VoiceParam; }

    static void SetSamplesPerFrame( s32 samplesPerFrame ) { s_SamplesPerFrame = samplesPerFrame; }
    static void SetSamplesPerSec  ( s32 samplesPerSec   ) { s_SamplesPerSec   = samplesPerSec; }

private:
    static const int VOICE_PRIORITY_FREE    = 0;
    static const u32 VOICE_PRIORITY_USE     = 16; // Voiceクラスが使用するプライオリティ
    static const u32 VOICE_PRIORITY_ALLOC   = VOICE_PRIORITY_USE + 1;
    static const u32 VOICE_PRIORITY_NODROP  = AX_PRIORITY_NODROP;

#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXVPB AXVPB;
    typedef nw::internal::winext::AXPBCHMIX AXPBCHMIX;
    typedef nw::internal::winext::AXPBCHVE AXPBCHVE;
    typedef nw::internal::winext::AXPBVE AXPBVE;
    typedef nw::internal::winext::AXPBADPCM AXPBADPCM;

    static const u32 AX_MAX_NUM_TVS = nw::internal::winext::AX_MAX_NUM_TVS;
    static const u32 AX_MAX_NUM_TV_CHS = nw::internal::winext::AX_MAX_NUM_TV_CHS;
    static const u32 AX_MAX_NUM_DRCS = nw::internal::winext::AX_MAX_NUM_DRCS;
    static const u32 AX_MAX_NUM_DRC_CHS = nw::internal::winext::AX_MAX_NUM_DRC_CHS;
    static const u32 AX_MAX_NUM_RMTS = nw::internal::winext::AX_MAX_NUM_RMTS;
    static const u32 AX_MAX_NUM_RMT_CHS = nw::internal::winext::AX_MAX_NUM_RMT_CHS;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXVPB AXVPB;
    typedef nw::internal::winext::AXPBCHMIX AXPBCHMIX;
    typedef nw::internal::winext::AXPBCHVE AXPBCHVE;
    typedef nw::internal::winext::AXPBVE AXPBVE;
    typedef nw::internal::winext::AXPBADPCM AXPBADPCM;

    static const u32 AX_MAX_NUM_TVS = nw::internal::winext::AX_MAX_NUM_TVS;
    static const u32 AX_MAX_NUM_TV_CHS = nw::internal::winext::AX_MAX_NUM_TV_CHS;
    static const u32 AX_MAX_NUM_DRCS = nw::internal::winext::AX_MAX_NUM_DRCS;
    static const u32 AX_MAX_NUM_DRC_CHS = nw::internal::winext::AX_MAX_NUM_DRC_CHS;
    static const u32 AX_MAX_NUM_RMTS = nw::internal::winext::AX_MAX_NUM_RMTS;
    static const u32 AX_MAX_NUM_RMT_CHS = nw::internal::winext::AX_MAX_NUM_RMT_CHS;
#elif defined( NW_USE_NINTENDO_SDK )
    // nn_audio
    typedef nw::internal::winext::AXVPB AXVPB;
    typedef nw::internal::winext::AXPBCHMIX AXPBCHMIX;
    typedef nw::internal::winext::AXPBCHVE AXPBCHVE;
    typedef nw::internal::winext::AXPBVE AXPBVE;
    typedef nw::internal::winext::AXPBADPCM AXPBADPCM;

    static const u32 AX_MAX_NUM_TVS = nw::internal::winext::AX_MAX_NUM_TVS;
    static const u32 AX_MAX_NUM_TV_CHS = nw::internal::winext::AX_MAX_NUM_TV_CHS;
    static const u32 AX_MAX_NUM_DRCS = nw::internal::winext::AX_MAX_NUM_DRCS;
    static const u32 AX_MAX_NUM_DRC_CHS = nw::internal::winext::AX_MAX_NUM_DRC_CHS;
    static const u32 AX_MAX_NUM_RMTS = nw::internal::winext::AX_MAX_NUM_RMTS;
    static const u32 AX_MAX_NUM_RMT_CHS = nw::internal::winext::AX_MAX_NUM_RMT_CHS;
#endif // defined( NW_PLATFORM_WIN32 )

    void Initialize( AXVPB* axvpb );
    void Finalize();

    void UpdateState( const OutputMode* mode ); // OUTPUT_DEVICE_COUNT 個だけセットされる

    void UpdateStatePlay( const VoiceParam& voiceParam, bool isRun, const OutputMode* mode );
    void UpdateStateStop( const VoiceParam& voiceParam,bool isRun );
    void UpdateStatePause( const VoiceParam& voiceParam, bool isRun );
    void UpdatePlayPosition();

    void SetupVoice( const VoiceParam& voiceParam, const WaveBuffer* waveBuffer,
            const OutputMode* mode );

    void SetupNextWaveBuffer( const WaveBuffer* nextWaveBuffer );

    void SetupOffset( const WaveBuffer* waveBuffer, const VoiceParam& voiceParam );
    void SetupAdpcm( const WaveBuffer* waveBuffer, const VoiceParam& voiceParam );

    void SetupTvMix( const VoiceParam& voiceParam, OutputMode mode );
    void SetupDrcMix( const VoiceParam& voiceParam, OutputMode mode);
    void SetupRmtMix( const VoiceParam& voiceParam );

    void UpdateTvMix( const VoiceParam& voiceParam, OutputMode mode );
    void UpdateDrcMix( const VoiceParam& voiceParam, OutputMode mode );
    void UpdateRmtMix( const VoiceParam& voiceParam );

    void UpdateVe( const VoiceParam& voiceParam, bool initialFlag );
    void UpdateSrc( const VoiceParam& voiceParam, bool initialFlag );

    void UpdateMonoFilter( const VoiceParam& voiceParam, bool initialFlag );
    void UpdateBiquadFilter( const VoiceParam& voiceParam, bool initialFlag );
    void UpdateRemoteFilter( const VoiceParam& voiceParam, bool initialFlag );

    void CalcSetupTvMix(AXPBCHMIX mix[], const OutputMix& src, bool surroundFlag);
    bool CalcUpdateTvMix(AXPBCHMIX mix[], const OutputMix& src, bool surroundFlag);

    void CalcSetupDrcMix(AXPBCHMIX* mix0, AXPBCHMIX* mix1,
            const OutputMix& src, bool surroundFlag, bool frontBypass);
    bool CalcUpdateDrcMix(AXPBCHMIX* mix0, AXPBCHMIX* mix1,
            const OutputMix& src, bool surroundFlag, bool frontBypass);

    //-------------------------
    // static関数
    //-------------------------
#if defined( NW_PLATFORM_CAFE )
    static void VoiceCallback(void * voiceIn, u32 context, u32 reason);
#elif defined( NW_PLATFORM_WIN32 )
    static void VoiceCallback(void * voiceIn);
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    static void VoiceCallback(void * voiceIn);
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    static void VoiceCallback(void * voiceIn);
#endif

    static void AppendVoiceList( AxVoice* voice );
    static void EraseVoiceList( AxVoice* voice );

    static bool UpdateDelta( s32 target, AXPBCHVE* ve );

    static u32 CalcVoiceOffsetFromSampleCount( u32 sampleCount, SampleFormat format );
    static u32 CalcVoiceOffsetFromByte( u32 byte, SampleFormat format );
    static u32 AddVoiceOffsetSampleCount( u32 voiceOffset, u32 sampleCount, SampleFormat format );

    // static const MixParam DEFAULT_MIXPARAM;
    static const AXPBADPCM AXPBADPCM_PCM8;
    static const AXPBADPCM AXPBADPCM_PCM16;
    static const u32 BASE_PITCH = 0x00010000;

    static ut::CriticalSection s_CriticalSection;
    static s32                 s_SamplesPerFrame;
    static s32                 s_SamplesPerSec;

    //-------------------------
    // DisposeCallback
    //-------------------------
    DisposeCallbackFunc m_DisposeCallbackFunc;
    void* m_DisposeCallbackArg;

    //-------------------------
    // AXVPB
    //-------------------------
    AXVPB*              m_pVpb;

    //-------------------------
    // WaveBuffer
    //-------------------------
    WaveBuffer*         m_WaveBufferListBegin;
    WaveBuffer*         m_WaveBufferListEnd;

    //-------------------------
    // ステート保持用変数
    //-------------------------
    AXPBVE              m_Ve;
    AXPBCHMIX           m_AxMixTv[AX_MAX_NUM_TVS][AX_MAX_NUM_TV_CHS];
    AXPBCHMIX           m_AxMixDrc[AX_MAX_NUM_DRCS][AX_MAX_NUM_DRC_CHS];
    AXPBCHMIX           m_AxMixRemote[AX_MAX_NUM_RMTS][AX_MAX_NUM_RMT_CHS];
    f32                 m_SrcRatio;

    u16                 m_RemoteOn;
    u16                 m_MonoFilterOn;
    u16                 m_BiquadFilterOn;
    u16                 m_RemoteFilterOn;

    //-------------------------
    // サンプルパラメータ
    //-------------------------
    SampleFormat        m_SampleFormat;
    u32                 m_SampleRate;
    AdpcmParam          m_AdpcmParam;

    //-------------------------
    // ボイスパラメータ
    //-------------------------
    VoiceState m_State;
    VoiceParam m_VoiceParam;
    VoiceRendererType m_Renderer;

    //-------------------------
    // プライオリティ
    //-------------------------
    u32                 m_Priority;

    //-------------------------
    // 再生パラメータ
    //-------------------------
    u32                 m_CurrentOffset;    // AXPBOFFSET::currentOffset を保持（ポーズ用）
    u32                 m_PlayPosition;

    u32                 m_LoopCount; // バッファ更新判定用

    bool                m_SetupInitialNextWaveBufferFlag;
    bool                m_SetupNextWaveBufferFlag;

    bool                m_PauseFlag; // AXボイスの停止状態が、本当は一時停止状態なのかどうか

    //-------------------------
    // 状態変数
    //-------------------------
    bool                m_IsAppendedToList; // s_VoiceListに登録されているかどうか

public:
    ut::LinkListNode    m_VoiceListNode;
    Voice* m_pVoice;    // for develop
};

}}} // namespace nw::snd

#endif //NW_SND_AXVOICE_H_
