﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_ENDIAN_H_
#define NW_SND_FND_ENDIAN_H_

#include <nw/ut/ut_Inlines.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(push)
#pragma warning(disable:4200)
#endif

#pragma pack(push, 1)

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  エンディアンユーティリティ用の名前空間
//---------------------------------------------------------------------------
namespace endian {

typedef union
{
    u64 UInt64;
    s64 SInt64;
#if defined( NW_ENABLE_FLOAT64 )
    f64 Float64;
#endif
} Type64;

typedef union
{
    u32 UInt32;
    s32 SInt32;
    f32 Float32;
} Type32;

typedef union
{
    u16 UInt16;
    s16 SInt16;
} Type16;

//---------------------------------------------------------------------------
//! @brief        u64 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline u64 ByteSwap( u64 val )
{
    const u64 MASK  = 0xFF00FF00FF00FF00ULL;
    const u64 MASK2 = 0xFFFF0000FFFF0000ULL;
    val = ( (val & MASK) >> 8 ) | ( (val << 8) & MASK );
    val = ( (val & MASK2) >> 16 ) | ( (val << 16) & MASK2 );
    return (val >> 32) | (val << 32);
}

//---------------------------------------------------------------------------
//! @brief        s64 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline s64 ByteSwap( s64 val )
{
    Type64 data;
    data.SInt64 = val;
    data.UInt64 = ByteSwap( data.UInt64 );
    return data.SInt64;
}

#if defined( NW_ENABLE_FLOAT64 )
//---------------------------------------------------------------------------
//! @brief        f64 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline f64 ByteSwap( f64 val )
{
    Type64 data;
    data.Float64 = val;
    data.UInt64 = ByteSwap( data.UInt64 );
    return data.Float64;
}
#endif

//---------------------------------------------------------------------------
//! @brief        u32 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline u32 ByteSwap( u32 val )
{
    const u32 MASK = 0xFF00FF00;
    val = ( (val & MASK) >> 8 ) | ( (val << 8) & MASK );
    return (val >> 16) | (val << 16);
}

//---------------------------------------------------------------------------
//! @brief        s32 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline s32 ByteSwap( s32 val )
{
    Type32 data;
    data.SInt32 = val;
    data.UInt32 = ByteSwap( data.UInt32 );
    return data.SInt32;
}

//---------------------------------------------------------------------------
//! @brief        f32 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline f32 ByteSwap( f32 val )
{
    Type32 data;
    data.Float32 = val;
    data.UInt32 = ByteSwap( data.UInt32 );
    return data.Float32;
}

//---------------------------------------------------------------------------
//! @brief        u16 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline u16 ByteSwap( u16 val )
{
    return (u16)( (val >> 8) | (val << 8) );
}

//---------------------------------------------------------------------------
//! @brief        s16 のエンディアンを変換します。
//! @param[in]    val     エンディアン変換する値です。
//! @return       変換結果を返します。
//---------------------------------------------------------------------------
inline s16 ByteSwap( s16 val )
{
    return (s16)( ((u16)val >> 8) | ((u16)val << 8) );
}

//---------------------------------------------------------------------------
//! @brief  エンディアン変換をサポートする数値クラスです。
//---------------------------------------------------------------------------
template <typename T>
class ReverseEndianNum
{
public:
    ReverseEndianNum() {}
    ReverseEndianNum(const ReverseEndianNum& other) : m_Bits(other.m_Bits) {}
    /* implicit */ ReverseEndianNum(const T val ) : m_Bits( ByteSwap( val ) ) {}

    void    operator = (const ReverseEndianNum& other) { m_Bits = other.m_Bits; }
    /* T */ operator T () const { return ByteSwap( m_Bits ); }
    void    operator = (T val) { m_Bits = ByteSwap( val ); }

    ReverseEndianNum operator+=(T num) { m_Bits = ByteSwap( static_cast<T>(*this) + num ); }
    ReverseEndianNum operator-=(T num) { m_Bits = ByteSwap( static_cast<T>(*this) - num ); }

private:
    T m_Bits;
};

} // namespace endian

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#pragma pack(pop)

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(pop)
#endif

#endif // NW_SND_FND_ENDIAN_H_
