﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_SOUND_EDIT_CONNECTION_H_
#define NW_SND_EDIT_SOUND_EDIT_CONNECTION_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#if defined(NW_PLATFORM_CAFE)
#include <cafe/fs.h>
#endif

#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/fnd/basis/sndfnd_Time.h>
#include <nw/snd/edit/hio/sndedit_HioManager.h>
#include <nw/snd/edit/hio/sndedit_HioSyncChannel.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/edit/handler/sndedit_SyncReplyHandler.h>
#include <nw/snd/edit/handler/sndedit_QueryItemsReplyHandler.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {

namespace internal {
namespace fnd {
class FileStream;
}
}

namespace edit {

class SoundArchiveEditor;

namespace internal {

class IErrorProvider;
class HioSyncChannel;
class HioAsyncChannel;
class QueryItemInfoPacket;

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  サウンド編集のコネクションを管理します。
//---------------------------------------------------------------------------
class SoundEditConnection
{
private: // 定数の定義
#if defined(NW_PLATFORM_CAFE)
    static const u32 MAX_HFIO_VOLUME_PATH = 32;
#endif

public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  初期化パラメータです。
    //---------------------------------------------------------------------------
    struct InitializeArgs
    {
        InitializeArgs() :
#if defined(NW_PLATFORM_CAFE)
            hfioVolumePath(NULL),
#endif
            buffer(NULL),
            bufferLength(0),
            cacheSyncInterval(0),
            sendTimeout(0),
            maxItemName(0),
            hioManager(NULL),
            syncChannel(NULL),
            funcChannel(NULL)
        {
        }

#if defined(NW_PLATFORM_CAFE)
        const char*      hfioVolumePath;    //!< HostFileIO ボリュームパスです。
#endif

        void*            buffer;            //!< SoundEditConnection が使用するバッファです。
        u32              bufferLength;      //!< SoundEditConnection が使用するバッファの長さです。
        u32              cacheSyncInterval; //!< キャッシュの同期間隔（ミリ秒）です。
        u32              sendTimeout;       //!< 送信タイムアウト（ミリ秒）です。
        u32              maxItemName;       //!< アイテム名の長さの上限値です。
        HioManager*      hioManager;        //!< HIO マネージャです。
        HioAsyncChannel* syncChannel;       //!< SYNC 用非同期チャンネルです。
        HioSyncChannel*  funcChannel;       //!< 機能チャンネルです。
    };

public: // コンストラクタ
    SoundEditConnection();

public: // メソッド
    //---------------------------------------------------------------------------
    //! @brief         初期化します。
    //!
    //! @param args TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result Initialize(const InitializeArgs& args);

    //---------------------------------------------------------------------------
    //! @brief         終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief      必要なメモリサイズを取得します。
    //!
    //! @param maxItemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetRequiredMemorySize(u32 maxItemName) const;

    //---------------------------------------------------------------------------
    //! @brief  ファイルを開くのに必要なメモリサイズを取得します。
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    static u32 GetRequiredMemorySizeForOpenFile();

    //---------------------------------------------------------------------------
    //! @brief   サウンド編集コネクションを開きます。
    //!
    //! @param soundArchiveEditor TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result Open(SoundArchiveEditor& soundArchiveEditor);

    //---------------------------------------------------------------------------
    //! @brief   サウンド編集コネクションを閉じます。
    //---------------------------------------------------------------------------
    void Close();

    void Update();

    //---------------------------------------------------------------------------
    //! @brief   アイテムのキャッシュ更新開始を通知します。
    //!
    //! @param itemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result BeginUpdateItemCache(const char* itemName);

    //---------------------------------------------------------------------------
    //! @brief   アイテムのキャッシュ削除開始を通知します。
    //!
    //! @param itemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result BeginRemoveItemCache(const char* itemName);

    //---------------------------------------------------------------------------
    //! @brief   全アイテムのキャッシュ削除開始を通知します。
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result BeginRemoveAllItemCaches();

    //---------------------------------------------------------------------------
    //! @brief   アイテムのキャッシュ更新終了を通知します。
    //!
    //! @param itemCacheResult TBD
    //! @param itemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result EndUpdateItemCache(Result itemCacheResult, const char* itemName);

    //---------------------------------------------------------------------------
    //! @brief   アイテムのキャッシュ削除終了を通知します。
    //!
    //! @param itemCacheResult TBD
    //! @param itemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result EndRemoveItemCache(Result itemCacheResult, const char* itemName);

    //---------------------------------------------------------------------------
    //! @brief   全アイテムのキャッシュ削除終了を通知します。
    //!
    //! @param itemCacheResult TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result EndRemoveAllItemCaches(Result itemCacheResult);

    //---------------------------------------------------------------------------
    //! @brief   アイテム情報を問い合わせます。
    //!
    //! @param replyPacket TBD
    //! @param itemName TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    Result QueryItemInfo(const HioPacket*& replyPacket, const char* itemName);

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief   ファイルを開きます。
    //---------------------------------------------------------------------------
    snd::internal::fnd::FileStream* OpenFile(void* buffer, u32 bufferLength, const char* filePath, FSCmdBlock* fsCmdBlock);
#else
    snd::internal::fnd::FileStream* OpenFile(void* buffer, u32 bufferLength, const char* filePath);
#endif

    //---------------------------------------------------------------------------
    //! @brief  ファイルの内容をすべて読み込みます。
    //!
    //! @param filePath TBD
    //! @param allocator TBD
    //! @param alignment TBD
    //! @param size TBD
    //! @param result TBD
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    const void* ReadFileData(
        const char* filePath,
        snd::internal::fnd::IAllocator& allocator,
        u32 alignment = snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT,
        u32* size = NULL,
        Result* result= NULL);

    //---------------------------------------------------------------------------
    //! @brief         初期化の有無を取得します。
    //!
    //! @return        初期化済みの場合は true、初期化されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_SyncChannel != NULL; }

    //---------------------------------------------------------------------------
    //! @brief         接続の有無を取得します。
    //!
    //! @return        接続済みの場合は true、接続されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsOpened() const
    {
        return m_SoundArchiveEditor != NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief         関連付けられたサウンドアーカイブエディタを取得します。
    //!
    //! @return TBD
    //---------------------------------------------------------------------------
    SoundArchiveEditor* GetSoundArchiveEditor() const
    {
        return m_SoundArchiveEditor;
    }

private: // メソッド
    void SetupSyncChannel(u32 maxItemName);
    void UnsetupSyncChannel();
    bool BeginCacheSync();

private: // メンバ変数
    SoundArchiveEditor* m_SoundArchiveEditor;           //!< 関連するサウンドアーカイブエディタです。
    HioManager*         m_HioManager;                   //!< HIO マネージャです。
    HioAsyncChannel*    m_SyncChannel;                  //!< SYNC 用非同期チャンネルです。
    HioSyncChannel*     m_FuncChannel;                  //!< 機能チャンネルです。

    u32                           m_CacheSyncInterval;  //!< キャッシュの同期間隔です。
    snd::internal::fnd::StopWatch m_CacheSyncStopWatch; //!< キャッシュの同期間隔を示すストップウォッチです。
    u32                           m_SendTimeout;        //!< 送信タイムアウト時間です。

    QueryItemsReplyHandler m_QueryItemsReplyHandler;    //!< QueryItemsReplyPacket のハンドラです。
    HioPacket*             m_SendPacketBuffer;          //!< パケット送信用のバッファです。
    u32                    m_SendPacketBufferSize;      //!< パケット送信用のバッファサイズです。

#if defined(NW_PLATFORM_CAFE)
    char m_HfioVolumePath[MAX_HFIO_VOLUME_PATH];        //!< HostFileIO ボリュームパスです。
#endif
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_SOUND_EDIT_SESSION_H_
