﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_SOUND_ARCHIVE_EDIT_CONTROLLER_H_
#define NW_SND_EDIT_SOUND_ARCHIVE_EDIT_CONTROLLER_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#if defined(NW_PLATFORM_CAFE)
#include <cafe/fs.h>
#endif

#include <nw/snd/fnd/os/sndfnd_CriticalSection.h>
#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/sndedit_Canceller.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>


#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {

class SoundHandle;

namespace internal {

class BasicSound;

namespace fnd {
class FileStream;
}
}

namespace edit {
namespace internal {

class IErrorProvider;
class SoundEditConnection;
class ResourceManager;

//---------------------------------------------------------------------------
//! @brief  リソースを操作するクラスです。
//---------------------------------------------------------------------------
class SoundArchiveEditController
{
public: // 型の定義
    typedef bool (*FILTER_FUNC)(u32 dataType);

    //! @brief  初期化パラメータです。
    struct InitializeArgs
    {
        IErrorProvider*      errorProvider;
        void*                buffer;            //!< OpenFile 用バッファです。
        u32                  bufferLength;      //!< buffer の長さです。
        u32                  maxItemName;       //!< アイテム名の長さの最大値です。
        ResourceManager*     resourceManager;
        SoundArchive*        soundArchive;
        SoundArchivePlayer*  soundArchivePlayer;

        InitializeArgs() :
            errorProvider(NULL),
            buffer(NULL),
            bufferLength(0),
            maxItemName(0),
            resourceManager(NULL),
            soundArchive(NULL),
            soundArchivePlayer(NULL)
        { }
    };

private: // 型の定義
    class ResSoundInfoApplier;

public: // コンストラクタ
    //! @brief  コンストラクタ
    SoundArchiveEditController();

    //! @brief  デストラクタ
    ~SoundArchiveEditController() {}

public: // メソッド
    //! @brief  初期化します。
    //! @param args TBD
    //! @return TBD
    Result Initialize(const InitializeArgs& args);

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  編集を開始します。
    //! @param connection TBD
    //! @return TBD
    Result Start(SoundEditConnection* connection);

    //! @brief  編集を停止します。
    void Stop();

    //! @brief  必要なメモリサイズを取得します。
    //! @param maxItemName TBD
    //! @return TBD
    u32 GetRequiredMemorySize(u32 maxItemName) const;

    //! @brief  合計メモリ使用量を取得します。
    //! @return TBD
    u32 GetTotalMemoryUsage() const;

    //! @brief  現在、メモリ不足状態かどうかを調べます。
    //! @return TBD
    bool IsOutOfMemory() const { return m_MemoryUsageAtOutOfMemory > 0; }

    //! @brief  アイテム数のオーバーフローの有無を取得します。
    //! @return TBD
    bool IsItemInfosOverflow() const { return m_IsItemInfosOverFlow; }

    //! @brief  ファイル数のオーバーフローの有無を取得します。
    //! @return TBD
    bool IsFilesOverflow() const { return m_IsFilesOverFlow; }

    //! @brief  初期化の有無を取得します。
    //! @return TBD
    bool IsInitialized() const { return m_ResourceManager != NULL; }

    //! @brief  編集開始の有無を取得します。
    //! @return TBD
    bool IsStarted() const { return IsInitialized() && m_Connection != NULL; }

    //! @brief  編集アイテムの変更通知コールバックを設定します。
    //! @param callback TBD
    //! @param callbackParam TBD
    void SetEditItemsChangedCallback(EditItemsChangedCallback callback, uptr callbackParam);

    //! @brief  キャッシュ状態をロックします。
    void LockCacheStates();

    //! @brief  キャッシュ状態のロックを解除します。
    void UnlockCacheStates();

    //! @brief  キャッシュを更新します。
    //! @param canceller TBD
    //! @param cancelerParam TBD
    //! @return TBD
    Result UpdateCaches(ICanceller* canceller = NULL, void* cancelerParam = NULL);

    //! @brief  指定した名前のサウンドパラメータ編集情報を取得します。
    //! @param name TBD
    //! @return TBD
    CacheState GetItemCacheState(const char* name) const;

    //! @brief  指定した名前のサウンドパラメータ編集情報を取得します。
    //! @param name TBD
    //! @param filter TBD
    //! @param itemInfoDataType TBD
    //! @return TBD
    const ResSoundInfo* GetSoundInfo(
        const char* name,
        FILTER_FUNC filter = NULL,
        ResDataType* itemInfoDataType = NULL) const;

    //! @brief  指定した名前のバンクパラメータ編集情報を取得します。
    //! @param name TBD
    //! @return TBD
    const ResBankInfo* GetBankInfo(const char* name) const;

    //! @brief  指定バッファを使って指定アイテムラベルに関連するファイルを開きます。
    //! @param buffer TBD
    //! @param bufferLength TBD
    //! @param filePath TBD
    //! @return TBD
    snd::internal::fnd::FileStream* OpenFile(void* buffer, u32 bufferLength, const char* filePath);

    //! @brief  指定したファイルデータを取得します。
    //! @param filePath TBD
    //! @param fileDataType TBD
    //! @param hashCode TBD
    //! @return TBD
    const void* GetFile(
        const char* filePath,
        ResDataType fileDataType,
        const Hash32** hashCode = NULL) const;

    //! @brief  全アイテム情報の更新を遅延処理します。
    void InvokeUpdateAllItemInfos();

    //! @brief  全アイテム情報の削除を遅延処理します。
    void InvokeRemoveAllItemInfos();

    //! @brief  指定アイテム情報の削除を遅延処理します。
    //! @param name TBD
    //! @return TBD
    Result InvokeRemoveItemInfo(const char* name);

    //! @brief  アイテム情報を編集可能状態にします。
    //! @param name TBD
    //! @return TBD
    Result MakeItemInfoEditable(const char* name);

    //! @brief  リソースマネージャを取得します。
    //! @return TBD
    ResourceManager* GetResourceManager() { return m_ResourceManager; }

    //! @brief  リソースマネージャを取得します。
    //! @return TBD
    const ResourceManager* GetResourceManager() const { return m_ResourceManager; }

private: // メソッド
    //! @brief  プラットフォーム向けの初期化を行います。
    Result InitializeForPlatform(snd::internal::fnd::FrameHeap& heap);

    //! @brief  プラットフォーム向けの終了処理を行います。
    void FinalizeForPlatform();

    //! @brief  プラットフォーム向けに必要なメモリサイズを取得します。
    u32 GetRequiredMemorySizeForPlatform() const;

    //! @brief  アイテム名バッファに必要なメモリサイズを取得します。
    u32 GetItemNameBufferRequiredMemorySize(u32 maxItemName) const;

    //! @brief  キャッシュ用に指定バッファを使って指定アイテムラベルに関連するファイルを開きます。
    snd::internal::fnd::FileStream* OpenFileForCache(void* buffer, u32 bufferLength, const char* filePath);

    //! @brief  編集アイテムの変更を通知します。
    void NotifyEditItemsChanged();

    //! @brief  すべてのアイテムを対象としたキャッシュ更新処理を行います。
    void UpdateCachesForAllItems();

    //! @brief  キャッシュを更新します。
    Result UpdateCache(const char* itemName);

    //! @brief  キャッシュを削除します。
    Result RemoveCache(const char* itemName);

    //! @brief  すべてのキャッシュを更新します。
    Result UpdateAllCaches();

    //! @brief  すべてのキャッシュを削除します。
    Result RemoveAllCaches();

    //! @brief  ウェーブサウンドデータをキャッシュします。
    Result UpdateWaveSoundFile(const ResWaveSoundInfo* currentInfo,const ResWaveSoundInfo* newInfo);

    //! @brief  シーケンスサウンドデータをキャッシュします。
    Result UpdateSequenceSoundFile(const ResSequenceSoundInfo* currentInfo,const ResSequenceSoundInfo* newInfo);

    //! @brief  ストリームサウンドデータをキャッシュします。
    Result UpdateStreamSoundFile(const ResStreamSoundInfo* currentInfo,const ResStreamSoundInfo* newInfo);

    //! @brief  バンクデータをキャッシュします。
    Result UpdateBankFile(const ResBankInfo* currentInfo,const ResBankInfo* newInfo);

    //! @brief  指定した名前のアイテム情報を問合せます。
    Result QueryItemInfo(
        const void** data,
        const char* name,
        ResDataType* itemDataType = NULL,
        FILTER_FUNC filter = NULL) const;

    //! @brief  指定ファイルパスを更新します。
    Result UpdateFile(
        const ResName* currentFilePath,
        const ResName* newFilePath,
        ResDataType fileDataType,
        s32 fileSize,
        Hash32* hashCode = NULL);

    Result ReadAndCacheFile(const char* filePath, ResDataType fileDataType, s32 fileSize, Hash32* hashCode = NULL);

    //! @brief  再生中のサウンドパラメータを更新します。
    void UpdatePlayingSoundParameters(const char* soundName);

    //! @brief  再生中のサウンドパラメータを指定サウンド編集情報を使って更新します。
    void UpdatePlayingSoundParameters(
        const char* soundName,
        const internal::ResSoundInfo* soundInfo,
        ResDataType soundInfoType);

private: // メンバ変数
    IErrorProvider*      m_ErrorProvider;       //!< エラープロバイダです。
    SoundEditConnection* m_Connection;
    ResourceManager*     m_ResourceManager;     //!< リソース管理クラスです。
    SoundArchive*        m_SoundArchive;        //!< 編集対象のサウンドアーカイブです。
    SoundArchivePlayer*  m_SoundArchivePlayer;  //!< 編集対象のサウンドアーカイブプレイヤーです。

    u32  m_LockCacheStatesCount;                //!< キャッシュノロックカウントです。

    void* m_OpenFileBuffer;                     //!< OpenFile 用バッファです。
    u32   m_OpenFileBufferLength;               //!< OpenFile 用バッファの長さです。

    char* m_ItemNameBuffer;                     //!< アイテム名格納用バッファです。
    u32   m_ItemNameBufferLength;               //!< アイテム名格納用バッファの長さです。

    u32  m_MemoryUsageAtOutOfMemory;            //!< メモリ不足時のメモリ使用量
    bool m_IsItemInfosOverFlow;                 //!< アイテム数のオーバーフロー状態
    bool m_IsFilesOverFlow;                     //!< ファイル数のオーバーフロー状態
    bool m_IsItemInfoInfoDictionaryInvalid;     //!< アイテム情報辞書の無効化フラグです。

    EditItemsChangedCallback m_EditItemsChangedCallback;       //!< 編集アイテムの変更通知コールバックです。
    uptr                     m_EditItemsChangedCallbackParam;  //!< 編集アイテムの変更通知コールバックに渡すユーザーパラメータです。

    mutable snd::internal::fnd::FndCriticalSection m_UnfreezeCacheLockObject;  //!< キャッシュフリーズ解除のロックオブジェクトです。
    mutable snd::internal::fnd::FndCriticalSection m_CacheLockObject;          //!< キャッシュのロックオブジェクトです。
    mutable snd::internal::fnd::FndCriticalSection m_ErrorItemLockObject;          //!< エラー情報のロックオブジェクトです。

#if defined(NW_PLATFORM_CAFE)
    FSCmdBlock* m_FSCmdBlock;                   //!< ストリーム再生用 CmdBlock です。
    FSCmdBlock* m_FSCmdBlockForCache;           //!< キャッシュ用 CmdBlock です。
#endif
};

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  指定サウンドハンドルにサウンド情報を反映する関数オブジェクトです。
//---------------------------------------------------------------------------
class SoundArchiveEditController::ResSoundInfoApplier
{
public:
    /* ctor */ ResSoundInfoApplier(
        SoundArchive::ItemId soundID,
        const internal::ResSoundInfo* m_SoundEditInfo,
        ResDataType soundInfoType);

    //! @brief  オペレータ() です。
    //! @param soundHandle TBD
    void operator()(SoundHandle& soundHandle);

private:
    void ApplySoundInitialVolume(snd::internal::BasicSound* sound);
    void ApplyStreamSoundTrackInitialVolume(snd::internal::BasicSound* sound);

private:
    SoundArchive::ItemId          m_SoundID;        //!< 編集対象のサウンドID です。
    const internal::ResSoundInfo* m_SoundEditInfo;  //!< サウンドの編集情報です。
    ResDataType                   m_SoundInfoType;  //!< サウンド編集情報のデータ種別です。
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_SOUND_ARCHIVE_EDIT_CONTROLLER_H_

