﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_HIO_PACKET_STREAM_H_
#define NW_SND_EDIT_HIO_PACKET_STREAM_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioProtocol.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace ut {

class IoStream;

} // namespace nw::ut
} // namespace nw

namespace nw {
namespace snd {
namespace edit {
namespace internal {

class HioStream;

//---------------------------------------------------------------------------
//! @brief    HIO パケットストリームクラスです。
//---------------------------------------------------------------------------
class HioPacketStream
{
private:
    enum ReadState
    {
        READ_STATE_NOT_READ = 0,    //!< 何も読み込まれていません。
        READ_STATE_HEADER_READ,     //!< ヘッダを読み込み済みです。
        READ_STATE_BODY_READING,    //!< ボディを読み込み中です。
        READ_STATE_BODY_READ,       //!< ボディを読み込み済みです。
        READ_STATE_ERROR            //!< エラーが発生しました。
    };

public:
    /* ctor */ HioPacketStream();
    /* dtor */ ~HioPacketStream() { }

public:
    //! @brief  初期化します。
    //! @param stream TBD
    //! @param buffer TBD
    //! @param bufferSize TBD
    //! @return TBD
    HioResult Initialize(HioStream& stream, void* buffer, u32 bufferSize);

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  初期化の有無を取得します。
    //! @return TBD
    bool IsInitialized() const { return m_Stream != NULL; }

    //! @brief  ストリームの有効性を取得します。
    //! @return TBD
    bool IsAvailable() const;

    //! @brief  読み込み可能なバイト数を取得します。
    //! @return TBD
    u32 GetReadableBytes() const;

    //! @brief  バッファを取得します。
    //! @return TBD
    void* GetBuffer() const { return m_Buffer; }

    //! @brief  バッファサイズを取得します。
    //! @return TBD
    u32 GetBufferSize() const { return m_BufferSize; }

    //! @brief  読み込み済みのヘッダを取得します。
    //! @return TBD
    const HioPacketHeader* GetReadHeader() const;

    //! @brief  読み込み中のパケットを取得します。
    //! @return TBD
    HioPacket* GetReadingPacket() const;

    //! @brief  読み込み済みのパケットを取得します。
    //! @return TBD
    const HioPacket* GetReadPacket() const;

    //! @brief  読み込み済みのボディサイズを取得します。
    //! @return TBD
    u32 GetReadBodySize() const { return m_ReadBodySize; }

    //! @brief  新しいパケットを読み込み可能かどうかを取得します。
    //! @return TBD
    bool CanReadNewPacket() const;

    //! @brief  パケットボディを読み込み可能かどうかを取得します。
    //! @return TBD
    bool CanReadBody() const;

    //! @brief  パケット全体の読み込みを試みます。
    //! @return TBD
    HioResult TryReadPacket();

    //! @brief  パケットヘッダの取得を試みます。
    //! @return TBD
    HioResult TryReadHeader();

    //! @brief  パケットボディの読み込みを試みます。
    //! @return TBD
    HioResult TryReadBody();

    //! @brief  パケットボディの一部の読み込みます。
    //! @param buffer TBD
    //! @param readLength TBD
    //! @return TBD
    HioResult TryReadBodyPart(void* buffer, u32 readLength);

    //! @brief  パケットボディの一部の読み込みます。
    //!         読み込みが完了するまでの間、ブロッキングします。
    //! @param buffer TBD
    //! @param readLength TBD
    //! @return TBD
    HioResult ReadBodyPart(void* buffer, u32 readLength);

    //! @brief  パケットヘッダをスキップします。
    //!         スキップが完了するまでの間、ブロッキングします。
    //! @return TBD
    HioResult SkipHeader();

    //! @brief  パケットボディをスキップします。
    //!         スキップが完了するまでの間、ブロッキングします。
    //! @return TBD
    HioResult SkipBody();

    //! @brief  パケットボディの一部をスキップします。
    //!         スキップが完了するまでの間、ブロッキングします。
    //! @param length TBD
    //! @return TBD
    HioResult SkipBodyPart(u32 length);

    //! @brief  パケットを書き込みます。
    //! @param packet TBD
    //! @return TBD
    HioResult WritePacket(const HioPacket& packet);

    //! @brief  バッファをクリアします。
    void ClearBuffer();

private:
    ReadState         m_ReadState;          //!< 読み込み状態です。
    HioStream*        m_Stream;             //!< HIO ストリームです。
    HioPacket*        m_Buffer;             //!< 受信用 HioPacket のバッファです。
    u32               m_BufferSize;         //!< 受信用 HioPacket のバッファサイズです。
    u32               m_ReadBodySize;       //!< 読み込み済みのボディサイズです。
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_HIO_PACKET_STREAM_H_
