﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_CTRL_SOUND_CONTROL_SESSION_H_
#define NW_SND_CTRL_SOUND_CONTROL_SESSION_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/basis/sndfnd_Time.h>
#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/edit/hio/sndedit_HioManager.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/ctrl/handler/sndctrl_PlaySoundHandler.h>
#include <nw/snd/ctrl/handler/sndctrl_StopSoundHandler.h>
#include <nw/snd/ctrl/handler/sndctrl_PauseSoundHandler.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {

namespace internal {
namespace fnd {
class FrameHeap;
}
}

namespace ctrl {

class SoundObjectController;
class SoundControlSession;

//---------------------------------------------------------------------------
//! @brief  【β版】サウンドのリモート制御セッションを管理します。
//!
//!         セッションを開くと、SoundMaker からのリモート制御要求を受け付けます。
//!
//!         SoundMaker との通信には MCS を利用します。
//---------------------------------------------------------------------------
class SoundControlSession
{
    NW_DISALLOW_COPY_AND_ASSIGN(SoundControlSession);

public: // 定数の定義
    static const u32 DEFAULT_CHANNEL_STREAM_BUFFER_SIZE = 64 * 1024;   //!< チャンネルバッファサイズのデフォルト値です。
    static const u32 DEFAULT_SYNC_TIMEOUT               = 200;         //!< SYNC タイムアウトのデフォルト値です。
    static const u32 DEFAULT_MAX_ITEM_NAME              = 256;         //!< アイテム名の長さのデフォルト上限値です。

public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  SoundControlSession 設定を格納する構造体です。
    //!
    //!         各パラメータには、コンストラクタでデフォルト値が設定されます。
    //!
    //! @see    Initialize()
    //! @see    GetRequiredMemorySize()
    //---------------------------------------------------------------------------
    struct Configs
    {
        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //---------------------------------------------------------------------------
        Configs() :
            channelStreamBufferSize(DEFAULT_CHANNEL_STREAM_BUFFER_SIZE),
            syncTimeout(DEFAULT_SYNC_TIMEOUT),
            maxItemName(DEFAULT_MAX_ITEM_NAME),
            port(edit::internal::HIO_SNDEDIT_CTRL_CHANNEL)
        {
        }

        u32 channelStreamBufferSize;    //!< チャンネル毎の受信用ストリームのバッファサイズです。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_CHANNEL_STREAM_BUFFER_SIZE が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 syncTimeout;                //!< SYNC タイムアウト（ミリ秒）です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_SYNC_TIMEOUT が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 maxItemName;                //!< アイテム名の長さの上限値です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_MAX_ITEM_NAME が使用されます。
                                        //!< @n
                                        //!< 非常に長いアイテム名を対象にしたい場合や、使用する名前が短いことが事前にわかっていて、
                                        //!< 少しでもメモリ使用量を減らしたい場合に設定を変更してください。
        edit::PORT port;                //!< 内部で使用するポートです。
    };

private: // 型の定義
    enum State
    {
        STATE_NOT_INITIALIZED = 0,  //!< 未初期化状態です。
        STATE_INITIALIZED = 1,      //!< 初期化済み状態です。
        STATE_OPENED                //!< セッションが開かれている状態です。
    };

public: // コンストラクタ
    //===========================================================================
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    SoundControlSession();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~SoundControlSession();

    //@}
    // コンストラクタ/デストラクタ
    //===========================================================================

public: // メソッド
    //===========================================================================
    //! @name 初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      SoundControlSession を初期化します。
    //!
    //!             この関数は、SoundControlSession で利用するメモリを初期化し、
    //!             SoundMaker との接続を開始するための準備を行います。
    //!             @n
    //!             SoundControlSession を利用する前に、必ずこの関数を呼び出してください。
    //!
    //!             引数 buffer は、通信の作業領域として利用されます。
    //!             @n
    //!             SoundControlSession::GetRequiredMemorySize() が返すより大きいサイズを指定する必要があります。
    //!
    //! @param[in]  buffer           通信処理の作業バッファです。
    //! @param[in]  bufferLength     buffer の長さを指定します。
    //! @param[in]  configs          SoundControlSession の設定を指定します。
    //!                              GetRequiredMemorySize() に渡した Configs を指定します。
    //!
    //! @return     処理結果を返します。
    //!
    //! @see        Configs
    //! @see        GetRequiredMemorySize()
    //! @see        Finalize()
    //---------------------------------------------------------------------------
    edit::Result Initialize(
        void* buffer,
        u32 bufferLength,
        const Configs& configs);

    //---------------------------------------------------------------------------
    //! @brief  SoundControlSession の終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief      SoundControlSession の利用に必要なメモリサイズを取得します。
    //!
    //!             SoundControlSession を利用するには、
    //!             この関数で取得したサイズ分のバッファを確保し、
    //!             Initialize() の引数 buffer、bufferLength に指定する必要があります。
    //!             @n
    //!             その際 Initialize() の引数 configs には、
    //!             GetRequiredMemorySize() に渡した Configs と同じものを指定してください。
    //!
    //! @param[in]  configs  SoundControlSession の設定を指定します。
    //!
    //! @return     必要なメモリサイズを返します。
    //!
    //! @see        Configs
    //! @see        Initialize()
    //---------------------------------------------------------------------------
    u32 GetRequiredMemorySize(const Configs& configs) const;

    //@}
    // 初期化
    //===========================================================================

    //===========================================================================
    //! @name 編集対象の登録
    //@{

    //---------------------------------------------------------------------------
    //! @brief      指定 SoundObjectController を登録します。
    //!
    //!             ここで指定した SoundObjectController が SoundMaker と接続する対象となります。
    //!             @n
    //!             SoundObjectController は１つだけ登録できます。
    //!
    //! @param[in]  soundObjectController  編集対象の SoundObjectController を指定します。
    //!
    //! @return     結果を返します。
    //!
    //! @see        UnregisterSoundObjectController()
    //---------------------------------------------------------------------------
    edit::Result RegisterSoundObjectController(SoundObjectController* soundObjectController);

    //---------------------------------------------------------------------------
    //! @brief      指定 SoundObjectController の登録を解除します。
    //!
    //! @param[in]  soundObjectController  編集対象の SoundObjectController を指定します。
    //!
    //! @see        RegisterSoundObjectController()
    //---------------------------------------------------------------------------
    void UnregisterSoundObjectController(SoundObjectController* soundObjectController);

    //@}
    // 編集対象の登録
    //===========================================================================

    //===========================================================================
    //! @name 状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief   初期化の有無を取得します。
    //!
    //! @return  初期化済みの場合は true、初期化されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_State > STATE_NOT_INITIALIZED; }

    //---------------------------------------------------------------------------
    //! @brief  セッションが開かれているかどうかを取得します。
    //!
    //! @return セッションが開かれている場合は true、開らかれていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsOpened() const
    {
        return IsInitialized() && m_State > STATE_INITIALIZED;
    }

    //---------------------------------------------------------------------------
    //! @brief   SoundMaker との接続の有無を取得します。
    //!
    //! @return  SoundMaker と接続済みの場合は true、接続されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsConnected() const
    {
        return IsInitialized() && m_Channel.IsOpened();
    }

    //@}
    // 状態の取得
    //===========================================================================

    //===========================================================================
    //! @name 開始と終了
    //@{

    //---------------------------------------------------------------------------
    //! @brief  サウンドリモート制御セッションを開きます。
    //!
    //!         この関数が呼び出されると、Update() のタイミングで
    //!         SoundMaker との通信が行われるようになります。
    //!
    //! @see    Close()
    //! @see    Update()
    //---------------------------------------------------------------------------
    void Open();

    //---------------------------------------------------------------------------
    //! @brief  サウンドリモート制御セッションを閉じます。
    //!
    //!         SoundMakerとの通信も終了します。
    //!
    //! @see    Open()
    //---------------------------------------------------------------------------
    void Close();

    //---------------------------------------------------------------------------
    //! @brief  サウンドリモート制御セッションを更新します。
    //!
    //!         セッションが開かれている間、SoundMaker からの要求を受け付けます。
    //!
    //!         セッションが開かれていない場合は、この関数を呼び出しても何も処理しません。
    //!
    //! @see    Open()
    //! @see    Close()
    //---------------------------------------------------------------------------
    void Update();

    //@}
    // 開始と終了
    //===========================================================================

private: // メソッド
    //! @brief  チャンネルを初期化します。
    edit::Result InitializeChannel(
        snd::internal::fnd::FrameHeap& allocator,
        u32 recvStreamBufferSize,
        u32 recvPacketBufferSize);

    //! @brief  チャンネルの終了処理を行います。
    void FinalizeChannel();

    //! @brief  メッセージハンドラを初期化します。
    void InitializeHandlers();

    //! @brief  メッセージハンドラの終了処理を行います。
    void FinalizeHandlers();

    //! @brief  チャンネルの受信パケットバッファサイズを取得します。
    u32 GetChannelRecvPacketBufferSize(u32 maxItemName) const;

    //! @brief  通信バッファをクリアします。
    void ClearBuffer();

    //! @brief  同期します。
    void Sync();

    //! @brief 内部で使用する作業バッファのサイズを取得します。
    u32 GetRequiredWorkBufferSize() const;

    //! @brief チャンネル識別番号からプラットフォームごとに必要なチャンネル情報を取得します。
    edit::internal::HioStream::ChannelType GetChannelInfo(edit::internal::HioChannelType channel) const;

private: // メンバ変数
    State m_State;                                          //!< セッションの状態です。
    edit::PORT m_Port;                                      //!< 内部で使用するポートです。

    edit::internal::HioManager      m_HioManager;           //!< HostIOマネージャです。
    edit::internal::HioAsyncChannel m_Channel;              //!< 非同期チャンネルです。

    u32                           m_SyncTimeout;            //!< SYNC タイムアウト時間です。
    snd::internal::fnd::StopWatch m_SyncStopWatch;          //!< SYNC 間隔を示すストップウォッチです。

    internal::PlaySoundHandler  m_PlaySoundHandler;         //!< PlaySoundPacket のハンドラです。
    internal::StopSoundHandler  m_StopSoundHandler;         //!< StopSoundPacket のハンドラです。
    internal::PauseSoundHandler m_PauseSoundHandler;        //!< PauseSoundPacket のハンドラです。

    SoundObjectController* m_SoundObjectController;         //!< サウンドオブジェクトコントローラです。
};

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_CTRL_SOUND_CONTROL_SESSION_H_
