﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_BASE_H_
#define NW_MCS_BASE_H_

#if defined(__CWCC__)
#pragma once
#endif

#include <nw/ut/ut_LinkList.h>
#include <nw/mcs/mcs_SimpleRingBuffer.h>
#include <nw/types.h>

namespace nw
{
namespace mcs
{

//!
//! @namespace nw::mcs
//!
//! @brief Mcs ライブラリの名前空間です。
//!
//! @details
//! Mcs (Multiple Channel Stream) ライブラリは、
//! 実機アプリケーションとホスト PC 間の通信機能を提供する、
//! 開発期間用のライブラリです。
//!
//! 基本的な通信機能と、その上で動作する以下の機能を提供します。
//!
//! ・ファイル入出力\n
//! ・ストリーム入出力\n
//! ・入力デバイス（キーボードとマウス）
//!
//! Mcs ライブラリの関数の一部は、内部で Cafe SDK の Host IO ライブラリの関数を使用します。
//!
//! そのため、マスター ROM に Host IO ライブラリの関数が含まれないように
//! Release 版では Mcs ライブラリの関数は空定義に置き換わります。
//! マスター ROM では Mcs ライブラリの関数は使用しない事を推奨します。
//! (ライブラリをリンクする事は問題ありません)
//!
//! mcs のサンプルデモも Release 版では動作しませんのでご注意ください。

//! @name 基本
//@{

//! @brief Mcs 関数の返り値に使用される列挙子です。
enum Result
{
    //! 関数は成功しました。
    MCS_ERROR_SUCCESS       = 0,
    //! 通信エラーが発生しました。
    MCS_ERROR_COMERROR,
    //! Mcs サーバと接続されていません。
    MCS_ERROR_NOTCONNECT,
    MCS_ERROR_TOO_LARGE_DATA
};

//! チャンネルの指定に使用する型です。
typedef u16 ChannelType;

#if defined(NW_MCS_ENABLE)

namespace internal
{

/* =======================================================================
    Type Definitions
   ======================================================================== */

//---------------------------------------------------------
// Information recorded for each channel
struct ChannelInfo
{
    ut::LinkListNode    link;       // Information for managing a linked list

    u32                 channel;    // Channel number
    SimpleRingBuffer    recvBuf;    // Data receive buffer
    void*               allocatedBuffer; // allocated buffer address.
};

}   // namespace internal

#endif  // defined(NW_MCS_ENABLE)

//---------------------------------------------------------
//! @brief デバイスタイプを表す列挙子です。
enum DEVICETYPE
{
    //! デバイスタイプが不明です。
    DEVICETYPE_UNKNOWN,
    //! PC 上でエミュレーションされているデバイスです。
    DEVICETYPE_PCIO2,
    //! CAT-DEV です。
    DEVICETYPE_CATDEV
};

//---------------------------------------------------------
//! @brief デバイス情報です。
//!
//! @details :category     基本
//---------------------------------------------------------
struct DeviceInfo
{
    u32         deviceType; //!< デバイスタイプ(mcs::DEVICETYPE)です。
    u32         param1;     //!< デバイス情報のパラメータ１。
    u32         param2;     //!< デバイス情報のパラメータ２。
    u32         param3;     //!< デバイス情報のパラメータ３。
};

//@}

namespace internal {

    //-----------------------------------------------------------
    // Pure abstract class that abstracts communications with devices.
    // By deriving from this class, actual processing is described for each device that is a communication target.
    //
    // The library makes it possible to flexibly handle various communication methods and targets by depending on this class while programming.
    //
    class CommDevice
    {
    #if defined(NW_MCS_ENABLE)

    public:
                            CommDevice(){}

        virtual             ~CommDevice() {}

        virtual u32         Open() = 0;

        virtual void        Close() = 0;

        virtual void        RegisterBuffer(
                                ChannelType channel,
                                void*       buf,
                                u32         bufSize
                            ) = 0;

        virtual void*       UnregisterBuffer(ChannelType channel) = 0;

        virtual void*       GetRegisteredBuffer(ChannelType channel) = 0;

        virtual u32         Polling() = 0;

        virtual u32         GetReadableBytes(ChannelType channel) = 0;

        virtual u32         Peek(
                                ChannelType channel,
                                void*       data,
                                u32         size
                            ) = 0;

        virtual u32         Read(
                                ChannelType channel,
                                void*       data,
                                u32         size
                            ) = 0;

        virtual u32         GetWritableBytes(
                                ChannelType channel,
                                u32*        pWritableBytes
                            ) = 0;

        virtual u32         Write(
                                ChannelType channel,
                                const void* data,
                                u32         size
                            ) = 0;

        virtual bool        IsServerConnect() = 0;

        virtual s64         GetServerTime() = 0;

    #endif  // #if defined(NW_MCS_ENABLE)
    };

} // namespace internal

/* =======================================================================
    Function Prototype
   ======================================================================== */

#if defined(NW_MCS_ENABLE)

    //! @name 基本・初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      Mcs ライブラリを初期化します。
    //!
    //! @details
    //! Mcs 関数を使用するときは、
    //! 使用する前に必ずこの関数を呼び出しておく必要があります。
    //! Mcs_Initialize() 自体はスレッドセーフではありません。
    //!
    //! Mcs_Initialize() の呼び出し回数はカウントされます。
    //! Mcs_Finalize() が同じ回数だけ呼び出されなければなりません。
    //!
    //! @sa Mcs_Finalize
    //! @sa Mcs_IsInitialized
    //! @sa Mcs_RegisterBuffer
    //---------------------------------------------------------------------------
    void                    Mcs_Initialize();

    //---------------------------------------------------------------------------
    //! @brief      Mcs ライブラリの終了処理をおこないます。
    //!
    //! @details
    //! 本関数を呼び出すと Mcs_IsInitialized() で取得できる
    //! 初期化回数がデクリメントされます。
    //!
    //! Mcs_Initialize() と同じ回数だけ本関数が呼ばれ、初期化回数が 0 になると、
    //! Mcs ライブラリの終了処理を行います。
    //!
    //! @sa Mcs_Initialize
    //! @sa Mcs_IsInitialized
    //---------------------------------------------------------------------------
    void                    Mcs_Finalize();

    //---------------------------------------------------------------------------
    //! @brief      Mcs ライブラリが初期化されたかどうかの状態を取得します。
    //!
    //! @return     Mcs_Initialize() が呼び出された回数を返します。
    //!
    //! 初期化回数は Mcs_Initialize() でインクリメントされ、
    //! Mcs_Finalize() でデクリメントされます。
    //!
    //! @sa Mcs_Initialize
    //! @sa Mcs_Finalize
    //---------------------------------------------------------------------------
    s32                     Mcs_IsInitialized();

    //---------------------------------------------------------------------------
    //! @brief      デバイス情報を取得します。
    //!
    //! @return     デバイス情報を返します。
    //---------------------------------------------------------------------------
    const DeviceInfo*       Mcs_GetDeviceInfo();

    //---------------------------------------------------------------------------
    //! @brief      データ受信用のバッファを登録します。
    //!
    //! @param[in]  channel     データ送受信の通信相手を一意に識別するための値。\n
    //!                         (0 ～ 0x7FFFの範囲の値)
    //! @param[in]  buf         登録する受信用バッファ。
    //! @param[in]  bufSize     登録する受信用バッファのサイズ。
    //!
    //! @details
    //! 引数 channel の値は 0 ～ 0x7FFFF の範囲でなければなりません。
    //! 0x8000 ～ 0xFFFF は NintendoWare アプリケーションおよび Mcs システム用に
    //! 予約されています。
    //!
    //! 指定したバッファに内部で使用するための情報エリアを確保します。
    //! そのため、バッファのサイズは少なくとも 64 以上である必要があります。
    //!
    //! 受信用バッファが受信データで一杯になり、新規に受信したデータを格納する
    //! だけの空き容量が無い場合は、その受信データは捨てられます。
    //! 従って、通信で使用するデータ量に合わせてバッファサイズを十分な大きさに
    //! 設定する必要があります。
    //!
    //! Mcs_Open() 後にバッファの登録を行った場合、
    //! バッファの登録以前に受信したデータは破棄されます。
    //!
    //! Mcs_Open() 後にバッファの登録を行えるようにするには、
    //! 実機側から通信を開始するようにプロトコルを設計する必要があります。
    //!
    //! @sa Mcs_Initialize
    //! @sa Mcs_Open
    //! @sa Mcs_UnregisterBuffer
    //---------------------------------------------------------------------------
    void                    Mcs_RegisterBuffer(
                                ChannelType channel,
                                void*       buf,
                                u32         bufSize);

    //---------------------------------------------------------------------------
    //! @brief      データ受信用のバッファの登録を解除します。
    //!
    //! @param[in]  channel         登録を解除したいチャンネルの値。\n
    //!                             Mcs_RegisterBuffer() で指定したチャンネル値を指定します。
    //!
    //! @return     Mcs_RegisterBuffer で登録したバッファアドレスを返します。
    //!
    //! @sa Mcs_RegisterBuffer
    //---------------------------------------------------------------------------
    void*                   Mcs_UnregisterBuffer(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief      データ受信用に登録されたバッファを取得します。
    //!
    //! @return     Mcs_RegisterBuffer で登録されたバッファです。
    //!             登録されたバッファが存在しない場合は NULL を返します。
    //---------------------------------------------------------------------------
    void*                   Mcs_GetRegisteredBuffer(ChannelType channel);

    //@}

    //! @name 基本・通信制御

    //---------------------------------------------------------------------------
    //! @brief      通信デバイスをオープンします。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @sa Mcs_Close
    //! @sa Mcs_RegisterBuffer
    //! @sa Mcs_Initialize
    //---------------------------------------------------------------------------
    u32                     Mcs_Open();

    //---------------------------------------------------------------------------
    //! @brief      通信デバイスをクローズします。
    //!
    //! @sa Mcs_Open
    //---------------------------------------------------------------------------
    void                    Mcs_Close();

    //---------------------------------------------------------------------------
    //! @brief      データの送受信動作を行います。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @details
    //! データの送受信動作を行います。
    //! メインループ内で定期的に呼び出すようにしてください。
    //!
    //! @sa Mcs_Open
    //---------------------------------------------------------------------------
    u32                     Mcs_Polling();

    //---------------------------------------------------------------------------
    //! @brief      Mcs サーバとの接続が確認されているかどうかの状態を取得します。
    //!
    //! @return     Mcs サーバとの間で接続が確認されている場合は true、
    //!             そうでない場合は false を返します。
    //!
    //! @details
    //! この関数が返す値は Mcs サーバの状態と完全に同期しているわけではありません。
    //! Mcs の通信機能を使用して状態を取得しているため、若干のタイムラグが生じます。
    //!
    //! 返り値が true のときは、Mcs サーバとの間で接続が確認されています。
    //! 返り値が false の場合は、Mcs サーバと一度も接続していないか、
    //! 接続した後で切断していることを示しています。
    //!
    //! @sa Mcs_Open
    //---------------------------------------------------------------------------
    bool                    Mcs_IsServerConnect();

    //@}

    //! @name 基本・入出力
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ブロックしないで読み込めるデータのサイズを取得します。
    //!
    //! @param[in]  channel     ストリームを識別するための値。
    //!
    //! @return     読み込み可能なデータのサイズ（バイト数）を返します。
    //!
    //! @sa Mcs_Read
    //! @sa Mcs_Peek
    //---------------------------------------------------------------------------
    u32                     Mcs_GetReadableBytes(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief      受信バッファから破棄しないで、取得可能なデータを読み込みます。
    //!
    //! @param[in]  channel     ストリームを識別するための値。
    //! @param[in]  data        読み込むデータを格納するバッファへのポインタ。
    //! @param[in]  size        読み込むデータを格納するバッファのサイズ。
    //!
    //! @return     実際に読み込まれたデータのサイズ(バイト数)を返します。
    //!
    //! @details
    //! 引数 size で指定したサイズが、その時点で受信バッファに格納されている
    //! データサイズより大きい場合は、受信バッファに格納されている
    //! サイズだけ読み込みます。
    //!
    //! @sa Mcs_GetReadableBytes
    //! @sa Mcs_Read
    //---------------------------------------------------------------------------
    u32                     Mcs_Peek(
                                ChannelType channel,
                                void*       data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief      データを読み込みます。
    //!
    //! @param[in]  channel     ストリームを識別するための値。
    //! @param[in]  data        読み込むデータを格納するバッファへのポインタ。
    //! @param[in]  size        読み込むデータを格納するバッファのサイズ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @details
    //! 引数 size で指定したサイズ全てを受信するまで制御を返しません。
    //!
    //! @sa Mcs_GetReadableBytes
    //! @sa Mcs_Peek
    //! @sa Mcs_Skip
    //---------------------------------------------------------------------------
    u32                     Mcs_Read(
                                ChannelType channel,
                                void*       data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief      受信データをスキップします。
    //!
    //! @param[in]  channel     ストリームを識別するための値。
    //! @param[in]  size        スキップするデータのサイズ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @details
    //! 引数 size で指定したサイズ全てをスキップするまで制御を返しません。
    //!
    //! @sa Mcs_GetReadableBytes
    //! @sa Mcs_Peek
    //! @sa Mcs_Read
    //---------------------------------------------------------------------------
    u32                     Mcs_Skip(
                                ChannelType channel,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief      一度に送信可能なサイズを取得します。
    //!
    //! @param[in]  channel         ストリームを識別するための値。
    //! @param[out] pWritableBytes  送信可能なバイト数を格納する変数へのポインタ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @details
    //! 一度に送信可能なサイズを超えたデータを書き込む場合、
    //! 書き込んだデータを PC 側が読み取るまで Mcs_Write() は制御を返しません。
    //!
    //! @sa Mcs_Write
    //---------------------------------------------------------------------------
    u32                     Mcs_GetWritableBytes(
                                ChannelType channel,
                                u32*        pWritableBytes);

    //---------------------------------------------------------------------------
    //! @brief      データを書き込みます。
    //!
    //! @param[in]  channel     ストリームを識別するための値。
    //! @param[in]  data        送信するデータを格納するバッファへのポインタ。
    //! @param[in]  size        送信するデータのサイズ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::Result)を返します。
    //!
    //! @details
    //! 引数 size で指定した送信するデータのサイズが、
    //! Mcs_GetWritableBytes() によって得られる送信可能なサイズを超えている場合、
    //! 書き込むデータ全てが PC 側で読み取られるまで制御を返しません。
    //!
    //! @sa Mcs_GetWritableBytes
    //---------------------------------------------------------------------------
    u32                     Mcs_Write(
                                ChannelType channel,
                                const void* data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @details :private
    //!
    //! @brief      ホストの時間を取得します。
    //!
    //! @return     2000/01/01 基準の msec です。
    //---------------------------------------------------------------------------
    s64                     Mcs_GetServerTime();

    // @}

#else   // #if defined(NW_MCS_ENABLE)

    inline void              Mcs_Initialize()
                             {}

    inline void              Mcs_Finalize()
                             {}

    inline s32               Mcs_IsInitialized()
                             { return false; }

    inline const DeviceInfo* Mcs_GetDeviceInfo() { return NULL; }

    inline void              Mcs_RegisterBuffer(ChannelType /* channel*/, void* /* buf */, u32 /* bufSize */)
                             {}

    inline void*             Mcs_UnregisterBuffer(ChannelType /* channel */)
                             { return NULL; }

    inline void*             Mcs_GetRegisteredBuffer(ChannelType /* channel */) { return NULL; }

    inline u32               Mcs_Open()
                             { return MCS_ERROR_COMERROR; }

    inline void              Mcs_Close()
                             {}

    inline u32               Mcs_Polling()
                             { return MCS_ERROR_COMERROR; }

    inline u32               Mcs_GetReadableBytes(ChannelType /* channel */)
                             { return 0; }

    inline u32               Mcs_Peek(ChannelType /* channel*/, void* /* Data*/, u32 /* size */)
                             { return 0; }

    inline u32               Mcs_Read(ChannelType /* channel*/, void* /* Data*/, u32 /* size */)
                             { return MCS_ERROR_COMERROR; }

    inline u32               Mcs_Skip(ChannelType /* channel*/, u32 /* size */)
                             { return MCS_ERROR_COMERROR; }

    inline u32               Mcs_GetWritableBytes(ChannelType /* channel*/, u32* /* pWritableBytes */)
                             { return MCS_ERROR_COMERROR; }

    inline u32               Mcs_Write(ChannelType /* channel*/, const void* /* Data*/, u32 /* size */)
                             { return MCS_ERROR_COMERROR; }

    inline bool              Mcs_IsServerConnect()
                             { return false; }

    inline s64               Mcs_GetServerTime()
                             { return 0; }

#endif  // #if defined(NW_MCS_ENABLE)

#if defined(NW_MCS_ENABLE)

namespace internal {

    //-----------------------------------------------------------
    // Pure abstract class implemented by classes that process enumerated devices.
    // The Find method for this object is called each time the library enumerates communication devices.
    //
    class IDeviceEnumerate
    {
    public:
        virtual ~IDeviceEnumerate() {}
        virtual bool    Find(const DeviceInfo& deviceInfo) = 0;
    };

    void                    Mcs_EnumDevices(
                                IDeviceEnumerate*   pEnumerate,
                                u32                 flag         = 0
                            );

    u32                     Mcs_GetDeviceObjectMemSize(const DeviceInfo& deviceInfo);

    void*                   Mcs_GetDeviceObjectMem(const DeviceInfo& deviceInfo);

    bool                    Mcs_CreateDeviceObject(
                                const DeviceInfo&   deviceInfo,
                                void*               buf,
                                u32                 bufSize);

    void                    Mcs_DestroyDeviceObject();

    bool                    Mcs_IsDeviceObjectCreated();

} // namespace internal

#endif


}   // namespace mcs
}   // namespace nw

/* NW_MCS_BASE_H_ */
#endif
