﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_TRANSFORM_H_
#define NW_MATH_TRANSFORM_H_

#include <nw/math/math_Config.h>
#include <nw/math/math_Vector2.h>
#include <nw/math/math_Vector3.h>
#include <nw/math/math_Vector4.h>

namespace nw { namespace math {

//---------------------------------------------------------------------------
//    Transform
//    SRTによる位置制御用クラス
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        2次の位置制御の構造体です。
//---------------------------------------------------------------------------
struct Transform2_
{
    VEC2    scale;
    VEC2    rotate;
    VEC2    translate;
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        3次の位置制御の構造体です。
//---------------------------------------------------------------------------
struct Transform3_
{
    VEC3    scale;
    VEC3    rotate;
    VEC3    translate;
};

//---------------------------------------------------------------------------
//! @brief        2次の位置制御クラスです。
//---------------------------------------------------------------------------
struct Transform2 : public Transform2_
{
public:
    typedef Transform2 self_type;
    typedef f32        value_type;
public:
    Transform2() {}
    explicit Transform2(const Transform2_& t)
    {
        scale = t.scale;
        rotate = t.rotate;
        translate = t.translate;
    }
    Transform2(const VEC2& s, const VEC2& r, const VEC2& t)
    {
        scale     = s;
        rotate    = r;
        translate = t;
    }

    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(Transform2)) == 0; }
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(Transform2)) != 0; }


    void Report(bool bNewline = true, const char* name = NULL) const;
};

//---------------------------------------------------------------------------
//! @brief        3次の位置制御クラスです。
//---------------------------------------------------------------------------
struct Transform3 : public Transform3_
{
public:
    typedef Transform3 self_type;
    typedef f32        value_type;
public:
    Transform3() {}
    explicit Transform3(const Transform3_& t)
    {
        scale = t.scale;
        rotate = t.rotate;
        translate = t.translate;
    }
    Transform3(const VEC3& s, const VEC3& r, const VEC3& t)
    {
        scale     = s;
        rotate    = r;
        translate = t;
    }

    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(Transform3)) == 0; }
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(Transform3)) != 0; }

    void Report(bool bNewline = true, const char* name = NULL) const;
};

//---------------------------------------------------------------------------
//! @brief        2次の位置制御クラスに単位行列を設定します。
//!
//! @param[out]   pOut    出力先のバッファです。
//!
//! @return       pOut のポインタを返します。
//---------------------------------------------------------------------------
inline Transform2*
Transform2Identity(Transform2* pOut)
{
    NW_ASSERT_NOT_NULL(pOut);
    pOut->scale     = VEC2(1.f, 1.f);
    pOut->rotate    = VEC2(0.f, 0.f);
    pOut->translate = VEC2(0.f, 0.f);
    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        3次の位置制御クラスに単位行列を設定します。
//!
//! @param[out]   pOut    出力先のバッファです。
//!
//! @return       pOut のポインタを返します。
//---------------------------------------------------------------------------
inline Transform3*
Transform3Identity(Transform3* pOut)
{
    NW_ASSERT_NOT_NULL(pOut);
    pOut->scale     = VEC3(1.f, 1.f, 1.f);
    pOut->rotate    = VEC3(0.f, 0.f, 0.f);
    pOut->translate = VEC3(0.f, 0.f, 0.f);
    return pOut;
}

// (x, y, z, 1)として変換
NW_MATH_INLINE VEC3* VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV);
NW_MATH_INLINE VEC3* VEC3Transform(VEC3* pOut, const MTX34* pM, const VEC3* pV);
NW_MATH_INLINE VEC3* VEC3Transform(VEC3* pOut, const MTX44* pM, const VEC3* pV);

NW_MATH_INLINE VEC4* VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV);
NW_MATH_INLINE VEC3* VEC3TransformArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, s32 count);
NW_MATH_INLINE VEC4* VEC3TransformArray(VEC4* pOut, const MTX44* pM, const VEC3* pV, u32 count);

NW_MATH_INLINE VEC3* VEC3TransformNormal(VEC3* pOut, const MTX34* pM, const VEC3* pV);
NW_MATH_INLINE VEC3* VEC3TransformNormalArray(VEC3* pOutArray, const MTX34* pM, const VEC3* pArray, u32 n);

NW_MATH_INLINE VEC3* VEC3TransformCoord(VEC3* pOut, const MTX44* pM, const VEC3* pV);
NW_MATH_INLINE VEC3* VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count);

NW_MATH_INLINE VEC4* VEC4Transform(VEC4* pOut, const MTX44* pM, const VEC4* pV);
NW_MATH_INLINE VEC4* VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 n);

NW_MATH_INLINE VEC4* VEC4Transform(VEC4* pOut, const MTX34* pM, const VEC4* pV);
NW_MATH_INLINE VEC4* VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 n);


//-- const 引数を参照にしたオーバーロード
inline VEC3* VEC3Transform(VEC3* pOut, const MTX33& m, const VEC3& v) { return VEC3Transform( pOut, &m, &v ); }
inline VEC3* VEC3Transform(VEC3* pOut, const MTX34& m, const VEC3& v) { return VEC3Transform( pOut, &m, &v ); }
inline VEC4* VEC3Transform(VEC4* pOut, const MTX44& m, const VEC3& v) { return VEC3Transform( pOut, &m, &v ); }
inline VEC3* VEC3TransformArray(VEC3* pOutArray, const MTX34& m, const VEC3* pArray, s32 count) { return VEC3TransformArray( pOutArray, &m, pArray, count ); }
inline VEC4* VEC3TransformArray(VEC4* pOutArray, const MTX44& m, const VEC3* pArray, u32 count) { return VEC3TransformArray( pOutArray, &m, pArray, count ); }

inline VEC3* VEC3TransformNormal(VEC3* pOut, const MTX34& m, const VEC3& v) { return VEC3TransformNormal( pOut, &m, &v ); }
inline VEC3* VEC3TransformNormalArray(VEC3* pOutArray, const MTX34& m, const VEC3* pArray, u32 n) { return VEC3TransformNormalArray( pOutArray, &m, pArray, n ); }

inline VEC3* VEC3TransformCoord(VEC3* pOut, const MTX44& m, const VEC3& v) { return VEC3TransformCoord( pOut, &m, &v ); }
inline VEC3* VEC3TransformCoordArray(VEC3* pOutArray, const MTX44& m, const VEC3* pArray, u32 count) { return VEC3TransformCoordArray( pOutArray, &m, pArray, count ); }

inline VEC4* VEC4Transform(VEC4* pOut, const MTX44& m, const VEC4& v) { return VEC4Transform( pOut, &m, &v ); }
inline VEC4* VEC4TransformArray(VEC4* pOutArray, const MTX44& m, const VEC4* pArray, u32 n) { return VEC4TransformArray( pOutArray, &m, pArray, n ); }



NW_MATH_INLINE VEC3&
VEC3::SetTransform(const MTX33& mtx, const VEC3& src)
{
    return *VEC3Transform(this, &mtx, &src);
}

NW_MATH_INLINE VEC3&
VEC3::SetTransform(const MTX34& mtx, const VEC3& src)
{
    return *VEC3Transform(this, &mtx, &src);
}

NW_MATH_INLINE VEC3&
VEC3::SetTransform(const MTX44& mtx, const VEC3& src)
{
    return *VEC3Transform(this, &mtx, &src);
}

NW_MATH_INLINE VEC3&
VEC3::SetTransformNormal(const MTX34& mtx, const VEC3& src)
{
    return *VEC3TransformNormal(this, &mtx, &src);
}


NW_MATH_INLINE VEC4&
VEC4::SetTransform(const MTX34& mtx, const VEC4& src)
{
    return *VEC4Transform(this, &mtx, &src);
}

NW_MATH_INLINE VEC4&
VEC4::SetTransform(const MTX44& mtx, const VEC4& src)
{
    return *VEC4Transform(this, &mtx, &src);
}

}}  // nw::math


/* NW_MATH_TRANSFORM_H_ */
#endif

