﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX33_H_
#define NW_MATH_MATRIX33_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX33;
struct MTX34;

namespace internal { namespace standard {

    VEC3* VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV);
    MTX33* MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2);

    template<typename TMatrix>
    NW_FORCE_INLINE TMatrix*
    MTX33Mult(TMatrix* pOut, const TMatrix* p1, const TMatrix* p2)
    {
        NW_ASSERT_NOT_NULL( p1 );
        NW_ASSERT_NOT_NULL( p2 );
        NW_ASSERT_NOT_NULL( pOut );

        TMatrix mTmp;
        TMatrix* __restrict pDst = (pOut == p1 || pOut == p2) ? &mTmp : pOut;

        pDst->f._00 = p1->f._00 * p2->f._00 + p1->f._01 * p2->f._10 + p1->f._02 * p2->f._20;
        pDst->f._01 = p1->f._00 * p2->f._01 + p1->f._01 * p2->f._11 + p1->f._02 * p2->f._21;
        pDst->f._02 = p1->f._00 * p2->f._02 + p1->f._01 * p2->f._12 + p1->f._02 * p2->f._22;

        pDst->f._10 = p1->f._10 * p2->f._00 + p1->f._11 * p2->f._10 + p1->f._12 * p2->f._20;
        pDst->f._11 = p1->f._10 * p2->f._01 + p1->f._11 * p2->f._11 + p1->f._12 * p2->f._21;
        pDst->f._12 = p1->f._10 * p2->f._02 + p1->f._11 * p2->f._12 + p1->f._12 * p2->f._22;

        pDst->f._20 = p1->f._20 * p2->f._00 + p1->f._21 * p2->f._10 + p1->f._22 * p2->f._20;
        pDst->f._21 = p1->f._20 * p2->f._01 + p1->f._21 * p2->f._11 + p1->f._22 * p2->f._21;
        pDst->f._22 = p1->f._20 * p2->f._02 + p1->f._21 * p2->f._12 + p1->f._22 * p2->f._22;

        if (pDst == &mTmp)
        {
            pOut->f._00 = pDst->f._00; pOut->f._01 = pDst->f._01; pOut->f._02 = pDst->f._02;
            pOut->f._10 = pDst->f._10; pOut->f._11 = pDst->f._11; pOut->f._12 = pDst->f._12;
            pOut->f._20 = pDst->f._20; pOut->f._21 = pDst->f._21; pOut->f._22 = pDst->f._22;
        }

        return pOut;
    }

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    VEC3* VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV);
    MTX33* MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2);

    template<typename TMatrix>
    NW_FORCE_INLINE TMatrix*
    MTX33Mult(TMatrix* pOut, const TMatrix* p1, const TMatrix* p2)
    {
        NW_ASSERT_NOT_NULL( p1 );
        NW_ASSERT_NOT_NULL( p2 );
        NW_ASSERT_NOT_NULL( pOut );

        // d = a . b;
        //
        // { f0 f4 } = { b00 b01  b02 }
        // { f1 f5 }   { b10 b11  b12 }
        // { f2 f6 }   { b20 b21  b22 }

        const f32x2 f0 = __PSQ_LX(p2, offsetof(TMatrix, m[0][0]), 0, 0);
        const f32x2 f1 = __PSQ_LX(p2, offsetof(TMatrix, m[1][0]), 0, 0);
        const f32x2 f2 = __PSQ_LX(p2, offsetof(TMatrix, m[2][0]), 0, 0);
        const f32x2 f4 = __PSQ_LX(p2, offsetof(TMatrix, m[0][2]), 1, 0);
        const f32x2 f5 = __PSQ_LX(p2, offsetof(TMatrix, m[1][2]), 1, 0);
        const f32x2 f6 = __PSQ_LX(p2, offsetof(TMatrix, m[2][2]), 1, 0);

        f32x2 f8, f9, f10;

        // { d00 d01 d02 } = { f9 f10 } := { a00 a01 a02 } . { f0 f4 }
        //                                                   { f1 f5 }
        //                                                   { f2 f6 }

        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[0][0]), 0, 0); // { a00, a01 }
        f9 = __PS_MULS0(f0, f8);
        f10 = __PS_MULS0(f4, f8);
        f9 = __PS_MADDS1(f1, f8, f9);
        f10 = __PS_MADDS1(f5, f8, f10);
        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[0][2]), 1, 0); // { a22 }
        f9 = __PS_MADDS0(f2, f8, f9);
        f10 = __PS_MADDS0(f6, f8, f10);
        __PSQ_STX(pOut, offsetof(TMatrix, m[0][0]), f9, 0, 0);
        __PSQ_STX(pOut, offsetof(TMatrix, m[0][2]), f10, 1, 0);

        //                                                   { f0 f4 }
        // { d10 d11 d12 } = { f9 f10 } := { a10 a11 a12 } . { f1 f5 }
        //                                                   { f2 f6 }

        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[1][0]), 0, 0); // { a10, a11 }
        f9 = __PS_MULS0(f0, f8);
        f10 = __PS_MULS0(f4, f8);
        f9 = __PS_MADDS1(f1, f8, f9);
        f10 = __PS_MADDS1(f5, f8, f10);
        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[1][2]), 1, 0); // { a12 }
        f9 = __PS_MADDS0(f2, f8, f9);
        f10 = __PS_MADDS0(f6, f8, f10);
        __PSQ_STX(pOut, offsetof(TMatrix, m[1][0]), f9, 0, 0);
        __PSQ_STX(pOut, offsetof(TMatrix, m[1][2]), f10, 1, 0);

        //                                                   { f0 f4 }
        //                                                   { f1 f5 }
        // { d20 d21 d22 } = { f9 f10 } := { a20 a21 a22 } . { f2 f6 }

        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[2][0]), 0, 0); // { a20, a21 }
        f9 = __PS_MULS0(f0, f8);
        f10 = __PS_MULS0(f4, f8);
        f9 = __PS_MADDS1(f1, f8, f9);
        f10 = __PS_MADDS1(f5, f8, f10);
        f8 = __PSQ_LX(p1, offsetof(TMatrix, m[2][2]), 1, 0); // { a22 }
        f9 = __PS_MADDS0(f2, f8, f9);
        f10 = __PS_MADDS0(f6, f8, f10);
        __PSQ_STX(pOut, offsetof(TMatrix, m[2][0]), f9, 0, 0);
        __PSQ_STX(pOut, offsetof(TMatrix, m[2][2]), f10, 1, 0);

        return pOut;
    }

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        3x3行列の積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左の行列へのポインタ。
//! @param[in]    p2    右の行列へのポインタ。
//!
//! @return       pOutを返します。
//---------------------------------------------------------------------------
template<typename TMatrix>
NW_FORCE_INLINE TMatrix*
MTX33Mult(TMatrix* pOut, const TMatrix* p1, const TMatrix* p2)
{
    return NW_MATH_IMPL_NS::MTX33Mult(pOut, p1, p2);
}

//@}

//---------------------------------------------------------------------------
//    MTX33用の関数
//---------------------------------------------------------------------------
NW_MATH_INLINE bool   MTX33IsIdentity(const MTX33* p);
NW_MATH_INLINE MTX33* MTX33Copy(MTX33* pOut, const MTX33* p);
NW_MATH_INLINE MTX33* MTX33Zero(MTX33* pOut);
NW_MATH_INLINE MTX33* MTX33Identity(MTX33* pOut);
NW_MATH_INLINE MTX33* MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2);

NW_MATH_INLINE u32 MTX34InvTranspose(MTX33* pOut, const MTX34* p);

NW_MATH_INLINE MTX33* MTX34ToMTX33(MTX33* pOut, const MTX34* pM);
NW_MATH_INLINE MTX34* MTX33ToMTX34(MTX34* pOut, const MTX33* pM);

//========================================================================
//        クラスの定義
//========================================================================

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        3行3列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX33_
{
    struct BaseData
    {
        f32 _00, _01, _02;
        f32 _10, _11, _12;
        f32 _20, _21, _22;
    };

    union
    {
        struct
        {
            f32 _00, _01, _02;
            f32 _10, _11, _12;
            f32 _20, _21, _22;
        };
        BaseData f;
        f32 m[3][3];
        f32 a[9];
        VEC3_ v[3];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//!   @brief 3行3列の行列クラスです。
//---------------------------------------------------------------------------
// 主に法線行列のために存在する
// とりあえず確実に必要になるものだけ実装
struct MTX33 : public MTX33_
{
public:
    static const int ROW_COUNT = 3; //!< 行数です。
    static const int COLUMN_COUNT = 3; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX33& Identity()
    {
        static const MTX33 identity(
            1.0f, 0.0f, 0.0f,
            0.0f, 1.0f, 0.0f,
            0.0f, 0.0f, 1.0f);

        return identity;
    }

    typedef MTX33 self_type; //!< 自分の型です。
    typedef f32   value_type; //!< 要素の型です。
public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX33() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX33(const f32* p) { MTX33Copy(this, reinterpret_cast<const MTX33*>(p)); }
    //! @brief コンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX33 です。
    explicit MTX33(const MTX34& rhs) { MTX34ToMTX33(this, &rhs); }
    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x02 メンバ変数 _02 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    //! @param[in] x12 メンバ変数 _12 の初期化に用いる浮動小数点値です。
    //! @param[in] x20 メンバ変数 _20 の初期化に用いる浮動小数点値です。
    //! @param[in] x21 メンバ変数 _21 の初期化に用いる浮動小数点値です。
    //! @param[in] x22 メンバ変数 _22 の初期化に用いる浮動小数点値です。
    MTX33(f32 x00, f32 x01, f32 x02,
          f32 x10, f32 x11, f32 x12,
          f32 x20, f32 x21, f32 x22)
    {
        f._00 = x00; f._01 = x01; f._02 = x02;
        f._10 = x10; f._11 = x11; f._12 = x12;
        f._20 = x20; f._21 = x21; f._22 = x22;
    }
    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX33));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX33));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX33));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX33));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX33* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX33));
        return reinterpret_cast<MTX33*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX33* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX33));
        return reinterpret_cast<const MTX33*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX33& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX33));
        return *reinterpret_cast<MTX33*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX33& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX33));
        return *reinterpret_cast<const MTX33*>( &fromPtr );
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC3& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC3*>(&this->v[index]);
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC3& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC3*>(&this->v[index]);
    }

    //! @brief VEC3 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC3 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC3 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        column.z = this->m[2][index];
        return column;
    }

    //! @brief VEC3 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC3 です。
    void SetColumn(int index, const VEC3& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
        this->m[2][index] = column.z;
    }
    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 単位行列に設定します。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetIdentity() { return *MTX33Identity(this); }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX33 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX33)) == 0; }

    //! @brief 同値でなければ true を返します
    //!
    //! @param[in] rhs 比較対象となる MTX33 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX33)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX33IsIdentity(this); }

    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX33 Matrix33;  //!< @briefprivate


}  // namespace math
}  // namespace nw


namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
template<typename TMatrix>
inline TMatrix* MTX33Mult(TMatrix* pOut, const TMatrix& m1, const TMatrix& m2) { return MTX33Mult( pOut, &m1, &m2 ); }
inline bool   MTX33IsIdentity(const MTX33& m) { return MTX33IsIdentity( &m ); }
inline MTX33* MTX33Copy(MTX33* pOut, const MTX33& m) { return MTX33Copy( pOut, &m ); }
inline MTX33* MTX33MAdd(MTX33* pOut, f32 t, const MTX33& m1, const MTX33& m2) { return MTX33MAdd( pOut, t, &m1, &m2 ); }
inline u32 MTX34InvTranspose(MTX33* pOut, const MTX34& m) { return MTX34InvTranspose( pOut, &m ); }
inline MTX33* MTX34ToMTX33(MTX33* pOut, const MTX34& m) { return MTX34ToMTX33( pOut, &m ); }
inline MTX34* MTX33ToMTX34(MTX34* pOut, const MTX33& m) { return MTX33ToMTX34( pOut, &m ); }


}  // namespace math
}  // namespace nw


#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix33.ipp>
#endif

#endif
