﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX22_H_
#define NW_MATH_MATRIX22_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX22;
struct MTX23;

namespace internal { namespace standard {

    MTX22* MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    MTX22* MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//-------------------------------------------------------------------------
//    MTX22用の関数
// ------------------------------------------------------------------------
NW_MATH_INLINE MTX22* MTX22Identity(MTX22* pOut);
NW_MATH_INLINE bool   MTX22IsIdentity(const MTX22* p);
NW_MATH_INLINE MTX22* MTX22Copy(MTX22* pOut, const MTX22* p);
NW_MATH_INLINE MTX22* MTX22Zero(MTX22* pOut);
NW_MATH_INLINE MTX22* MTX23ToMTX22(MTX22* pOut, const MTX23* pM);
NW_MATH_INLINE MTX22* MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2);

//========================================================================
//        クラスの定義
//========================================================================

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        2行2列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX22_
{
    struct BaseData
    {
        f32 _00, _01;
        f32 _10, _11;
    };

    union
    {
        struct
        {
            f32 _00, _01;
            f32 _10, _11;
        };
        BaseData f;
        f32 m[2][2];
        f32 a[4];
        VEC2_ v[2];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//! @brief        2行2列の行列の構造体です
//---------------------------------------------------------------------------
struct MTX22 : public MTX22_
{
public:
    static const int ROW_COUNT = 2; //!< 行数です。
    static const int COLUMN_COUNT = 2; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX22& Identity()
    {
        static const MTX22 identity(
            1.0f, 0.0f,
            0.0f, 1.0f);

        return identity;
    }

    typedef MTX22 self_type; //!< 自分の型です
    typedef f32   value_type; //!< 要素の型です。
public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX22() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX22(const f32* p);
    //! @brief コンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX22 です。
    explicit MTX22(const MTX23& rhs);
    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    MTX22(f32 x00, f32 x01,
          f32 x10, f32 x11)
    {
        f._00 = x00; f._01 = x01;
        f._10 = x10; f._11 = x11;
    }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX22));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX22));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX22));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX22));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX22* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX22));
        return reinterpret_cast<MTX22*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX22* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX22));
        return reinterpret_cast<const MTX22*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX22& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX22));
        return *reinterpret_cast<MTX22*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX22& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX22));
        return *reinterpret_cast<const MTX22*>( &fromPtr );
    }

    //! @brief VEC2 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC2& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC2*>(&this->v[index]);
    }

    //! @brief VEC2 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC2& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC2*>(&this->v[index]);
    }

    //! @brief VEC2 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC2 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC2 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        return column;
    }

    //! @brief VEC2 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC2 です。
    void SetColumn(int index, const VEC2& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
    }
    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 単位行列に設定します。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetIdentity() { return *MTX22Identity(this); }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX22 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX22)) == 0; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX22 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX22)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX22IsIdentity(this); }

    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX22 Matrix22;  //!< @briefprivate

}  // namespace math
}  // namespace nw

namespace nw {
namespace math {

inline
MTX22::MTX22(const f32* p) { MTX22Copy(this, reinterpret_cast<const MTX22*>(p)); }

inline
MTX22::MTX22(const MTX23& rhs) { MTX23ToMTX22(this, &rhs); }

//-- const 引数を参照にしたオーバーロード
inline bool   MTX22IsIdentity(const MTX22& m) { return MTX22IsIdentity( &m ); }
inline MTX22* MTX22Copy(MTX22* pOut, const MTX22& m) { return MTX22Copy(pOut, &m); }
inline MTX22* MTX23ToMTX22(MTX22* pOut, const MTX23& m) { return MTX23ToMTX22(pOut, &m); }
inline MTX22* MTX22MAdd(MTX22* pOut, f32 t, const MTX22& m1, const MTX22& m2) { return MTX22MAdd(pOut, t, &m1, &m2); }

}  // namespace math
}  // namespace nw

#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix22.ipp>
#endif


#endif
