﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//        MTX22
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_MATH_INLINE MTX22*
MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = t * p1->f._00 + p2->f._00;
    pOut->f._01 = t * p1->f._01 + p2->f._01;

    pOut->f._10 = t * p1->f._10 + p2->f._10;
    pOut->f._11 = t * p1->f._11 + p2->f._11;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE MTX22*
MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_MADDS0F(tof32x2(p1->a[0]), t, tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_MADDS0F(tof32x2(p1->a[2]), t, tof32x2(p2->a[2]));

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        行列が単位行列かどうか判定します。
//!
//! @param[in]    p  判定対象の行列へのポインタ。
//!
//! @return       p が単位行列であれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
MTX22IsIdentity(const MTX22* p)
{
    return p->f._00 == 1.f && p->f._01 == 0.f &&
           p->f._10 == 0.f && p->f._11 == 1.f;
}

//---------------------------------------------------------------------------
//! @brief        2x2 行列をコピーします。
//!
//! @param[out]   pOut    コピー先の行列のポインタです。
//! @param[in]    p       コピー元の行列のポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX22*
MTX22Copy(MTX22* pOut, const MTX22* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    if (pOut != p)
    {
        *pOut = *p;
    }

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ゼロ行列を作成します。
//!
//! @param[out]   pOut    ゼロ行列を格納するバッファへのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX22*
MTX22Zero(MTX22* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    pOut->f._00 = pOut->f._01 =
    pOut->f._10 = pOut->f._11 = 0.f;
    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        2x2 の単位行列を作成します。
//!
//! @param[out]   pOut    単位行列を格納するバッファへのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX22*
MTX22Identity(MTX22* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    MTX22Copy(pOut, MTX22::Identity());

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列を実数倍して、別の行列を足します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指しても構いません。
//! @param[in]    t       かける数です。
//! @param[in]    p1      元の行列へのポインタです。
//! @param[in]    p2      足す行列へのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX22*
MTX22MAdd(MTX22* pOut, f32 t, const MTX22* p1, const MTX22* p2)
{
    return NW_MATH_IMPL_NS::MTX22MAdd(pOut, t, p1, p2);
}

//@}

}  // namespace math
}  // namespace nw
