﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_TEXT_SEARCHER_H_
#define NW_LYT_TEXT_SEARCHER_H_

#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{

class Layout;
class TextBox;

//---------------------------------------------------------------------------
//! @brief テキストIDからテキストの検索を行う基底クラスです。
//!
//! @details
//! 純粋仮想クラスとなっています。
//!
//---------------------------------------------------------------------------
class TextSearcher
{
public:
    /// テキストの情報を保持する構造体です。
    struct TextInfo {
        const char16* text;     ///< テキストです。NULLだった場合は、レイアウトエディタで指定された文字列をセットします。
                                ///< テキストボックスを空にする場合は、空文字を指定してください。
        u32 textLength;         ///< テキストの長さです。バイト数ではなく文字数で指定してください。0のときはtextがnull終端される箇所までの長さを設定します。
        u32 bufferLength;       ///< テキストを保持するために確保するバッファの長さです。バイト数ではなく文字数で指定してください。
                                ///< 0のときは、textLengthで指定された長さを設定します。レイアウトエディタの「文字列の長さを強制的に指定する」機能で指定
                                ///< された長さを使いたい場合、forceAssignedTextLengthをコピーしてください。
        s32 forceAssignedTextLength;    ///< レイアウトエディタで「文字列の長さを強制的に指定する」のチェックが入っている場合、このメンバに指定された
                                        ///< 文字数が入った状態でSearchTextに渡されます。チェックが入っていない場合は-1が設定されます。
                                        ///< 出力としては使われません。

        TextInfo() : text(NULL), textLength(0), bufferLength(0), forceAssignedTextLength(-1) {;}
    };

    //! @brief デストラクタです。
    //!
    virtual ~TextSearcher() {}

    //! @brief テキストIDからテキストの検索を行います。
    //!
    //! @details
    //! LayoutクラスのBuildメソッド内で、引数にTextSearcherを設定したときに、TextBoxそれぞれについて呼び出されます。
    //!
    //! TextBoxにテキストIDが設定していない場合も呼び出されますのでご注意ください。その場合、引数messageIdはNULLと
    //! なります。
    //!
    //! 指定されたテキストIDに対応するテキストを探し、引数textInfoに設定してください。TextInfoの引数textを設定せずに
    //! returnした場合は、レイアウトエディタで指定された文字列がテキストボックスに入ります。
    //!
    //! @param[in,out] textInfo     テキストの設定情報。メンバforceAssignedTextLengthは入力で、それ以外は設定の出力となります。
    //!                             設定方法については @ref TextInfo をご参照ください。
    //! @param[in] messageId        テキストボックスに設定されたテキストID。テキストIDが設定されていない場合はNULLとなります。
    //! @param[in] layout           テキストボックスを保持するレイアウト。ただし構築中であることにご注意ください。GetNameは使用できます。
    //! @param[in] textBox          テキストを設定するテキストボックス
    //! @param[in] resourceLayout   テキストボックスを構築するときに使用しているリソースを保持しているレイアウト。
    //!                             部品のプロパティを使用して上書きしたときに、その上書きを行ったレイアウトのポインタが渡されます。
    //!                             それ以外の場合は、引数layoutと同じポインタが渡されます。
    //!
    virtual void SearchText(TextInfo* textInfo, const char* messageId, Layout* layout, TextBox* textBox, Layout* resourceLayout) = 0;

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_BOUNDING_H_

