﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_TEXMAP_H_
#define NW_LYT_TEXMAP_H_

#include <nw/types.h>
#include <nw/lyt/lyt_Types.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <gl/glew.h>
#endif

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace lyt
{

namespace res
{
    struct Image;
}

class TextureInfo;

//---------------------------------------------------------------------------
//! @brief テクスチャマップを制御するクラスです。
//!
//! @details
//! このクラスは通常、Materialクラスによって生成されます。
//! ライブラリユーザ側で生成する必要はありません。
//---------------------------------------------------------------------------
class TexMap
{
public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    TexMap();

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定のテクスチャ情報で初期化します。
    //!
    //! @param[in] textureInfo  テクスチャ情報です。
    //!
    explicit TexMap(const TextureInfo* textureInfo);

    //! @brief デストラクタです。
    //!
    ~TexMap();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テクスチャオブジェクトを取得します。
    //!
    //! @return テクスチャ名ブジェクトのハンドルを返します。
    //!
    u32 GetTextureObject() const
    {
        return m_TexObject;
    }

    //! @brief テクスチャオブジェクトを設定します。
    //!
    //! @param[in] texObject    テクスチャオブジェクトのハンドルです。
    //!
    void SetTextureObject(u32 texObject)
    {
        m_TexObject = texObject;
    }

    //! @brief テクスチャの S 方向のラップモードを取得します。
    //!
    //! @return S 方向のラップモードを返します。
    //!
    TexWrap GetWrapModeS() const
    {
        return TexWrap(m_Bits.wrapS);
    }

    //! @brief テクスチャの T 方向のラップモードを取得します。
    //!
    //! @return T 方向のラップモードを返します。
    //!
    TexWrap GetWrapModeT() const
    {
        return TexWrap(m_Bits.wrapT);
    }

    //! @brief テクスチャのラップモードを設定します。
    //!
    //! @param[in] wrapS    テクスチャの S 方向のラップモードです。
    //! @param[in] wrapT    テクスチャの T 方向のラップモードです。
    //!
    void SetWrapMode(TexWrap wrapS, TexWrap wrapT);

    //! @brief テクスチャが縮小されるときに適用されるフィルタモードを取得します。
    //!
    //! @return テクスチャが縮小されるときに適用されるフィルタモードを返します。
    //!
    TexFilter GetMinFilter() const
    {
        return TexFilter(m_Bits.minFilter);
    }

    //! @brief テクスチャが拡大されるときに適用されるフィルタモードを取得します。
    //!
    //! @return テクスチャが拡大されるときに適用されるフィルタモードを返します。
    //!
    TexFilter GetMagFilter() const
    {
        return TexFilter(m_Bits.magFilter);
    }

    //! @brief テクスチャのフィルタモードを設定します。
    //!
    //! @param[in] minFlt   テクスチャが縮小されるときに適用されるフィルタモードです。
    //! @param[in] magFlt   テクスチャが拡大されるときに適用されるフィルタモードです。
    //!
    void SetFilter(TexFilter minFlt, TexFilter magFlt);

    //! @brief テクスチャの幅を取得します。
    //!
    //! @return テクスチャの幅を返します。
    //!
    u16 GetWidth() const
    {
        return m_Width;
    }

    //! @brief テクスチャの高さを取得します。
    //!
    //! @return テクスチャの高さを取得します。
    //!
    u16 GetHeight() const
    {
        return m_Height;
    }

    //! @brief テクスチャのサイズ(幅、高さ)を取得します。
    //!
    //! @return テクスチャのサイズ(幅、高さ)を返します。
    //!
    const TexSize GetSize() const
    {
        return TexSize(m_Width, m_Height);
    }

    //! @brief テクスチャのサイズ(幅、高さ)を設定します。
    //!
    //! @param[in] width    テクスチャの幅です。
    //! @param[in] height   テクスチャの高さです。
    //!
    void SetSize(u16 width, u16 height)
    {
        m_Width = width;
        m_Height = height;
    }

    //! @brief テクスチャのサイズ(幅、高さ)を設定します。
    //!
    //! @param[in] size テクスチャのサイズ(幅、高さ)です。
    //!
    void SetSize(const TexSize& size)
    {
        SetSize(size.width, size.height);
    }

    //! @brief テクスチャのフォーマットを取得します。
    //!
    //! @return テクスチャのフォーマットを返します。
    //!
    TexFormat GetFormat() const
    {
        return TexFormat(m_Bits.format);
    }

    //! @brief テクスチャのフォーマットを設定します。
    //!
    //! @param[in] format   テクスチャのフォーマットです。
    //!
    void SetFormat(TexFormat format)
    {
        m_Bits.format = format;
    }

    //! @brief テクスチャの情報を設定します。
    //!
    //! @details
    //! TextureInfo の値をコピーします。
    //!
    //! @param[in] src  コピー元です。
    //!
    void Set(const TextureInfo* src);

    //! @brief テクスチャの情報を取得します。
    //!
    //! @details
    //! Cafe版の場合は、コピー先のTextureInfoの中のGX2Textureの実体に、
    //! 内部で保持しているGX2Textureのポインタが指しているのと同じ
    //! 情報をコピーします。
    //!
    //! @param[in] dst  テクスチャの情報を受け取るオブジェクトです。
    //!
    void GetTextureInfo(TextureInfo* dst) const;

    //! @brief テクスチャの情報をコピーします。
    //!
    //! @details
    //! Cafe版の場合は、コピー元のTexMapに設定したTextureInfoのGX2Textureの
    //! ポインタをそのままコピーしますので、TextureInfo内のGX2Textureの実体を
    //! コピー元と共有することになります。
    //!
    //! @param[in] src  テクスチャの情報のコピー元です。
    //!
    void CopyTextureInfo(const TexMap& src);

    //! @brief サンプラーオブジェクトを取得します。
    //!
    //! @return サンプラーを返します。
    //!
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    GLuint GetSampler() const { return m_Sampler; }
#elif defined(NW_PLATFORM_CAFE)
    const GX2Sampler& GetSampler() const { return m_Sampler; }
#endif

#if defined(NW_PLATFORM_CAFE)
    //! @brief GX2 のテクスチャを取得します。
    //!
    //! @return テクスチャを返します。
    //!
    const GX2Texture* GetTexture() const { return m_Texture; }
#endif
    //@}

private:
    u32 m_TexObject;
    u16 m_Width;
    u16 m_Height;

    //! @brief 内部用機能のため使用禁止です。
    struct Bits
    {
        u32 wrapS: 2;
        u32 wrapT: 2;
        u32 minFilter: 3;
        u32 magFilter: 1;
        u32 format: 8;
    };

    Bits m_Bits;

    void InitSampler();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    GLuint m_Sampler;
#elif defined(NW_PLATFORM_CAFE)
    GX2Sampler m_Sampler;
    const GX2Texture* m_Texture;
#endif

    // コピーを禁止します。
    TexMap( const TexMap& );
    const TexMap& operator=( const TexMap& );

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_TEXMAP_H_
