﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_RESOURCES_H_
#define NW_LYT_RESOURCES_H_

#include <nw/types.h>

#include <nw/math/math_types.h>
#include <nw/ut/ut_Color.h>

#include <nw/lyt/lyt_Types.h>

//! @namespace nw::lyt::res
//!
//! @brief レイアウトのリソースデータ型を定義する名前空間です。
//!
//! @details
//! リソースファイルの構造については以下を参照してください。
//!
//! ・ nw::lyt::res::Lyt\n
//! ・ nw::lyt::res::Lan\n
//! ・ nw::lyt::res::Lim

namespace nw
{
namespace lyt
{
namespace res
{

/* ========================================================================
    定数定義
   ======================================================================== */

//! @name バージョン
//@{

//! @brief リソースのメジャーバージョンです。
//!
//! @details
//! 互換性のない変更を表します。
//!
const u8    MajorVersion  = 5;

//! @brief リソースのマイナーバージョンです。
//!
//! @details
//! 後方互換のある変更を表します。新しいライブラリは古いリソースを解釈できます。
//!
const u8    MinorVersion  = 3;

//! @brief リソースのマイクロバージョンです。
//!
//! @details
//! 両互換の変更を表します。新旧のライブラリでお互いのリソースを解釈できます。
//!
const u8    MicroVersion  = 0;

//! @brief リソースのバグフィックスバージョンです。
//!
//! @details
//! 前方互換のみの変更を表します。新しいライブラリで古いリソースが解釈できないことを表します。
//!
const u8    BugfixVersion = 0;

//! @brief リソースのバージョンです。
const u32   BinaryFileFormatVersion     = NW_UT_MAKE_VERSION(MajorVersion, MinorVersion, MicroVersion, BugfixVersion);

//! @brief リソースのバージョンです。(コンバーターから参照されています。nw ライブラリへの依存がなくなったら削除してください)
const u32   BinaryFileFormatVersionUi2d = NW_UT_MAKE_VERSION(6, 1, 0, 0);

//@}

//! @name レイアウト・リソース
//@{

//! @brief レイアウト・リソースファイルのシグニチャです。
//! @sa Lyt
const u32   FILESIGNATURE_FLYT          = NW_UT_MAKE_SIGWORD('F', 'L', 'Y', 'T');

//! @brief Layout データブロックのシグニチャです。
const u32   DATABLOCKKIND_LAYOUT        = NW_UT_MAKE_SIGWORD('l', 'y', 't', '1');
//! @brief Control データブロックのシグニチャです。
const u32   DATABLOCKKIND_CONTROL       = NW_UT_MAKE_SIGWORD('c', 'n', 't', '1');
//! @brief PaneBegin データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANEBEGIN     = NW_UT_MAKE_SIGWORD('p', 'a', 's', '1');
//! @brief PaneEnd データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANEEND       = NW_UT_MAKE_SIGWORD('p', 'a', 'e', '1');
//! @brief Pane データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANE          = NW_UT_MAKE_SIGWORD('p', 'a', 'n', '1');
//! @brief Picture データブロックのシグニチャです。
const u32   DATABLOCKKIND_PICTURE       = NW_UT_MAKE_SIGWORD('p', 'i', 'c', '1');
//! @brief TextBox データブロックのシグニチャです。
const u32   DATABLOCKKIND_TEXTBOX       = NW_UT_MAKE_SIGWORD('t', 'x', 't', '1');
//! @brief Window データブロックのシグニチャです。
const u32   DATABLOCKKIND_WINDOW        = NW_UT_MAKE_SIGWORD('w', 'n', 'd', '1');
//! @brief Bounding データブロックのシグニチャです。
const u32   DATABLOCKKIND_BOUNDING      = NW_UT_MAKE_SIGWORD('b', 'n', 'd', '1');
//! @brief Parts データブロックのシグニチャです。
const u32   DATABLOCKKIND_PARTS         = NW_UT_MAKE_SIGWORD('p', 'r', 't', '1');
//! @brief GroupBegin データブロックのシグニチャです。
const u32   DATABLOCKKIND_GROUPBEGIN    = NW_UT_MAKE_SIGWORD('g', 'r', 's', '1');
//! @brief GroupEnd データブロックのシグニチャです。
const u32   DATABLOCKKIND_GROUPEND      = NW_UT_MAKE_SIGWORD('g', 'r', 'e', '1');
//! @brief Group データブロックのシグニチャです。
const u32   DATABLOCKKIND_GROUP         = NW_UT_MAKE_SIGWORD('g', 'r', 'p', '1');
//! @brief FontList データブロックのシグニチャです。
const u32   DATABLOCKKIND_FONTLIST      = NW_UT_MAKE_SIGWORD('f', 'n', 'l', '1');
//! @brief TextureList データブロックのシグニチャです。
const u32   DATABLOCKKIND_TEXTURELIST   = NW_UT_MAKE_SIGWORD('t', 'x', 'l', '1');
//! @brief MaterialList データブロックのシグニチャです。
const u32   DATABLOCKKIND_MATERIALLIST  = NW_UT_MAKE_SIGWORD('m', 'a', 't', '1');
//! @brief ExtUserDataList データブロックのシグニチャです。
const u32   DATABLOCKKIND_USERDATALIST  = NW_UT_MAKE_SIGWORD('u', 's', 'd', '1');

//@}

//! @name アニメーション・リソース
//@{

//! @brief アニメーション・リソースファイルのシグニチャです。
//! @sa Lan
const u32   FILESIGNATURE_FLAN          = NW_UT_MAKE_SIGWORD('F', 'L', 'A', 'N');

//! @brief AnimationTagBlock データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANEANIMTAG   = NW_UT_MAKE_SIGWORD('p', 'a', 't', '1');
//! @brief AnimationShareBlock データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANEANIMSHARE = NW_UT_MAKE_SIGWORD('p', 'a', 'h', '1');
//! @brief AnimationBlock データブロックのシグニチャです。
const u32   DATABLOCKKIND_PANEANIMINFO  = NW_UT_MAKE_SIGWORD('p', 'a', 'i', '1');

//@}

//! @name アニメーション・タイプ
//@{

//! @brief アニメーションの対象はペインのSRTアニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_PANESRT       = NW_UT_MAKE_SIGWORD('F', 'L', 'P', 'A');
//! @brief アニメーションの対象はビジビリティ・アニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_VISIBILITY    = NW_UT_MAKE_SIGWORD('F', 'L', 'V', 'I');
//! @brief アニメーションの対象は頂点カラー・アニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_VTXCOLOR      = NW_UT_MAKE_SIGWORD('F', 'L', 'V', 'C');
//! @brief アニメーションの対象はマテリアル・アニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_MATCOLOR      = NW_UT_MAKE_SIGWORD('F', 'L', 'M', 'C');
//! @brief アニメーションの対象はテクスチャのSRTアニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_TEXSRT        = NW_UT_MAKE_SIGWORD('F', 'L', 'T', 'S');
//! @brief アニメーションの対象はテクスチャパターン・アニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_TEXPATTERN    = NW_UT_MAKE_SIGWORD('F', 'L', 'T', 'P');
//! @brief アニメーションの対象はインダイレクトのSRTです。(現状Transはサポートしていません)
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_INDIRECTSRT   = NW_UT_MAKE_SIGWORD('F', 'L', 'I', 'M');
//! @brief アニメーションの対象はアルファコンペアです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_ALPHACOMPARE   = NW_UT_MAKE_SIGWORD('F', 'L', 'A', 'C');
//! @brief アニメーションの対象はフォント影のパラメータです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_FONTSHADOW   = NW_UT_MAKE_SIGWORD('F', 'L', 'F', 'S');
//! @brief アニメーションの対象は文字単位アニメーションのパラメータです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_PERCHARACTERTRANSFORM   = NW_UT_MAKE_SIGWORD('F', 'L', 'C', 'T');
//! @brief アニメーションの対象は文字単位アニメーションカーブのパラメータです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_PERCHARACTERTRANSFORMCURVE    = NW_UT_MAKE_SIGWORD('F', 'L', 'C', 'C');
//! @brief アニメーションの対象は拡張ユーザーデータアニメーションです。
//! @sa AnimationInfo
const u32   ANIMATIONTYPE_EXTUSERDATA   = NW_UT_MAKE_SIGWORD('F', 'L', 'E', 'U');

//@}

//! @name テクスチャイメージ・リソース
//@{

//! @brief テクスチャイメージ・リソースファイルのシグニチャです。
//! @sa Lim
const u32   FILESIGNATURE_FLIM          = NW_UT_MAKE_SIGWORD('F', 'L', 'I', 'M');

//! @brief res::Image データブロックのシグニチャです。
const u32   DATABLOCKKIND_IMAGE         = NW_UT_MAKE_SIGWORD('i', 'm', 'a', 'g');

//@}

//! @name リソースタイプ
//@{

//! @brief リソースはレイアウトです。
//! @sa ResType
const u32   RESOURCETYPE_LAYOUT         = 'blyt';
//! @brief リソースはアニメーションです。
//! @sa ResType
const u32   RESOURCETYPE_ANIMATION      = 'anim';
//! @brief リソースはテクスチャイメージです。
//! @sa ResType
const u32   RESOURCETYPE_TEXTURE        = 'timg';
//! @brief リソースはリソースフォントです。
//! @sa ResType
const u32   RESOURCETYPE_FONT           = 'font';
//! @brief リソースはアーカイブフォントです。
//! @sa ResType
const u32   RESOURCETYPE_ARCHIVEFONT    = 'fnta';
//! @brief リソースはシェーダイメージです。
//! @sa ResType
const u32   RESOURCETYPE_SHADER         = 'bgsh';

//@}

/* ========================================================================
    型定義
   ======================================================================== */

//!--------------------------------------------------------------------------*
//! @name レイアウト・リソース
//@{

//---------------------------------------------------------------------------
//! @brief リソース内で２次元ベクトルを保持します。
//!
//---------------------------------------------------------------------------
struct Vec2
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    Vec2()
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @param[in] x    X座標の値です。
    //! @param[in] y    Y座標の値です。
    //!
    Vec2(f32 x, f32 y)
    {
        this->x = x;
        this->y = y;
    }

    //@}

    //! @brief 変換オペレータです。
    //!
    operator const math::VEC2() const
    {
        return math::VEC2(x, y);
    }

    //! X座標の値です。
    ut::ResF32 x;
    //! Y座標の値です。
    ut::ResF32 y;
};

//---------------------------------------------------------------------------
//! @brief リソース内で３次元ベクトルを保持します。
//!
//---------------------------------------------------------------------------
struct Vec3
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    Vec3()
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @param[in] x    X座標の値です。
    //! @param[in] y    Y座標の値です。
    //! @param[in] z    Z座標の値です。
    //!
    Vec3(f32 x, f32 y, f32 z)
    {
        this->x = x;
        this->y = y;
        this->z = z;
    }

    //@}

    //! @brief 変換オペレータです。
    //!
    operator const math::VEC3() const
    {
        return math::VEC3(x, y, z);
    }

    //! X座標の値です。
    ut::ResF32 x;

    //! Y座標の値です。
    ut::ResF32 y;

    //! Z座標の値です。
    ut::ResF32 z;
};

//---------------------------------------------------------------------------
//! @brief テクスチャ SRT 変換行列の構造体です。
//!
//---------------------------------------------------------------------------
struct TexSRT
{
    //! 平行移動です。
    res::Vec2 translate;
    //! 回転です。
    ut::ResF32 rotate;
    //! 拡大率です。
    res::Vec2 scale;
};

//---------------------------------------------------------------------------
//! @brief 投影テクスチャ座標生成の構造体です。
//!
//---------------------------------------------------------------------------
struct ProjectionTexGenParamaters
{
    //! 平行移動です。
    res::Vec2 translate;
    //! 拡大率です。
    res::Vec2 scale;

    //! フラグです(TexProjectionFlag)。
    ut::ResU8   flag;
    //! （未使用）
    ut::ResU8   reserve[3];
};

//---------------------------------------------------------------------------
//! @brief テクスチャ座標生成に設定する値を持つ構造体です。
//!
//---------------------------------------------------------------------------
struct TexCoordGen
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    TexCoordGen()
    {
        Set(TEXGENTYPE_MTX2x4, TEXGENSRC_TEX0);
    }

    //! @brief コンストラクタです。
    //!
    //! @param[in] aTexGenType  テクスチャ座標の生成方法です。
    //! @param[in] aTexGenSrc   テクスチャ座標のソースです。
    //!
    TexCoordGen(
        TexGenType      aTexGenType,
        TexGenSrc       aTexGenSrc
    )
    {
        Set(aTexGenType, aTexGenSrc);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テクスチャ座標生成の設定を変更します。
    //!
    //! @param[in] aTexGenType  テクスチャ座標の生成方法です。
    //! @param[in] aTexGenSrc   テクスチャ座標のソースです。
    //!
    void Set(
        TexGenType    aTexGenType,
        TexGenSrc     aTexGenSrc
                        )
    {
        texGenType = static_cast<ut::ResU8>(aTexGenType);
        texGenSrc = static_cast<ut::ResU8>(aTexGenSrc);
    }

    //! @brief テクスチャ座標の生成方法を取得します。
    //!
    //! @return テクスチャ座標の生成方法を返します。
    //!
    TexGenType GetTexGenType() const
    {
        return TexGenType(texGenType);
    }

    //! @brief テクスチャ座標のソースを取得します。
    //!
    //! @return テクスチャ座標のソースを返します。
    //!
    TexGenSrc GetTexGenSrc() const
    {
        return TexGenSrc(texGenSrc);
    }

    //! @brief 投影テクスチャ座標かどうかを取得します。
    //!
    //! @return 投影テクスチャなら true を返します。
    //!
    bool IsProjection() const
    {
        const TexGenSrc texGenSource = GetTexGenSrc();
        return texGenSource == TEXGENSRC_ORTHO_PROJECTION || texGenSource == TEXGENSRC_PANE_BASE_ORTHO_PROJECTION || texGenSource == TEXGENSRC_PERS_PROJECTION;
    }

    //! @brief 投影テクスチャパラメータを取得します。
    //!
    //! @param[in] pParam   投影テクスチャパラメータです。
    //!
    void SetProjectionTexGenParamaters(const ProjectionTexGenParamaters* pParam)
    {
        pProjectionTexGenParams = pParam;
    }

    //@}

private:
    //! テクスチャ座標生成のタイプです。
    ut::ResU8           texGenType;
    //! テクスチャ座標生成に使用するソースです。
    ut::ResU8           texGenSrc;
    //! （未使用）
    ut::ResU8           reserve[2];
    //! 投影テクスチャ座標生成です。(初期化時に設定されます。)
    const ProjectionTexGenParamaters* pProjectionTexGenParams;
};

//---------------------------------------------------------------------------
//! @brief TEV ステージの設定を持つ構造体です。
//!
//---------------------------------------------------------------------------
struct TevStage
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    TevStage()
    {
        Set(
            TEVMODE_REPLACE,
            TEVMODE_REPLACE);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief すべての項目を設定します。
    //!
    //! @param[in] _combineRgb      RGBの演算のタイプです。
    //! @param[in] _combineAlpha    アルファの演算のタイプです。
    //!
    void Set(
        TevMode         _combineRgb,
        TevMode         _combineAlpha
    )
    {
        combineRgb = static_cast<ut::ResU8>(_combineRgb);
        combineAlpha = static_cast<ut::ResU8>(_combineAlpha);
    }

    //! @brief RGBの演算のタイプを設定します。
    //!
    //! @param[in] value    設定する値です。
    //!
    void SetCombineRgb(TevMode value)
    {
        combineRgb = static_cast<ut::ResU8>(value);
    }

    //! @brief アルファの演算のタイプを設定します。
    //!
    //! @param[in] value    設定する値です。
    //!
    void SetCombineAlpha(TevMode value)
    {
        combineAlpha = static_cast<ut::ResU8>(value);
    }

    //! @brief RGBの演算のタイプを取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    TevMode GetCombineRgb() const
    {
        return static_cast<TevMode>(combineRgb);
    }

    //! @brief アルファの演算のタイプを取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    TevMode GetCombineAlpha() const
    {
        return static_cast<TevMode>(combineAlpha);
    }
    //@}

private:
    ut::ResU8   combineRgb;
    ut::ResU8   combineAlpha;
    //! （未使用）
    ut::ResU8   reserve[2];

};

//---------------------------------------------------------------------------
//! @brief アルファコンペアの設定を持つ構造体です。
//!
//---------------------------------------------------------------------------
struct AlphaCompare
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 常に通過する設定で初期化します。
    //!
    AlphaCompare()
    {
        Set(ALPHATEST_ALWAYS, 0.f);
    }

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定の設定で初期化します。
    //!
    //! @param[in] aFunc    比較の条件です。
    //! @param[in] aRef     比較の対象値です。
    //!
    AlphaCompare(
        AlphaTest   aFunc,
        f32         aRef
    )
    {
        Set(aFunc, aRef);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 設定を変更します
    //!
    //! @param[in] aFunc    比較の条件です。
    //! @param[in] aRef     比較の対象値です。
    //!
    void Set(
        AlphaTest   aFunc,
        f32         aRef
    )
    {
        NW_ASSERTMSG(0 <= aFunc && aFunc < ALPHATEST_MAX, "out of bounds: 0 <= aFunc[%d] < ALPHATEST_MAX", aFunc);

        func = u8(aFunc);
        ref = aRef;
    }

    //! @brief 設定を変更します
    //!
    //! @param[in] aRef 比較の対象値です。
    //!
    void SetRef(
        f32         aRef
    )
    {
        ref = aRef;
    }

    //! @brief アルファ比較の条件を取得します。
    //!
    //! @return 設定値を返します。
    //!
    AlphaTest GetFunc() const
    {
        return AlphaTest(func);
    }

    //! @brief アルファ比較の対象の値を取得します。
    //!
    //! @return 設定値を返します。
    //!
    f32 GetRef() const
    {
        return ref;
    }

    //@}

private:
    u8 func;
    ut::ResF32 ref;
};

//---------------------------------------------------------------------------
//! @brief ブレンドモードの設定を持つ構造体です。
//!
//---------------------------------------------------------------------------
struct BlendMode
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! ブレンドを行わない(上書き)で初期化します。
    //!
    BlendMode()
    {
        Set(
            BLENDOP_DISABLE,
            BLENDFACTOR_SRC_ALPHA,
            BLENDFACTOR_INV_SRC_ALPHA,
            LOGICOP_DISABLE);
    }

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定の設定で初期化します。
    //!
    //! @param[in] aBlendOp     ブレンド演算のタイプを指定します。
    //! @param[in] aSrcFactor   ピクセルカラーに乗算する値を指定します。
    //! @param[in] aDstFactor   フレームバッファカラーに乗算する値を指定します。
    //! @param[in] aLogicOp     論理演算のタイプを指定します。
    //!
    BlendMode(
        BlendOp         aBlendOp,
        BlendFactor     aSrcFactor,
        BlendFactor     aDstFactor,
        LogicOp         aLogicOp
    )
    {
        Set(aBlendOp, aSrcFactor, aDstFactor, aLogicOp);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 設定を変更します。
    //!
    //! @param[in] aBlendOp     ブレンド演算のタイプを指定します。
    //! @param[in] aSrcFactor   ピクセルカラーに乗算する値を指定します。
    //! @param[in] aDstFactor   フレームバッファカラーに乗算する値を指定します。
    //! @param[in] aLogicOp     論理演算のタイプを指定します。
    //!
    void Set(
        BlendOp         aBlendOp,
        BlendFactor     aSrcFactor,
        BlendFactor     aDstFactor,
        LogicOp         aLogicOp
    )
    {
        blendOp = u8(aBlendOp);
        srcFactor = u8(aSrcFactor);
        dstFactor = u8(aDstFactor);
        logicOp = u8(aLogicOp);
    }

    //! @brief ブレンド演算のタイプを取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    BlendOp GetBlendOp() const
    {
        return BlendOp(blendOp);
    }

    //! @brief ピクセルカラーに乗算する値を取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    BlendFactor GetSrcFactor() const
    {
        return BlendFactor(srcFactor);
    }

    //! @brief フレームバッファカラーに乗算する値を取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    BlendFactor GetDstFactor() const
    {
        return BlendFactor(dstFactor);
    }

    //! @brief 論理演算のタイプを取得します。
    //!
    //! @return 現在の設定値を返します。
    //!
    LogicOp GetLogicOp() const
    {
        return LogicOp(logicOp);
    }

    //@}

private:
    ut::ResU8           blendOp;
    ut::ResU8           srcFactor;
    ut::ResU8           dstFactor;
    ut::ResU8           logicOp;
};

//---------------------------------------------------------------------------
//! @brief インダイレクトのパラメータを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct IndirectParameter
{
    ut::ResF32          rotate;     //!< ローテート値です。
    Vec2                scale;      //!< スケール値です。

    IndirectParameter() : rotate(0.f), scale(1.f, 1.f) {}
};

//---------------------------------------------------------------------------
//! @brief フォント影のパラメータを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct FontShadowParameter
{
    ut::ResU8 blackInterporateColor[3]; //!< 黒補間色です。
    ut::ResU8 whiteInterporateColor[4]; //!< 白補間色です。
    ut::ResU8 reserve; //!<（未使用）

    FontShadowParameter()
    {
        memset(blackInterporateColor, 0, sizeof(blackInterporateColor));
        memset(whiteInterporateColor, 255, sizeof(whiteInterporateColor));
    }
};

//---------------------------------------------------------------------------
//! @brief ペインの各辺の拡大量を持つ構造体です。
//!
//---------------------------------------------------------------------------
struct InflationLRTB
{
    ut::ResS16          l; //!< 左辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    ut::ResS16          r; //!< 右辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    ut::ResS16          t; //!< 上辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
    ut::ResS16          b; //!< 下辺の拡大量です。固定少数(WindowInflationFiexdPoint)です。
};

//---------------------------------------------------------------------------
//! @brief ウィンドウフレームサイズを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct FrameSizeLRTB
{
    ut::ResU16          l; //!< 左辺のサイズです。
    ut::ResU16          r; //!< 右辺のサイズです。
    ut::ResU16          t; //!< 上辺のサイズです。
    ut::ResU16          b; //!< 下辺のサイズです。
};

//---------------------------------------------------------------------------
//! @brief 拡張ユーザデータクラスです。
//!
//! @details
//! 名前文字列とデータがオブジェクトに続いて格納されます。
//!
//! @sa Pane
//! @sa res::ExtUserDataList
//!
//---------------------------------------------------------------------------
class ExtUserData
{
public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @param[in] nameStrOffset    拡張ユーザデータの名前へのオフセットです(オブジェクト先頭から)。
    //! @param[in] dataOffset       拡張ユーザデータへのオフセットです(オブジェクト先頭から)。
    //! @param[in] num              拡張ユーザデータの個数です。
    //! @param[in] type             拡張ユーザデータのデータタイプです。
    //!
    ExtUserData(u32 nameStrOffset, u32 dataOffset, u16 num, u8 type)
        : m_NameStrOffset(nameStrOffset),
          m_DataOffset(dataOffset),
          m_Num(num),
          m_Type(type),
          m_Padding(0)
    {}

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 拡張ユーザデータの名前を取得します。
    //!
    //! @return 拡張ユーザデータに設定されている名前を返します。
    //!
    const char* GetName() const
    {
        return m_NameStrOffset ? internal::ConvertOffsToPtr<const char>(this, m_NameStrOffset): 0;
    }

    //! @brief 拡張ユーザデータの型を取得します。
    //!
    //! @details
    //! 文字列、整数配列、浮動小数点数配列のいずれかになります。
    //!
    //! @return データタイプを返します。
    //!
    ExtUserDataType GetType() const
    {
        return ExtUserDataType(m_Type);
    }

    //! @brief 拡張ユーザデータの個数を取得します。
    //!
    //! @details
    //! データのタイプが文字列の場合は、文字列の長さになります。
    //!
    //! データのタイプが整数配列の場合は、配列の要素数になります。
    //!
    //! データのタイプが浮動小数点数配列の場合は、配列の要素数になります。
    //!
    //! @return データの個数を返します。
    //!
    u16 GetNum() const
    {
        return m_Num;
    }

    //! @brief 拡張ユーザデータを文字列として取得します。
    //!
    //! @details
    //! データのタイプが文字列である場合に有効です。
    //!
    //! @return 文字列のポインタを返します。
    //!
    const char* GetString() const
    {
        NW_ASSERTMSG(GetType() == EXTUSERDATATYPE_STRING, "GetType()[%d] must be equal to EXTUSERDATATYPE_STRING", GetType());
        return internal::ConvertOffsToPtr<const char>(this, m_DataOffset);
    }

    //! @brief 拡張ユーザデータを整数配列として取得します。
    //!
    //! @details
    //! データのタイプが整数配列である場合に有効です。
    //!
    //! @return ResS32配列の先頭要素のポインタを返します。
    //!
    const ut::ResS32* GetIntArray() const
    {
        NW_ASSERTMSG(GetType() == EXTUSERDATATYPE_INT, "GetType()[%d] must be equal to EXTUSERDATATYPE_INT", GetType());
        return internal::ConvertOffsToPtr<const ut::ResS32>(this, m_DataOffset);
    }

    //! @brief 拡張ユーザデータを浮動小数点数配列として取得します。
    //!
    //! @details
    //! データのタイプが浮動小数点配列である場合に有効です。
    //!
    //! @return ResF32配列の先頭要素のポインタを返します。
    //!
    const ut::ResF32* GetFloatArray() const
    {
        NW_ASSERTMSG(GetType() == EXTUSERDATATYPE_FLOAT, "GetType()[%d] must be equal to EXTUSERDATATYPE_FLOAT", GetType());
        return internal::ConvertOffsToPtr<const ut::ResF32>(this, m_DataOffset);
    }

    //@}

private:
    //! データ名のオブジェクトの先頭を基準にしたオフセットです。
    ut::ResU32 m_NameStrOffset;
    //! データのオブジェクトの先頭を基準にしたオフセットです。
    ut::ResU32 m_DataOffset;

    ut::ResU16 m_Num;
    ut::ResU8 m_Type;
    ut::ResU8 m_Padding;
};

//---------------------------------------------------------------------------
//! @brief アニメーション区間タグで指定されるグループの情報です。
//!
//---------------------------------------------------------------------------
struct AnimationGroupRef
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    AnimationGroupRef()
    :   flag(0)
    {
        std::memset(name,    0, sizeof(name));
        std::memset(padding, 0, sizeof(padding));
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief グループ名を取得します。
    //!
    //! @return グループ名を返します。
    //!
    const char* GetName() const
    {
        return name;
    }

    //@}

    //! グループ名です。
    char                name[GroupNameStrMax + 1];
    //! フラグです。
    ut::ResU8           flag;
    //! （未使用です）
    ut::ResU8           padding[2];
};

//---------------------------------------------------------------------------
//! @brief アニメーション共有情報です。
//!
//---------------------------------------------------------------------------
struct AnimationShareInfo
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    AnimationShareInfo()
    {
        std::memset(srcPaneName,     0, sizeof(srcPaneName));
        std::memset(targetGroupName, 0, sizeof(targetGroupName));
        std::memset(padding,         0, sizeof(padding));
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アニメーションの共有元となるペインの名前を取得します。
    //!
    //! @return ペインの名前を返します。
    //!
    const char* GetSrcPaneName() const
    {
        return srcPaneName;
    }

    //! @brief アニメーションの共有対象となるペインが登録されているグループの名前を取得します。
    //!
    //! @return グループの名前を返します。
    //!
    const char* GetTargetGroupName() const
    {
        return targetGroupName;
    }

    //@}

    //! アニメーションの共有元となるペインの名前です。
    char                srcPaneName[ResourceNameStrMax + 1];
    //! アニメーションの共有対象となるペインが登録されているグループの名前です。
    char                targetGroupName[ResourceNameStrMax + 1];
    //! （未使用です）
    ut::ResU8           padding[2];
};

//---------------------------------------------------------------------------
//! @brief レイアウト・リソースファイルのヘッダです。
//!
//! @details
//! fileHeader の kind には res::FILESIGNATURE_FLYT が指定されます。
//!
//! この構造体に以下のデータブロックの組み合わせが続きます。
//!
//! ・ Layout\n
//! ・ FontList\n
//! ・ TextureList\n
//! ・ MaterialList\n
//! ・ ExtUserDataList\n
//! ・ PaneBegin\n
//! ・ PaneEnd\n
//! ・ Pane\n
//! ・ Picture\n
//! ・ TextBox\n
//! ・ Window\n
//! ・ Bounding\n
//! ・ GroupBegin\n
//! ・ GroupEnd\n
//! ・ Group
//!
//---------------------------------------------------------------------------
struct Lyt
{
    //! ファイルヘッダです。
    ut::BinaryFileHeader fileHeader;
};

//---------------------------------------------------------------------------
//! @brief レイアウトを定義するデータブロックです。
//!
//---------------------------------------------------------------------------
struct Layout
{
    //! @brief ヘッダです。
    //!
    //! @details
    //! kind には res::DATABLOCKKIND_LAYOUT が指定されます。
    //!
    ut::BinaryBlockHeader blockHeader;

    //! @brief スクリーンの向きです。(未使用です)
    //! @sa ScreenOriginType
    ut::ResU8           originType;

    //! @brief (未使用です)
    ut::ResU8           padding[3];

    //! @brief レイアウトの画面サイズです。
    Vec2                layoutSize;

    //! @brief 部品としてのサイズ(ペインが実際に覆っている領域のサイズ)です。
    Vec2                partsSize;

    /* Additional Info

    //! @brief レイアウト名です。長さの上限はありませんが、必ずnull終端されます。また、バッファのサイズは4の倍数になります。
    char                name[?];

    */
};

//---------------------------------------------------------------------------
//! @brief コントロールを定義するデータブロックです。
//!
//---------------------------------------------------------------------------
struct Control
{
    //! @brief ヘッダです。
    //!
    //! @details
    //! kind には res::DATABLOCKKIND_CONTROL が指定されます。
    //!
    ut::BinaryBlockHeader blockHeader;

    //! コントロールにユーザーが付けた名前の、オブジェクトの先頭からの相対位置です。
    ut::ResU32          controlUserNameOffset;

    //! Additional Infoのメンバー、controlFunctionalPaneNamesの、オブジェクトの先頭からの相対位置です。
    ut::ResU32          controlFunctionalPaneNamesOffset;

    //! コントロールの機能ペインの数
    ut::ResU16          controlFunctionalPaneNum;

    //! コントロールの機能アニメの数
    ut::ResU16          controlFunctionalAnimNum;

    //! Additional Infoのメンバー、controlFunctionalPaneParameterNameOffsetsの、オブジェクトの先頭からの相対位置です。
    ut::ResU32          controlFunctionalPaneParameterNameOffsetsOffset;

    //! Additional Infoのメンバー、controlFunctionalAnimParameterNameOffsetsの、オブジェクトの先頭からの相対位置です。
    ut::ResU32          controlFunctionalAnimParameterNameOffsetsOffset;

    /* Additional Info

    //! コントロールの名前。長さの上限はありませんが、必ずnull終端されます。また、バッファのサイズは4の倍数になります。
    char        controlName[?];

    //! コントロールのユーザーが付けた名前。長さの上限はありませんが、必ずnull終端されます。また、バッファのサイズは4の倍数になります。
    char        controlUserName[?];

    //! コントロールの機能ペインの名前の配列。XMLに定義された順番になっている。controlFunctionalPaneNumだけある
    char        controlFunctionalPaneNames[?][ResourceNameStrMax];

    //! コントロールの機能アニメの名前(タグ名)の、controlFunctionalAnimNameOffsetsからのオフセット。
    //! XMLに定義された順番になっている。controlFunctionalAnimNumだけある
    //! オブジェクトの先頭からではなく、このメンバからのオフセットになっていることに注意。
    ut::ResU32  controlFunctionalAnimNameOffsets[?];

    //! コントロールの機能アニメの名前(タグ名)。設定されていない場合はnull文字列。可変長で、null終端されている。controlFunctionalAnimNumだけある
    char*       controlFunctionalAnimNames;

    //! コントロールの機能ペインのパラメータ名の、controlFunctionalPaneParameterNameOffsetsからのオフセット。
    //! XMLに定義された順番になっている。controlFunctionalPaneNumだけある
    //! オブジェクトの先頭からではなく、このメンバからのオフセットになっていることに注意。
    ut::ResU32  controlFunctionalPaneParameterNameOffsets[?];

    //! コントロールの機能ペインのパラメータ名。設定されていない場合はnull文字列。可変長で、null終端されている。controlFunctionalPaneNumだけある
    char*       controlFunctionalPaneParameterNames;

    //! コントロールの機能アニメのパラメータ名の、controlFunctionalAnimParameterNameOffsetsからのオフセット。
    //! XMLに定義された順番になっている。controlFunctionalAnimNumだけある
    //! オブジェクトの先頭からではなく、このメンバからのオフセットになっていることに注意。
    ut::ResU32  controlFunctionalAnimParameterNameOffsets[?];

    //! コントロールの機能アニメのパラメータ名。設定されていない場合はnull文字列。可変長で、null終端されている。controlFunctionalAnimNumだけある
    char*       controlFunctionalAnimParameterNames;

    */
};

//---------------------------------------------------------------------------
//! @brief フォントの参照情報です。
//!
//! @sa FontList
//!
//---------------------------------------------------------------------------
struct Font
{
    //! @brief リソース名のオフセットです(FontList の次のアドレスから)。
    ut::ResU32    nameStrOffset;
};

//---------------------------------------------------------------------------
//! @brief フォントの参照情報のリストを格納したデータブロックです。
//!
//! @details
//! ペインが参照するフォントのリストをペインに先立って定義します。
//!
//! blockHeader の kind には res::DATABLOCKKIND_FONTLIST が指定されます。
//!
//! blockHeader の size は nameStrPool までを含みます。
//!
//! この構造体には以下のデータ構造が続きます。
//!
//! ・ res::Font fonts[fontNum]\n
//! ・ u8 nameStrPool[]
//!
//! @sa Layout
//!
//---------------------------------------------------------------------------
struct FontList
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! リストに含まれるフォント参照の総数です。
    ut::ResU16          fontNum;

    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info
    Font                fonts[fontNum];
    u8                  nameStrPool[];
    */
};

//---------------------------------------------------------------------------
//! @brief テクスチャイメージの参照情報です。
//!
//! @sa TextureList
//!
//---------------------------------------------------------------------------
struct Texture
{
    //! @brief リソース名のオフセットです(TextureList の次のアドレスから)。
    ut::ResU32    nameStrOffset;
};

//---------------------------------------------------------------------------
//! @brief テクスチャイメージの参照情報のリストを格納したデータブロックです。
//!
//! @details
//! ペインが参照するテクスチャのリストをペインに先立って定義します。
//!
//! blockHeader の kind には res::DATABLOCKKIND_TEXTURELIST が指定されます。
//!
//! blockHeader の size は nameStrPool までを含みます。
//!
//! この構造体には以下のデータ構造が続きます。
//!
//! ・ res::Texture textures[texNum]\n
//! ・ u8 nameStrPool[]
//!
//! @sa Layout
//!
//---------------------------------------------------------------------------
struct TextureList
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! リストに含まれるテクスチャイメージ参照の総数です。
    ut::ResU16          texNum;

    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info
    Texture             textures[texNum];
    u8                  nameStrPool[];
    */
};

//---------------------------------------------------------------------------
//! @brief テクスチャのマッピング情報です。
//!
//! @sa Material
//!
//---------------------------------------------------------------------------
struct TexMap
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    TexMap()
    :   texIdx(0),
        wrapSflt(0),
        wrapTflt(0)
    {}

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief S方向のラップモードを取得します。
    //!
    //! @return S方向のラップモードを返します。
    //!
    TexWrap GetWarpModeS() const
    {
        return TexWrap(internal::GetBits(wrapSflt,  0, 2));
    }

    //! @brief T方向のラップモードを取得します。
    //!
    //! @return T方向のラップモードを返します。
    //!
    TexWrap GetWarpModeT() const
    {
        return TexWrap(internal::GetBits(wrapTflt,  0, 2));
    }

    //! @brief 縮小フィルタの設定を取得します。
    //!
    //! @return 縮小フィルタの設定を返します。
    //!
    TexFilter GetMinFilter() const
    {
        return TexFilter(internal::GetBits(wrapSflt,  2, 2));
    }

    //! @brief 拡大フィルタの設定を取得します。
    //!
    //! @return 拡大フィルタの設定を返します。
    //!
    TexFilter GetMagFilter() const
    {
        return TexFilter(internal::GetBits(wrapTflt,  2, 2));
    }

    //! @brief S方向のラップモードを設定します。
    //!
    //! @param[in] value    S方向のラップモードです。 lyt::TexWrap を指定します。
    //!
    void SetWarpModeS(u8 value)
    {
        NW_ASSERTMSG(value < TEXWRAP_MAX, "out of bounds: value[%d] < TEXWRAP_MAX", value);
        internal::SetBits(&wrapSflt,  0, 2, value);
    }

    //! @brief T方向のラップモードを設定します。
    //!
    //! @param[in] value    T方向のラップモードです。 lyt::TexWrap を指定します。
    //!
    void SetWarpModeT(u8 value)
    {
        NW_ASSERTMSG(value < TEXWRAP_MAX, "out of bounds: value[%d] < TEXWRAP_MAX", value);
        internal::SetBits(&wrapTflt,  0, 2, value);
    }

    //! @brief 縮小フィルタを設定します。
    //!
    //! @param[in] value    縮小フィルタの設定です。 lyt::TexFilter を指定します。
    //!
    void SetMinFilter(u8 value)
    {
        NW_ASSERTMSG(value < TEXFILTER_MAX, "out of bounds: value[%d] < TEXFILTER_MAX", value);
        internal::SetBits(&wrapSflt,  2, 2, value);
    }

    //! @brief 拡大フィルタを設定します。
    //!
    //! @param[in] value    拡大フィルタの設定です。 lyt::TexFilter を指定します。
    //!
    void SetMagFilter(u8 value)
    {
        NW_ASSERTMSG(value < TEXFILTER_MAX, "out of bounds: value[%d] < TEXFILTER_MAX", value);
        internal::SetBits(&wrapTflt,  2, 2, value);
    }

    //@}

    //! @brief 内部用機能のため使用禁止です。
    ut::ResU16 texIdx;

    //! @brief 内部用機能のため使用禁止です。
    ut::ResU8 wrapSflt;                       // TexWrap, TexFilter

    //! @brief 内部用機能のため使用禁止です。
    ut::ResU8 wrapTflt;                       // TexWrap, TexFilter
};

//---------------------------------------------------------------------------
//! @brief マテリアルが持つ情報の数です。
//!
//! @sa Material
//!
//---------------------------------------------------------------------------
struct MaterialResourceNum
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    MaterialResourceNum()
        :   bits(0)
    {}

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テクスチャの数を取得します。
    //!
    //! @return テクスチャの数を返します。
    //!
    u8 GetTexMapNum() const
    {
        return u8(internal::GetBits(bits,  0, 2));
    }

    //! @brief TexSRT の数を取得します。
    //!
    //! @return TexSRT の数を返します。
    //!
    u8 GetTexSRTNum() const
    {
        return u8(internal::GetBits(bits,  2, 2));
    }

    //! @brief TexCoordGen の数を取得します。
    //!
    //! @return TexCoordGen の数を返します。
    //!
    u8 GetTexCoordGenNum() const
    {
        return u8(internal::GetBits(bits,  4, 2));
    }

    //! @brief TevStage の数を取得します。
    //!
    //! @return TevStage の数を返します。
    //!
    u8 GetTevStageNum() const
    {
        return u8(internal::GetBits(bits,  6, 3));
    }

    //! @brief AlphaCompare を持っているか調べます。
    //!
    //! @return AlphaCompare を持っていれば true を返します。
    //!
    bool HasAlphaCompare() const
    {
        return internal::TestBit(bits, 9);
    }

    //! @brief BlendMode を持っているか調べます。
    //!
    //! @return BlendMode を持っていれば true を返します。
    //!
    bool HasBlendMode() const
    {
        return internal::TestBit(bits, 10);
    }

    //! @brief テクスチャのみを使うか調べます。
    //!
    //! @return テクスチャのみを使う場合は true を返します。
    //!
    bool IsTextureOnly() const
    {
        return internal::TestBit(bits, 11);
    }

    //! @brief アルファの BlendMode をカラーとは別に持っているか調べます。
    //!
    //! @return アルファの BlendMode をカラーとは別に持っていれば true を返します。
    //!
    bool IsSeparateBlendMode() const
    {
        return internal::TestBit(bits, 12);
    }

    //! @brief インダイレクトのパラメータを持っているか調べます。
    //!
    //! @return インダイレクトのパラメータを持っていればtrueを返します。
    //!
    bool HasIndirectParameter() const
    {
        return internal::TestBit(bits, 14);
    }

    //! @brief 投影テクスチャのパラメータ数を調べます。
    //!
    //! @return 投影テクスチャのパラメータ数を返します。
    //!
    u8 GetProjectionTexGenNum() const
    {
        return u8(internal::GetBits(bits,  15, 2));
    }

    //! @brief フォント影のパラメータを持っているか調べます。
    //!
    //! @return フォント影のパラメータを持っていれば true を返します。
    //!
    bool HasFontShadowParameter() const
    {
        return internal::TestBit(bits, 17);
    }

    //! @brief 閾値によるアルファ補間を用いるか調べます。
    //!
    //! @return 閾値によるアルファ補間を用いる場合は true を返します。
    //!
    bool IsThresholdingAlphaInterpolation() const
    {
        return internal::TestBit(bits, 18);
    }

    //! @brief TexMap の数を設定します。
    //!
    //! @param[in] value    TexMap の数です。
    //!
    void SetTexMapNum(u32 value)
    {
        NW_ASSERTMSG(value <= TexMapMax, "out of bounds: value[%u] <= TexMapMax", value);
        this->SetBits(0, 2, value);
    }

    //! @brief TexSRT の数を設定します。
    //!
    //! @param[in] value    TexSRT の数です。
    //!
    void SetTexSRTNum(u32 value)
    {
        NW_ASSERTMSG(value <= TexMapMax, "out of bounds: value[%u] <= TexMapMax", value);
        this->SetBits(2, 2, value);
    }

    //! @brief TexCoordGen の数を設定します。
    //!
    //! @param[in] value    TexCoordGen の数です。
    //!
    void SetTexCoordGenNum(u32 value)
    {
        NW_ASSERTMSG(value <= TexMapMax, "out of bounds: value[%u] <= TexMapMax", value);
        this->SetBits(4, 2, value);
    }

    //! @brief TevStage の数を設定します。
    //!
    //! @param[in] value    TevStage の数です。
    //!
    void SetTevStageNum(u32 value)
    {
        NW_ASSERTMSG(value <= TevStageMax, "out of bounds: value[%u] <= TevStageMax", value);
        this->SetBits(6, 3, value);
    }

    //! @brief AlphaCompare を持つかを設定します。
    //!
    //! @param[in] b    AlphaCompare を持つ場合には true を指定します。
    //!
    void SetAlphaCompare(bool b)
    {
        this->SetBit(9, b);
    }

    //! @brief BlendMode を持つかを設定します。
    //!
    //! @param[in] b    BlendMode を持つ場合には true を指定します。
    //!
    void SetBlendMode(bool b)
    {
        this->SetBit(10, b);
    }

    //! @brief テクスチャのみを使うかを設定します。
    //!
    //! @param[in] b    テクスチャのみを使う場合には true を指定します。
    //!
    void SetTextureOnly(bool b)
    {
        this->SetBit(11, b);
    }

    //! @brief アルファの BlendMode をカラーとは別に持っているかを設定します。
    //!
    //! @param[in] b    アルファの BlendMode をカラーとは別に持っている場合には true を指定します。
    //!
    void SetSeparateBlendMode(bool b)
    {
        this->SetBit(12, b);
    }

    //! @brief インダイレクトのパラメータを持っているかを設定します。
    //!
    //! @param[in] b    インダイレクトのパラメータを持っている場合には true を指定します。
    //!
    void SetIndirectParameter(bool b)
    {
        this->SetBit(14, b);
    }

    //! @brief 投影テクスチャのパラメータ数を設定します。
    //!
    //! @param[in] value    投影テクスチャのパラメータ数 を指定します。
    //!
    void SetProjectionTexGenNum(u32 value)
    {
        NW_ASSERTMSG(value <= TevStageMax, "out of bounds: value[%u] <= TevStageMax", value);
        this->SetBits(15, 2, value);
    }

    //! @brief フォント影のパラメータを持っているかを設定します。
    //!
    //! @param[in] b    フォント影のパラメータを持っている場合には true を指定します。
    //!
    void SetFontShadowParameter(bool b)
    {
        this->SetBit(17, b);
    }

    //! @brief 閾値によるアルファ補間を用いるか設定します。
    //!
    //! @param[in] b    閾値によるアルファ補間を用いる場合には true を設定します。
    //!
    void SetThresholdingAlphaInterpolation(bool b)
    {
        this->SetBit(18, b);
    }

    //@}

    //! @brief 内部用機能のため使用禁止です。
    ut::ResU32 bits;

private:
    void SetBits(int pos, int len, u32 value)
    {
        u32 work = bits;
        internal::SetBits(&work, pos, len, value);
        bits = work;
    }

    void SetBit(int pos, bool value)
    {
        u32 work = bits;
        internal::SetBit(&work, pos, value);
        bits = work;
    }

};

//---------------------------------------------------------------------------
//! @brief カラー情報です。
//!
//! @sa Lyt
//! @sa Material
//! @sa Picture
//! @sa WindowContent
//!
//---------------------------------------------------------------------------
struct Color
{
    //! @brief 型変換を行います。
    //!
    //! @return ut::Color8 型のオブジェクトを返します。
    //!
    operator ut::Color8() const
    {
        return ut::Color8(r,g,b,a);
    }

    //! 赤色の成分です。
    ut::ResU8 r;
    //! 緑色の成分です。
    ut::ResU8 g;
    //! 青色の成分です。
    ut::ResU8 b;
    //! アルファ成分です。
    ut::ResU8 a;
};

//---------------------------------------------------------------------------
//! @brief マテリアル情報です。
//!
//! @details
//! この構造体に続いて、詳細情報の構造体が続きます。
//!
//! ・ TexMap resTexMaps[texNum]\n
//! ・ TexSRT texSRTs[texSRTNum]\n
//! ・ TexCoordGen texCoordGen[texCoordGenNum]\n
//! ・ TevStage tevStages[tevStageNum]\n
//! ・ AlphaCompare alphaCompare\n
//! ・ BlendMode blendMode
//! ・ BlendMode blendModeAlpha
//! ・ HsbAdjustment hsbAdjustment
//! ・ IndirectParameter indirectParameter
//!
//! どの詳細情報をいくつ持つかは resNum の値によって決まります。
//!
//! @sa MaterialList
//!
//---------------------------------------------------------------------------
struct Material
{
    //! 名前です。
    char                name[MaterialNameStrMax];
    //! カラー情報です。
    Color               colors[MatColorMax];

    //! 詳細設定の数です。
    MaterialResourceNum resNum;

    /* Additional Info
    TexMap              resTexMaps[texNum];
    TexSRT              texSRTs[texSRTNum];
    TexCoordGen         texCoordGen[texCoordGenNum];
    TevStage            tevStages[tevStageNum];
    AlphaCompare        alphaCompare;
    BlendMode           blendMode;
    BlendMode           blendModeAlpha;
    HsbAdjustment       hsbAdjustment;
    IndirectParameter   indirectParameter;
    FontShadowParameter fontShadowParameter;
    */
};

//---------------------------------------------------------------------------
//! @brief マテリアル情報のリストです。
//!
//! @details
//! ペインが参照するマテリアルのリストをペインに先立って定義します。
//!
//! blockHeader の kind には res::DATABLOCKKIND_MATERIALLIST が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ ut::ResU32 materialOffsetTable[materialNum]\n
//! ・ Material material0\n
//! ・ …
//!
//! materialOffsetTable は各マテリアル情報へのオフセット(データブロック先頭から)です。
//!
//! @sa Layout
//!
//---------------------------------------------------------------------------
struct MaterialList
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! マテリアルの数です。
    ut::ResU16          materialNum;

    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info
    ut::ResU32          materialOffsetTable[materialNum];
    Material            materials[materialNum];
    */
};

//---------------------------------------------------------------------------
//! @brief 子ペインの定義を開始するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANEBEGIN が指定されます。
//!
//! PaneBegin と PaneEnd により子ペイン列が定義されます。
//!
//! PaneBegin の前の Pane が親ペインとなります。
//!
//! ・ Pane : pane0\n
//! ・ PaneBegin\n
//! ・ Pane : pane1\n
//! ・ Pane : pane2\n
//! ・ PaneEnd
//!
//! この場合、 pane1 と pane2 は pane0 の子ペインとなります。
//!
//! Pane の代わりに Picture, Window, TextBox, Bounding も指定可能です。
//!
//! @sa Lyt
//!
//---------------------------------------------------------------------------
struct PaneBegin
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;
};

//---------------------------------------------------------------------------
//! @brief 子ペインの定義を完了するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANEEND が指定されます。
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct PaneEnd
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;
};

//---------------------------------------------------------------------------
//! @brief Nullペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANE が指定されます。
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct Pane
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! フラグです (PaneFlag)。
    ut::ResU8           flag;
    //! 位置基準です (ベースポジションのHとV、親ペインでの原点位置のHとVがそれぞれ2ビットずつ)。
    //! 型はHがlyt::HorizontalPosition、Vがlyt::VerticalPositionとなります。
    ut::ResU8           basePosition;
    //! アルファ値です。
    ut::ResU8           alpha;
    //! 追加のフラグです (PaneFlagEx)。
    ut::ResU8           flagEx;

    //! ペインの名前です。
    char                name[ResourceNameStrMax];
    //! ユーザデータです。
    char                userData[UserDataStrMax];
    //! トランスレート値です。
    Vec3                translate;
    //! ローテート値です。
    Vec3                rotate;
    //! スケール値です。
    Vec2                scale;
    //! サイズです。
    Vec2                size;
};

//---------------------------------------------------------------------------
//! @brief Pictureペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PICTURE が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ res::Vec2 texCoords[texCoordNum][VERTEX_MAX];
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct Picture : public Pane
{
    //! 頂点カラーです。
    Color               vtxCols[VERTEXCOLOR_MAX];

    //! マテリアルのインデックスです。
    ut::ResU16          materialIdx;
    //! テクスチャ座標の数です。
    ut::ResU8           texCoordNum;
    //! (未使用です)
    ut::ResU8           padding[1];

    /* Additional Info
    res::Vec2           texCoords[texCoordNum][VERTEX_MAX];
    */
};

//---------------------------------------------------------------------------
//! @brief テキストボックスの文字単位のアニメーションの情報です。
//!
//! @sa TextBox
//!
//---------------------------------------------------------------------------
struct PerCharacterTransform
{
    //! 文字単位のアニメーションのカーブオフセットです。
    ut::ResF32          evalTimeOffset;

    //! 文字単位のアニメーションのカーブ幅です。
    ut::ResF32          evalTimeWidth;

    //! 文字単位のアニメーションのループの種類です。
    ut::ResU8           loopType;

    //! 文字単位のアニメーションの回転中心です。
    ut::ResU8           originV;

    //! 文字単位のアニメーションのAnimationInfoがあるか否かです。0以外の場合は、この構造体の
    //! すぐ後ろにAnimationInfoがあります。
    ut::ResU8           hasAnimationInfo;

    //! (未使用です)
    ut::ResU8           padding[1];

};

//---------------------------------------------------------------------------
//! @brief TextBoxペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_TEXTBOX が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ char16 text[];
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct TextBox : public Pane
{
    //! 文字列用バッファに確保するサイズです。
    ut::ResU16          textBufBytes;
    //! 初期テキスト文字列のバイト数です。
    ut::ResU16          textStrBytes;

    //! マテリアルのインデックスです。
    ut::ResU16          materialIdx;
    //! フォントへのインデックスです。
    ut::ResU16          fontIdx;

    //! テキスト表示位置基準です(lyt::HorizontalPosition * HORIZONTALPOSITION_MAX + lyt::VerticalPosition)。
    ut::ResU8           textPosition;
    //! 行そろえ指定です(TextAlignment)。
    ut::ResU8           textAlignment;
    //! フラグです(TextBoxFlag)。
    ut::ResU8           textBoxFlag;
    //! (未使用です)
    ut::ResU8           padding[1];

    //! 斜体の傾斜率です(-1.0 ～ 1.0 ゼロの時にOFF)。
    ut::ResF32          italicRatio;

    //! text へのオフセットです(データブロック先頭から)。
    ut::ResU32          textStrOffset;
    //! テキストの表示色です。
    Color               textCols[TEXTCOLOR_MAX];
    //! フォントのサイズです。
    Vec2                fontSize;
    //! テキストの文字間隔です。
    ut::ResF32          charSpace;
    //! テキストの行間隔です。
    ut::ResF32          lineSpace;
    //! textID へのオフセットです(データブロック先頭から)。textIDが存在しない場合は0となります。
    ut::ResU32          textIDOffset;

    //! 影のオフセットです。
    Vec2                shadowOffset;

    //! 影のスケールです。
    Vec2                shadowScale;

    //! 影の色です。
    Color               shadowCols[TEXTCOLOR_MAX];

    //! 影の斜体傾斜率です(-1.0 ～ 1.0 ゼロの時にOFF)。
    ut::ResF32          shadowItalicRatio;

    //! 文字単位のアニメーションの情報へのオフセットです(データブロック先頭から)。存在しない場合は0となります。
    ut::ResU32          perCharacterTransformOffset;

    /* Additional Info
    char16           text[];                     // テキスト
    char                textID[];                   // テキストID
    PerCharacterTransform perCharacterTransform     // 文字単位のアニメーションの情報
    AnimationInfo       perCharacterTransformAnimationInfo;     // 文字単位のアニメーションのアニメーション情報
    */
};

//---------------------------------------------------------------------------
//! @brief Windowのフレーム情報です。
//!
//! @sa Window
//!
//---------------------------------------------------------------------------
struct WindowFrame
{
    //! マテリアルのインデックスです。
    ut::ResU16          materialIdx;
    //! テクスチャの反転表示の指定です(lyt::TextureFlip)。
    ut::ResU8           textureFlip;
    //! (未使用です)
    ut::ResU8           padding[1];
};

//---------------------------------------------------------------------------
//! @brief Windowのコンテント情報です。
//!
//! @details
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ res::Vec2 texCoords[texCoordNum][VERTEX_MAX]
//!
//! @sa Window
//!
//---------------------------------------------------------------------------
struct WindowContent
{
    //! 頂点カラーです。
    Color               vtxCols[VERTEXCOLOR_MAX];

    //! マテリアルのインデックスです。
    ut::ResU16          materialIdx;
    //! テクスチャ座標の数です。
    ut::ResU8           texCoordNum;
    //! (未使用です)
    ut::ResU8           padding[1];

    /* Additional Info
    math::VEC2          texCoords[texCoordNum][VERTEX_MAX];
    */
};

//---------------------------------------------------------------------------
//! @brief Windowペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_WINDOW が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ WindowContent content\n
//! ・ ut::ResU32 frameOffsetTable[frameNum]\n
//! ・ WindowFrame frame0\n
//! ・ …
//!
//! frameOffsetTable は WindowFrame へのオフセット(データブロック先頭から)です。
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct Window : public Pane
{
    //! ペインの各辺の拡大量です。
    InflationLRTB       inflation;

    //! フレームサイズです。
    FrameSizeLRTB       frameSize;

    //! フレーム数です。
    ut::ResU8           frameNum;
    //! ウィンドウ設定フラグです(WindowFlag 列挙子を使ってアクセスします)。
    ut::ResU8           windowFlags;
    //! (未使用です)
    ut::ResU8           padding[2];

    //! content へのオフセットです(データブロック先頭から)。
    ut::ResU32          contentOffset;
    //! frameOffsetTable へのオフセットです(データブロック先頭から)。
    ut::ResU32          frameOffsetTableOffset;

    /* Additional Info

    WindowContent       content;

    internal::ResU32    frameOffsetTable[frameNum];
    WindowFrame         frames;

    */
};

//---------------------------------------------------------------------------
//! @brief Boundingペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_BOUNDING が指定されます。
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct Bounding : public Pane
{
};

//---------------------------------------------------------------------------
//! @brief Partsペインで、ペイン基本情報を上書きするための構造体です。
//!
//---------------------------------------------------------------------------
struct PartsPaneBasicInfo
{
    //! ユーザデータです。
    char                userData[UserDataStrMax];
    //! トランスレート値です。
    Vec3                translate;
    //! ローテート値です。
    Vec3                rotate;
    //! スケール値です。
    Vec2                scale;
    //! サイズです。
    Vec2                size;
    //! 透明度です。
    ut::ResU8           alpha;
    //! (未使用です)
    ut::ResU8           padding[3];
};

//---------------------------------------------------------------------------
//! @brief Partsペインのプロパティを定義するための構造体です。
//!
//---------------------------------------------------------------------------
struct PartsProperty
{
    //! プロパティのペイン名です。
    char        name[ResourceNameStrMax];

    //! 上書きの用途を指定するフラグです。解釈は上書き対象の種類によって異なります。
    ut::ResU8   usageFlag;

    //! 基本部分の上書きの用途を指定するフラグです。(BasicOverrideUsageFlag列挙子のビットフラグです。)
    ut::ResU8   basicUsageFlag;

    //! マテリアルの部分上書きの用途を指定するフラグです。(MaterialOverrideUsageFlag列挙子のビットフラグです。)
    ut::ResU8   materialUsageFlag;

    //! (未使用です)
    ut::ResU8   padding[1];

    //! プロパティの実データ(res::Pictureまたはres::TextBoxまたはres::Window)のオフセット値。Partデータブロックの先頭からの値になります。
    ut::ResU32  propertyOffset;

    //! ExtUserDataOverrideOffsetの値、もしくは拡張ユーザーデータへのオフセット値(Partデータブロックの先頭からの値)です。
    ut::ResU32  extUserDataOffset;

    //! ペイン基本情報の上書きを行うための情報(PartsPaneBasicInfo)へのオフセット値。Partデータブロックの先頭からの値になります。
    ut::ResU32  paneBasicInfoOffset;
};

//---------------------------------------------------------------------------
//! @brief Partsペインを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PARTS が指定されます。
//!
//! @sa Lyt
//! @sa PaneBegin
//!
//---------------------------------------------------------------------------
struct Parts : public Pane
{
    //! プロパティの数です。
    ut::ResU32      propertyNum;

    //! 部品倍率です。
    Vec2            magnify;

    /* Additional Info

    //! プロパティの情報がどこにあるかを示すテーブルです。propertyNum個の配列となります。
    PartsProperty   propertyTable[propertyNum];

    //! @brief 部品になるレイアウト名です。長さの上限はありませんが、必ずnull終端されます。また、バッファのサイズは4の倍数になります。
    char            name[?];

    //! プロパティの数だけ、PictureデータブロックまたはTextBoxデータブロックまたはPartsデータブロックが続きます。
    //! propertyTable内のオフセットを利用してアクセスします。
    Picture         picture;
    TextBox         textBox;
    Parts           parts;
    */
};

//---------------------------------------------------------------------------
//! @brief 拡張ユーザデータを定義するデータブロックです。
//!
//! @details
//! 直近に定義されたペインに拡張ユーザデータを設定します。
//!
//! blockHeader の kind には res::DATABLOCKKIND_USERDATALIST が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ ExtUserData extUserData[num]\n
//! ・ u8 data[]
//!
//! @sa Lyt
//! @sa Pane
//!
//---------------------------------------------------------------------------
struct ExtUserDataList
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! ユーザデータの数です。
    ut::ResU16          num;
    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info

    ExtUserData         extUserData[num];

    */
};

//---------------------------------------------------------------------------
//! @brief 子グループの定義を開始するデータブロックです。
//!
//! @details
//! GroupBegin と GroupEnd により子グループ列が定義されます。
//!
//! GroupBegin の前の Group が親グループとなります。
//!
//! @sa Lyt
//!
//---------------------------------------------------------------------------
struct GroupBegin
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;
};

//---------------------------------------------------------------------------
//! @brief 子グループの定義を完了するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_GROUPEND が指定されます。
//!
//! @sa Lyt
//! @sa GroupBegin
//!
//---------------------------------------------------------------------------
struct GroupEnd
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;
};

//---------------------------------------------------------------------------
//! @brief グループを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_GROUP が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ char panes[paneNum][ResourceNameStrMax]
//!
//! panes にはグループに属するペインの名前が格納されます。
//!
//! @sa Lyt
//! @sa GroupBegin
//!
//---------------------------------------------------------------------------
struct Group
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! グループ名です。
    char                name[GroupNameStrMax + 1];

    //! (未使用です)
    ut::ResU8           padding[1];
    //! グループに属するペインの数です。
    ut::ResU16          paneNum;

    /* Additional Info

    char                panes[paneNum][ResourceNameStrMax];

    */
};

//@}

//!--------------------------------------------------------------------------*
//! @name アニメーション・リソース
//@{

//---------------------------------------------------------------------------
//! @brief アニメーション・リソースファイルのヘッダです。
//!
//! @details
//! fileHeader の kind には res::FILESIGNATURE_FLAN が指定されます。
//!
//! この構造体に以下のデータブロックの組み合わせが続きます。
//!
//! ・ AnimationTagBlock\n
//! ・ AnimationShareBlock\n
//! ・ AnimationBlock
//!
//---------------------------------------------------------------------------
struct Lan
{
    //! ファイルヘッダです。
    ut::BinaryFileHeader fileHeader;
};

//---------------------------------------------------------------------------
//! @brief アニメーション区間タグを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANEANIMTAG が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ char name[]\n
//! ・ AnimationGroupRef groups[groupNum]
//!
//! @sa Lan
//!
//---------------------------------------------------------------------------
struct AnimationTagBlock
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! アニメーション区間タグの通し番号です。
    ut::ResU16          tagOrder;
    //! アニメーション区間タグのグループ数です。
    ut::ResU16          groupNum;

    //! name へのオフセットです(データブロック先頭から)。
    ut::ResU32          nameOffset;
    //! groups へのオフセットです(データブロック先頭から)。
    ut::ResU32          groupsOffset;

    //! アニメーション区間の開始フレームです。
    ut::ResS16          startFrame;
    //! アニメーション区間の終了フレームです。
    ut::ResS16          endFrame;

    //! フラグです(lyt::AnimTagFlag)。
    ut::ResU8           flag;
    //! (未使用です)
    ut::ResU8           padding[3];

    /* Additional Info

    char                name[];
    AnimationGroupRef   groups[groupNum];

    */
};

// 本来は、 nn::ui2d::ResAnimationTagBlock を参照すべきところだが、
// コンバーターが、nw::lyt を参照しているので、一時的に 構造体を定義します。
// nw::lyt 依存が排除された際に、削除してください。
struct AnimationTagBlockUi2d
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! アニメーション区間タグの通し番号です。
    ut::ResU16          tagOrder;
    //! アニメーション区間タグのグループ数です。
    ut::ResU16          groupNum;

    //! name へのオフセットです(データブロック先頭から)。
    ut::ResU32          nameOffset;
    //! groups へのオフセットです(データブロック先頭から)。
    ut::ResU32          groupsOffset;
    //! ユーザーデーターリストへのオフセットです(データブロック先頭から)。存在しない場合は、0 が設定されます。
    ut::ResU32          userDataListOffset;

    //! アニメーション区間の開始フレームです。
    ut::ResS16          startFrame;
    //! アニメーション区間の終了フレームです。
    ut::ResS16          endFrame;

    //! フラグです(lyt::AnimTagFlag)。
    ut::ResU8           flag;
    //! (未使用です)
    ut::ResU8           padding[3];

    /* Additional Info

    char                name[];
    AnimationGroupRef   groups[groupNum];
    ResExtUserDataList  userDataList;

    */
};

//---------------------------------------------------------------------------
//! @brief アニメーション共有情報を定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANEANIMSHARE が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ AnimationShareInfo animShareInfos[shareNum]
//!
//! @sa Lan
//!
//---------------------------------------------------------------------------
struct AnimationShareBlock
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! animShareInfos へのオフセットです(データブロック先頭から)。
    ut::ResU32          animShareInfoOffset;

    //! アニメーション共有情報の数です。
    ut::ResU16          shareNum;
    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info

    AnimationShareInfo  animShareInfos[shareNum];

    */
};

//---------------------------------------------------------------------------
//! @brief アニメーションを定義するデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_PANEANIMINFO が指定されます。
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ ut::ResU32 fileNameOffsets[fileNum]\n
//! ・ u8 namePool[]\n
//! ・ ut::ResU32 animContOffsets[paneNum]\n
//! ・ AnimationContent animCont0\n
//! ・ …
//!
//! fileNameOffsets 配列には参照するテクスチャリソース名への
//! オフセット(fileNameOffsetsより)を格納します。
//!
//! animContOffsets 配列にはアニメーション内容への
//! オフセット(データブロック先頭より)を格納します。
//!
//! @sa Lan
//!
//---------------------------------------------------------------------------
struct AnimationBlock
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;
    //! フレーム数です。
    ut::ResU16          frameSize;
    //! ループする場合は 1 を格納します。
    ut::ResU8           loop;
    //! (未使用です)
    ut::ResU8           padding[1];

    //! 参照するテクスチャリソースの数です。
    ut::ResU16          fileNum;
    //! アニメーション内容の数です。
    ut::ResU16          animContNum;

    //! animContOffsets へのオフセットです(データブロック先頭から)。
    ut::ResU32          animContOffsetsOffset;

    /* Additional Info

    ut::ResU32          fileNameOffsets[fileNum];
    u8                  namePool[];
    ut::ResU32          animContOffsets[paneNum];

    */
};

//---------------------------------------------------------------------------
//! @brief アニメーション内容を定義します。
//!
//! @details
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ ut::ResU32 animInfoOffsets[num]\n
//! ・ AnimationInfo animInfo0\n
//! ・ …
//!
//! @sa Lan
//! @sa AnimationBlock
//!
//---------------------------------------------------------------------------
struct AnimationContent
{
    //! アニメーションの名前です。
    char                name[MaterialNameStrMax];   // ペイン名とマテリアル名の長い方の文字数分確保

    //! アニメーション情報の数です。
    ut::ResU8           num;
    //! タイプ(lyt::AnimContentType)です。
    ut::ResU8           type;
    //! (未使用です)
    ut::ResU8           padding[2];

    /* Additional Info

    ut::ResU32    animInfoOffsets[num];

    */
};

//---------------------------------------------------------------------------
//! @brief アニメーション情報を定義します。
//!
//! @details
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ ut::ResU32 animTargetOffsets[num]\n
//! ・ AnimationTarget animTarget0\n
//! ・ …
//!
//! animTargetOffsets 配列は AnimationTarget へのオフセット(AnimationInfo 先頭から)を格納します。
//!
//! @sa Lan
//! @sa AnimationContent
//!
//---------------------------------------------------------------------------
struct AnimationInfo
{
    ut::SigWord         kind;

    //! アニメーション・ターゲットの数です。
    ut::ResU8           num;
    //! (未使用です)
    ut::ResU8           padding[3];

    /* Additional Info

    ut::ResU32          animTargetOffsets[num];

    */
};

//---------------------------------------------------------------------------
//! @brief アニメーション・ターゲットを定義します。
//!
//! @details
//! アニメーションの対象とカーブの情報です。
//!
//! target にはアニメーションの対象に応じて以下の列挙型のいずれかが指定されます。
//!
//! ・ lyt::AnimTargetPane\n
//! ・ lyt::AnimTargetVisibility\n
//! ・ lyt::AnimTargetPaneColor\n
//! ・ lyt::AnimTargetMatColor\n
//! ・ lyt::AnimTargetTexSRT\n
//! ・ lyt::AnimTargetIndirectSRT\n
//! ・ lyt::AnimTargetTexPattern
//!
//! この構造体に以下のデータ構造が続きます。
//!
//! ・ HermiteKey keys[keyNum]
//!
//! または
//!
//! ・ StepKey keys[keyNum]
//!
//! @sa Lan
//! @sa AnimationInfo
//!
//---------------------------------------------------------------------------
struct AnimationTarget
{
    //! アニメーション対象が配列の場合のインデックスです。
    ut::ResU8           id;
    //! アニメーション対象の属性をあらわす列挙値です。
    ut::ResU8           target;
    //! カーブの種類です(lyt::AnimCurve)。
    ut::ResU8           curveType;
    //! (未使用です)
    ut::ResU8           padding[1];

    //! キーの数です。
    ut::ResU16          keyNum;
    //! (未使用です)
    ut::ResU8           padding2[2];

    //! keys へのオフセットです(AnimationTarget先頭から)。
    ut::ResU32          keysOffset;

    /* Additional Info

    HermiteKey|StepKey  keys[keyNum];

    */
};

//---------------------------------------------------------------------------
//! @brief エルミート曲線のキーを定義します。
//!
//! @sa Lan
//! @sa AnimationTarget
//!
//---------------------------------------------------------------------------
struct HermiteKey
{
    //! フレームです。
    ut::ResF32          frame;
    //! 値です。
    ut::ResF32          value;
    //! 傾きです。
    ut::ResF32          slope;
};

//---------------------------------------------------------------------------
//! @brief 階段状に変化するアニメーションカーブのキーを定義します。
//!
//! @sa Lan
//! @sa AnimationTarget
//!
//---------------------------------------------------------------------------
struct StepKey
{
    //! フレームです。
    ut::ResF32          frame;
    //! 値です。
    ut::ResU16          value;
    //! (未使用です)
    ut::ResU16          padding[1];
};

//@}

//!--------------------------------------------------------------------------*
//! @name テクスチャイメージ・リソース
//@{

//---------------------------------------------------------------------------
//! @brief テクスチャイメージ・リソースファイルのヘッダです。
//!
//! @details
//! fileHeader の kind には res::FILESIGNATURE_FLIM が指定されます。
//!
//! テクスチャイメージ・リソースファイルは以下のような構造になっています。
//!
//! ・ u8 textureImage[imageSize]\n
//! ・ Lim limHeader\n
//! ・ Image imageInfo\n
//! ・ ImageSize imageSize
//!
//! テクスチャイメージはハードウェアによるアライメント制限があるため
//! ファイルの先頭に配置されます。
//!
//! ファイルの末尾には ImageSize 構造体が格納されます。
//!
//---------------------------------------------------------------------------
struct Lim
{
    //! ファイルヘッダです。
    ut::BinaryFileHeader fileHeader;
};

//---------------------------------------------------------------------------
//! @brief テクスチャイメージ情報のデータブロックです。
//!
//! @details
//! blockHeader の kind には res::DATABLOCKKIND_IMAGE が指定されます。
//!
//! @sa Lim
//!
//---------------------------------------------------------------------------
struct Image
{
    //! ヘッダです。
    ut::BinaryBlockHeader blockHeader;

    //! イメージの幅です。
    ut::ResU16          width;

    //! イメージの高さです。
    ut::ResU16          height;

    //! 必要なアライメントのバイト数です。
    ut::ResU16          neededAlignment;

    //! 画像データのフォーマットです(lyt::TexFormat)。
    ut::ResU8           format;

    //! 0-4ビットがタイルモード、5-7ビットがスイズルパターンを格納します。
    ut::ResU8           tileModeAndSwizzlePattern;
};

//---------------------------------------------------------------------------
//! @brief テクスチャイメージのサイズ情報です。
//!
//! @details
//! テクスチャイメージ・リソースファイルの先頭に配置される画像データの
//! バイト数の情報を持ちます。
//!
//! @sa Lim
//!
//---------------------------------------------------------------------------
struct ImageSize
{
    //! 画像データのバイト数です。(4の倍数)
    ut::ResU32          imageSize;
};

} // namespace nw::lyt::res
} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_RESOURCES_H_
