﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_RESOURCEACCESSOR_H_
#define NW_LYT_RESOURCEACCESSOR_H_

#include <nw/lyt/lyt_Types.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

namespace nw
{
namespace font
{

class Font;

}

namespace lyt
{

//---------------------------------------------------------------------------
//! @brief リソースのタイプを表します。
//!
//! @sa res::RESOURCETYPE_LAYOUT
//! @sa res::RESOURCETYPE_ANIMATION
//! @sa res::RESOURCETYPE_TEXTURE
//! @sa res::RESOURCETYPE_FONT
//! @sa res::RESOURCETYPE_ARCHIVEFONT
//!
//---------------------------------------------------------------------------
typedef u32 ResType;

//---------------------------------------------------------------------------
//! @brief リソースにアクセスするための基底クラスです。
//!
//---------------------------------------------------------------------------
class ResourceAccessor
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

protected:
    //! @brief コンストラクタです。
    //!
    ResourceAccessor();

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //! @brief デストラクタです。
    //!
    //! @details
    //! 導出クラスでは管理下のフォントとテクスチャの破棄を
    //! 行わなければなりません。
    //!
    //! @date 2010/01/26 フォントとテクスチャの破棄について加筆しました。
    virtual ~ResourceAccessor();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief リソースを取得します。
    //!
    //! @param[in] resType  リソースの種類です。
    //! @param[in] name     リソースの名前です。
    //! @param[out] pSize   リソースのサイズを受け取る変数へのポインタです。
    //!                     サイズを受け取る必要がない場合は NULL を渡してください。
    //!
    //! @return
    //! リソースの先頭アドレスへのポインタを返します。
    //! リソースが見つからなかった場合は NULL を返します。
    //!
    virtual void* GetResource(
        ResType resType,
        const char* name,
        u32* pSize = 0
    ) = 0;

    //! @brief フォントを取得します。
    //!
    //! @details
    //! name で指定したフォントリソース名と関連付けられた
    //! フォントオブジェクトを取得します。
    //!
    //! フォントを所有していない場合にはロードを試みます。
    //!
    //! 実装には LoadFont() を利用することができます。
    //!
    //! @param[in] name フォントリソースの名前です。
    //!
    //! @return フォントへのポインタを返します。
    //! フォントを取得できなかった場合には NULL を返します。
    //!
    //! @sa LoadFont
    //!
    //! @date 2010/07/16 純粋仮想関数に変更しました。
    //!
    virtual font::Font* GetFont(const char* name) = 0;

    //! @brief テクスチャ情報を取得します。
    //!
    //! @details
    //! name で指定したテクスチャリソース名と関連付けられた
    //! テクスチャ情報を取得します。
    //!
    //! テクスチャを所有していない場合にはロードを試みます。
    //!
    //! 実装には LoadTexture() を利用することができます。
    //!
    //! @param[in] name テクスチャリソースの名前です。
    //!
    //! @return テクスチャ情報を返します。
    //! テクスチャを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはテクスチャオブジェクトには無効な値
    //! (TextureInfo::INVALID) が設定されます。
    //!
    //! @sa LoadTexture
    //!
    //! @date 2010/01/26 非バーチャル関数に変更しました。
    //! @date 2010/06/18 純粋仮想関数に変更しました。
    //!
    virtual const TextureInfo* GetTexture(const char* name) = 0;

    //! @brief シェーダ情報を取得します。
    //!
    //! @details
    //! name で指定したシェーダリソース名と関連付けられた
    //! シェーダ情報を取得します。
    //!
    //! シェーダを所有していない場合にはロードを試みます。
    //!
    //! 実装には LoadShader() を利用することができます。
    //!
    //! @param[in] name シェーダリソースの名前です。
    //!
    //! @return シェーダ情報を返します。
    //! シェーダを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはシェーダオブジェクトには無効な値
    //! が設定されます。
    //!
    //! @sa LoadShader
    //!
    virtual const ArchiveShaderInfo* GetShader(const char* name) = 0;

    //@}

protected:
    //! @brief テクスチャをロードします。
    //!
    //! @details
    //! テクスチャが初めて参照されたときに GetTexture() から呼び出されます。
    //!
    //! GetResource() でテクスチャリソースを取得し、
    //! lyt::LoadTexture() でデバイスにロードします。
    //!
    //! @param[out] textureInfo ロードしたテクスチャの情報を格納します。
    //! @param[in] name         テクスチャリソースの名前です。
    //!
    //! @return ロードが成功した場合は true が返ります。
    //!
    //! @sa GetTexture
    //! @sa lyt::LoadTexture
    //!
    virtual bool LoadTexture(TextureInfo* textureInfo, const char* name);

    //! @brief リソースフォントを生成します。
    //!
    //! @details
    //! フォントが初めて参照されたときに GetFont() から呼び出されます。
    //!
    //! font::ResFont を生成し GetResource() で取得した
    //! フォントリソースを設定します。
    //!
    //! @param[in] name フォントリソースの名前です。
    //!
    //! @return font::ResFont へのポインタを返します。
    //!
    //! @sa GetFont
    //!
    virtual font::Font* LoadFont(const char* name);

    //! @brief シェーダをロードします。
    //!
    //! @details
    //! シェーダが初めて参照されたときに GetShader() から呼び出されます。
    //!
    //! @param[out] pShaderInfo ロードしたシェーダの情報を格納します。
    //! @param[in] name         シェーダリソースの名前です。
    //!
    //! @return ロードが成功した場合は true が返ります。
    //!
    //! @sa GetShader
    //!
    virtual bool LoadShader(ArchiveShaderInfo* pShaderInfo, const char* name);
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_RESOURCEACCESSOR_H_
