﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_PICTURE_H_
#define NW_LYT_PICTURE_H_

#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#include <nw/lyt/lyt_TexMap.h>
#include <nw/lyt/lyt_Pane.h>

namespace nw
{
namespace lyt
{

namespace res
{

struct Picture;

}   // namespace res

class DrawInfo;

//---------------------------------------------------------------------------
//! @brief 画像の表示を行うペインです。
//!
//---------------------------------------------------------------------------
class Picture : public Pane
{
    typedef Pane Base;

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Picture::Base);

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定したテクスチャの枚数分のメモリを確保して、オブジェクトを生成します。
    //!
    //! @param[in] texNum   使用するテクスチャの最大数です。
    //!
    explicit Picture(u8 texNum);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定したテクスチャを使用するオブジェクトを生成します。
    //! 保持できるテクスチャ座標の数は 1 に設定されます。
    //!
    //! @param[in] textureInfo   TextureInfo への参照です。
    //!
    explicit Picture(const TextureInfo& textureInfo);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定したリソースからオブジェクトを生成します。
    //!
    //! @param[in] pBaseBlock       このペインが所属するレイアウトが保持しているリソースへのポインタです。
    //! @param[in] pOverrideBlock   上書きを行ったリソースへのポインタです。上書きが行われていないときはNULLになります。
    //! @param[in] buildArgSet      構築時の引数への参照です。
    //!
    Picture(
        const res::Picture* pBaseBlock,
        const res::Picture* pOverrideBlock,
        const BuildArgSet& buildArgSet);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したペインをコピーしたペインを作成します。
    //!
    //! ペインの元々のレイアウトデータの状態ではなく、そのときの状態でコピーします
    //! のでご注意ください。つまり、アニメーションを再生した後はそのままの状態で
    //! コピーします。
    //!
    //! ただし、このメソッドではペインの親子関係はコピーしませんので、ご注意ください。
    //! つまり、ペインは親を持たず、子のリストが空の状態で作られます。
    //! ペインツリーに登録するためには、明示的にペインにAppendChildしてください。
    //!
    //! また、アニメーションについても、何もバインドされていない状態になります。
    //!
    //! @param[in] picture  コピー元のペインです。
    //!
    explicit Picture(const Picture& picture);

    //! @brief デストラクタです。
    //!
    virtual ~Picture();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テクスチャを追加します。
    //!
    //! @param[in] textureInfo   TextureInfo への参照です。
    //!
    virtual void Append(const TextureInfo& textureInfo);

    //! @brief 頂点カラーを取得します。
    //!
    //! @details
    //! idx で指定した頂点カラーを取得します。
    //!
    //! @param[in] idx  インデックスです。 lyt::VertexColor を指定します。
    //!
    //! @return カラーを返します。
    //!
    //! @sa SetVtxColor
    //! @sa GetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual const ut::Color8 GetVtxColor(u32 idx) const;

    //! @brief 頂点カラーを設定します。
    //!
    //! @param[in] idx      インデックスです。 lyt::VertexColor を指定します。
    //! @param[in] value    頂点カラーです。
    //!
    //! @sa GetVtxColor
    //! @sa SetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual void SetVtxColor(u32 idx, ut::Color8 value);

    //! @brief 頂点カラーの成分を取得します。
    //!
    //! @details
    //! idx で指定した頂点カラーの成分を取得します。
    //!
    //! idx の指定には lyt::ANIMTARGET_VERTEXCOLOR_* を使用してください。
    //!
    //! このメソッドは、R,G,B など色成分を個別に取得します。色成分をまとめて取得する場合は GetVtxColor を利用してください。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return 頂点カラーの成分を返します。 lyt::AnimTargetPaneColor　を指定します。
    //!
    //! @sa SetVtxColorElement
    //! @sa GetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual u8 GetVtxColorElement(u32 idx) const;

    //! @brief 頂点カラーの成分を設定します。
    //!
    //! idx の指定には lyt::ANIMTARGET_VERTEXCOLOR_* を使用してください。
    //!
    //! このメソッドは、R,G,B など色成分を個別に設定します。色成分をまとめて設定する場合は SetVtxColor を利用してください。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //! @param[in] value    カラーの成分値です。
    //!
    //! @sa GetVtxColorElement
    //! @sa SetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual void SetVtxColorElement(u32 idx, u8 value);

    //! @brief テクスチャ座標を保持するためのメモリ領域を確保します。
    //!
    //! @details
    //! 保持可能なテクスチャ座標の数の初期値はコンストラクタによって異なります。
    //!
    //! @param[in] num  テクスチャ座標の保持数です。
    //!
    //! @sa Picture::Picture
    //!
    void ReserveTexCoord(u8 num);

    //! @brief テクスチャ座標の保持数を取得します。
    //!
    //! @return 保持しているテクスチャ座標の数を返します。
    //!
    //! @sa SetTexCoordNum
    //!
    u8 GetTexCoordNum() const;

    //! @brief テクスチャ座標の保持数を設定します。
    //!
    //! @details
    //! num に指定できる値は保持可能なテクスチャ座標数以下でなければなりません。
    //!
    //! @param[in] num  テクスチャ座標の保持数です。
    //!
    //! @sa GetTexCoordNum
    //! @sa ReserveTexCoord
    //! @sa Picture::Picture
    //! @sa GetTexCoord
    //!
    void SetTexCoordNum(u8 num);

    //! @brief テクスチャ座標を取得します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[out] coords  テクスチャ座標を格納する領域へのポインタです。
    //!
    //! @sa SetTexCoord
    //!
    void GetTexCoord(u32 idx, TexCoordQuad coords) const;

    //! @brief テクスチャ座標を設定します。
    //!
    //! @param[in] idx  インデックスです。
    //! @param[in] coords   テクスチャ座標です。
    //!
    //! @sa GetTexCoord
    //!
    void SetTexCoord(u32 idx, const TexCoordQuad coords);

    using Base::GetMaterial;

    //! @brief ペインが持つマテリアルの数を取得します。
    //!
    //! @return マテリアルを持っていれば 1 を返します。
    //!
    virtual u8 GetMaterialNum() const;

    //! @brief ペインのマテリアルを取得します。
    //!
    //! @details
    //! idx には 0 のみ指定できます。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa SetMaterial
    //!
    virtual Material* GetMaterial(u32 idx) const;

    //! @brief マテリアルを設定します。
    //!
    //! @details
    //! 現在設定されているマテリアルが Picture オブジェクトの生成時に
    //! 同時に生成されたものだった場合には、そのマテリアルは破棄されます。
    //!
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    void SetMaterial(Material* pMaterial);

    //@}

protected:
    virtual void DrawSelf(DrawInfo& drawInfo);

private:
    void Init(u8 texNum);

    Material* m_pMaterial;
    ut::Color8 m_VtxColors[VERTEXCOLOR_MAX];
    internal::TexCoordAry m_TexCoordAry;
    // コピー演算子を禁止します。
    const Picture& operator=( const Picture& );
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_PICTURE_H_

