﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_LAYOUT_H_
#define NW_LYT_LAYOUT_H_

#include <new>

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_LinkList.h>

#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Animator.h>
#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Group.h>
#include <nw/lyt/lyt_Types.h>
#include <nw/lyt/lyt_Parts.h>
#include <nw/font/font_DispStringBuffer.h>

namespace nw
{
namespace ut
{

struct Rect;

} // namespace ut

namespace font
{

    template <typename CharType>
    class TagProcessorBase;

} // namespace font

namespace lyt
{
namespace res
{

struct AnimShareInfoList;
struct PartsProperty;
struct ExtUserDataList;
struct Vec2;

} // namespace res

class ResourceAccessor;
struct BuildResSet;
class DrawInfo;
class PaneAnimator;
class GroupAnimator;
class ControlCreator;
class TextSearcher;

//! @brief アニメーションの一覧を保持します。
//!
typedef ut::LinkList<AnimTransform, offsetof(AnimTransform, m_Link)> AnimTransformList;

//! @brief 部品ペインの一覧を保持するリストの定義です。
//!
typedef ut::LinkList<Parts, offsetof(Parts, m_PartsLink)>   PartsPaneList;

//---------------------------------------------------------------------------
//! @brief レイアウト全体を管理するためのクラスです。
//!
//! @sa Pane
//! @sa AnimTransform
//!
//---------------------------------------------------------------------------
class Layout
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //! @brief レイアウトに関する定数です。
    enum {
        LAYOUT_NAME_MAX = 64, //!< レイアウト名の上限。リソース的には上限はないが、Layoutクラスの実装の都合上上限を設けている

        DEFAULT_ALIGNMENT = 4
    };

    //----------------------------------------
    //! @name メモリ管理
    //@{

    //! @brief 現在のメモリアロケータを取得します。
    //!
    //! @return メモリアロケータへのポインタを返します。
    //!
    //! @sa SetAllocator
    //!
    static nw::ut::IAllocator* GetAllocator()
    {
        return s_pAllocator;
    }

    //! @brief メモリブロックの確保に使用するメモリアロケータを設定します。
    //!
    //! @param[in] pAllocator   メモリアロケータへのポインタです。
    //!
    //! @sa AllocMemory
    //! @sa GetAllocator
    //!
    static void SetAllocator(nw::ut::IAllocator* pAllocator);

    //! @brief メモリブロックを確保します。
    //!
    //! @details
    //! size バイトのメモリブロックを確保します。
    //! この関数を使用する前に Layout::SetAllocator() によりメモリアロケータを
    //! 設定してください。
    //!
    //! @param[in] size         確保するメモリブロックのサイズです。
    //! @param[in] alignment    確保するメモリブロックのアライメント値です。
    //!
    //! @return
    //! メモリブロックを確保できた場合はメモリブロックの先頭アドレスを、
    //! 確保できなかった場合は NULL を返します。
    //!
    //! @sa SetAllocator
    //! @sa FreeMemory
    //!
    static void* AllocMemory(u32 size, u32 alignment = DEFAULT_ALIGNMENT);

    //! @brief メモリブロックを開放します。
    //!
    //! @param[in] mem  開放するメモリブロックの先頭アドレスです。
    //!
    //! @sa AllocMemory
    //!
    static void FreeMemory(void* mem);

    //@}

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    Layout();

    /**
     *  @brief コピーコンストラクタです。
     *
     *  @details
     *  ペインツリーごとコピーします。そのペインツリー内に部品レイアウトがあった場合は、その
     *  部品レイアウトもコピーします。
     *
     *  部品レイアウトをコピーした場合、そのルートペインはペインのツリー階層にぶら下がって
     *  いませんので、任意の位置にAppendChild等でぶら下げてください。
     *
     *  コピー元のレイアウトでアニメを作成していてもここではコピーしませんのでご注意ください。
     *  グループについてはコピーします。
     *
     *  アプリケーションで独自にペインを継承して拡張している場合は、このメソッドの内部で使用
     *  しているClonePaneTreeWithPartsLayoutメソッドを含めて、Layoutクラスの継承先でコピー
     *  コンストラクタを再実装する必要がありますので、ご注意ください。
     *
     *  @param[in] src              コピー元のレイアウト
     *  @param[in] parentLayout     部品レイアウトをコピーする場合、親のレイアウトのポインタを渡すと、
     *                              親のレイアウトの部品ペインリストにコピーしたレイアウトのルートを
     *                              登録します。
     *  @param[in] rootPaneName     ルートペインの名前を変更する場合に指定します。
     *                              ルートペインの名前を変更しない場合はNULLを指定してください。
     *                              特に部品レイアウトをコピーする場合は、元と同じ名前だとペインの
     *                              検索を行う場合等に問題が発生するため、名前を変更することを推奨します。
     */
    explicit Layout(const Layout& src, Layout* parentLayout = NULL, const char* rootPaneName = NULL);

    //! @brief デストラクタです。
    //!
    virtual ~Layout();

    //@}

    //----------------------------------------
    //! @name 初期化
    //@{

    //! @brief 初期化のオプションを表す構造体です。
    //!
    struct BuildOption
    {
        //! @brief テキストボックスの文字属性バッファをレイアウト内で共有するか否か。
        //!
        //! @details
        //! デフォルトはtrueとなっています。共有すると、メモリ消費が削減できます。特に
        //! テキストボックスが多いデータで効果が見込めます。
        //!
        //! 共有するバッファサイズは初期化時に決まるため、レイアウトを作成した後に別の
        //! レイアウトにくっつけるような使い方をした場合、共有のバッファのサイズが不足
        //! する場合があります。そのような場合は、くっつける方のレイアウトを、この値を
        //! falseにして構築してください。
        //!
        bool isShareTextBoxCharAttributeBuffer;

        //! @brief ルートペインを部品ペインにするか否か
        //!
        //! @details
        //! デフォルトではfalseになっています。
        //!
        //! 通常、レイアウトのルートペインはnw::lyt::Pane(ヌルペイン)になりますが、この
        //! オプションをtrueにすると、ルートペインがnw::lyt::Parts(部品ペイン)になります。
        //! これにより、構築したレイアウトを何か他のレイアウトにくっつけたときに、部品
        //! としてレイアウトエディタ上で配置したのと同じ状態にすることができ、
        //! FindPaneByNameしたときに、くっつけたレイアウトの中を探しに行かないようになります。
        //!
        //! なお、構築したレイアウトを他のレイアウトにくっつける場合、部品として配置した
        //! のと同じ状態にするには、ペインのChildListに入れるだけでなく、レイアウトの
        //! PartsPaneListに入れる必要があることにご注意ください。詳しくはGetPartsPaneList
        //! を参照してください。
        //!
        //! @sa FindPaneByName
        //! @sa GetPartsPaneList
        //!
        bool isRootPaneParts;

        //! @brief デフォルトコンストラクタです。
        //!
        BuildOption()
         : isShareTextBoxCharAttributeBuffer(true)
         , isRootPaneParts(false)
        {}
    };

    //! @brief リソースをもとにレイアウトの初期化を行います。
    //!
    //! @details
    //! リソースのフォーマットが正しくなければ失敗します。
    //! なお、このメソッドで指定したリソースアクセサは内部に保持され、
    //! CreateAnimTransformやGetAnimResourceDataで使われます。
    //!
    //! @param[in] lytResBuf            レイアウトリソースへのポインタです。
    //! @param[in,out] pResAcsr         リソースアクセサへのポインタです。
    //! @param[in,out] pControlCreator  コントロールの作成を行うオブジェクトへのポインタです。
    //!                                 NULLだったときは、レイアウトリソースにコントロールが含まれて
    //!                                 いてもコントロールを作成しません。デフォルトではNULLです。
    //! @param[in,out] pTextSearcher    TextBoxにテキストIDが設定されているときに、対応するテキスト
    //!                                 を検索するためのオブジェクトへのポインタです。NULLだったときは、
    //!                                 リソースのテキストがそのまま設定されます。デフォルトはNULLです。
    //! @param[in] buildOption          構築時のオプションです。
    //!
    //! @return 初期化に成功した場合は true を、失敗した場合は false を返します。
    //!
    bool Build(
        const void* lytResBuf,
        ResourceAccessor* pResAcsr,
        ControlCreator* pControlCreator = NULL,
        TextSearcher* pTextSearcher = NULL,
        const BuildOption& buildOption = BuildOption());

    //! @brief
    //! 指定されたレイアウトファイル名でリソースアクセサからリソースを取得し、
    //! それをもとにレイアウトの初期化を行います。
    //!
    //! @details
    //! レイアウトファイル名が正しくなければ失敗します。
    //! なお、このメソッドで指定したリソースアクセサは内部に保持され、
    //! CreateAnimTransformやGetAnimResourceDataで使われます。
    //!
    //! @param[in] layoutFileName      レイアウトファイル名です。コンバート後の拡張子(bflyt)付きで指定してください。
    //! @param[in,out] pResAcsr         リソースアクセサへのポインタです。
    //! @param[in,out] pControlCreator  コントロールの作成を行うオブジェクトへのポインタです。
    //!                                 NULLだったときは、レイアウトリソースにコントロールが含まれて
    //!                                 いてもコントロールを作成しません。デフォルトではNULLです。
    //! @param[in,out] pTextSearcher    TextBoxにテキストIDが設定されているときに、対応するテキスト
    //!                                 を検索するためのオブジェクトへのポインタです。NULLだったときは、
    //!                                 リソースのテキストがそのまま設定されます。デフォルトはNULLです。
    //! @param[in] buildOption          構築時のオプションです。
    //!
    //! @return 初期化に成功した場合は true を、失敗した場合は false を返します。
    //!
    bool BuildWithName(
        const char* layoutFileName,
        ResourceAccessor* pResAcsr,
        ControlCreator* pControlCreator = NULL,
        TextSearcher* pTextSearcher = NULL,
        const BuildOption& buildOption = BuildOption());
    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトの作成のみ行います。
    //!
    //! @return 生成したアニメーションへのポインタを返します。
    //!
    template <typename T>
    T* CreateAnimTransform()
    {
        NW_ASSERT_NOT_NULL(s_pAllocator);

        T *const pAnimTrans = NewObj<T>();
        if (pAnimTrans)
        {
            this->GetAnimTransformList().PushBack(pAnimTrans);
        }
        return pAnimTrans;
    }

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] animResBuf   アニメーションリソースへのポインタです。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合は NULL を返します。
    //!
    //! @sa BindAnimationAuto
    //!
    template <typename T>
    T* CreateAnimTransform(const void* animResBuf)
    {
        return CreateAnimTransform<T>(AnimResource(animResBuf));
    }

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] animRes  アニメーションリソースオブジェクトです。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合は NULL を返します。
    //!
    //! @sa BindAnimationAuto
    //!
    template <typename T>
    T* CreateAnimTransform(const AnimResource& animRes)
    {
        const res::AnimationBlock *const pAnimBlock = animRes.GetResourceBlock();
        if (! pAnimBlock)
        {
            return NULL;
        }

        T *const pAnimTrans = CreateAnimTransform<T>();
        if (pAnimTrans)
        {
            pAnimTrans->SetResource(pAnimBlock, m_ResourceAccessor);
        }

        return pAnimTrans;
    }

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたタグ名のリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    //! @sa BindAnimationAuto
    //!
    template <typename T>
    T* CreateAnimTransform(const char* tagName)
    {
        return CreateAnimTransform<T>(GetAnimResourceData(tagName));
    }

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトの作成のみ行います。
    //!
    //! @return 生成したアニメーションへのポインタを返します。
    //!
    AnimTransformBasic* CreateAnimTransformBasic();

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] animResBuf   アニメーションリソースへのポインタです。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合は NULL を返します。
    //!
    //! @sa BindAnimationAuto
    //!
    AnimTransformBasic* CreateAnimTransformBasic(const void* animResBuf);

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] animRes  アニメーションリソースオブジェクトです。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合は NULL を返します。
    //!
    //! @sa BindAnimationAuto
    //!
    AnimTransformBasic* CreateAnimTransformBasic(const AnimResource& animRes);

    //! @brief アニメーションを作成します。
    //!
    //! @details
    //! アニメーションオブジェクトを作成し、指定されたタグ名のリソースを関連付けます。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! アニメーションリソースにアニメーション共有情報が含まれる場合は、
    //! BindAnimationAuto() を使用してください。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    //! @sa BindAnimationAuto
    //!
    AnimTransformBasic* CreateAnimTransformBasic(const char* tagName);

    //! @brief タグ名と関連づけたいペインへのポインタからPaneAnimatorを作成します。
    //!
    //! @details
    //! PaneAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、指定されたペインにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] pane     関連づけるペインへのポインタです。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    PaneAnimator* CreatePaneAnimator(const char* tagName, Pane* pane, bool enable = true);

    //! @brief タグ名と関連づけたいペイン名からPaneAnimatorを作成します。
    //!
    //! @details
    //! PaneAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、指定されたペインにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] paneName 関連づけるペインの名前です。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    PaneAnimator* CreatePaneAnimator(const char* tagName, const char* paneName, bool enable = true)
    {
        return CreatePaneAnimator(tagName, m_pRootPane->FindPaneByName(paneName), enable);
    }

    //! @brief タグ名と関連づけたいグループへのポインタからGroupAnimatorを作成します。
    //!
    //! @details
    //! GroupAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、指定されたグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] group    関連づけるグループへのポインタです。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションへのポインタを返します。
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    GroupAnimator* CreateGroupAnimator(const char* tagName, Group* group, bool enable = true);

    //! @brief タグ名と関連づけたいグループ名からGroupAnimatorを作成します。
    //!
    //! @details
    //! GroupAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、指定されたグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName      アニメーションのタグ名です。
    //! @param[in] groupName    関連づけるグループの名前です。
    //! @param[in] enable       開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションへのポインタを返します。
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    GroupAnimator* CreateGroupAnimator(const char* tagName, const char* groupName, bool enable = true)
    {
        return CreateGroupAnimator(tagName, m_pGroupContainer->FindGroupByName(groupName), enable);
    }

    //! @brief 指定したタグ名のGroupAnimatorを作成し、アニメタグに関連づけられたグループにバインドします。
    //!
    //! @details
    //! GroupAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、タグの関連グループの先頭のグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションへのポインタを返します。
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    GroupAnimator* CreateGroupAnimator(const char* tagName, bool enable = true)
    {
        return CreateGroupAnimatorWithIndex(tagName, 0, enable);
    }

    //! @brief タグ名と関連づけたいグループのインデックスからGroupAnimatorを作成します。
    //!
    //! @details
    //! GroupAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、タグの関連グループのindex番目のグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! アニメタグに複数のグループを関連づけているときに、バインドしたいグループを
    //! インデックスを指定する場合に使用するメソッドです。
    //!
    //! @param[in] tagName      アニメーションのタグ名です。
    //! @param[in] groupIndex   アニメタグの関連グループ内で、関連づけたいグループが何番目にあるかを指定します。
    //! @param[in] enable       開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションへのポインタを返します。
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    GroupAnimator* CreateGroupAnimatorWithIndex(const char* tagName, u32 groupIndex, bool enable = true);

    //! @brief 指定したタグ名のGroupArrayAnimatorを作成し、アニメタグに関連づけられた全てのグループにバインドします。
    //!
    //! @details
    //! GroupArrayAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、アニメタグに関連づけられた全てのグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! アニメタグにグループが一つも設定されていない場合は構築に失敗し、
    //! NULLを返します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    GroupArrayAnimator* CreateGroupArrayAnimator(const char* tagName, bool enable = true);

    //! @brief 指定したタグ名のAnimatorを作成し、アニメタグに関連づけられたグループ全てにバインドします。
    //!
    //! @details
    //! 指定されたタグの関連グループが1つの場合はGroupAnimatorを、2つ以上の場合は
    //! GroupArrayAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、アニメタグに関連づけられた全てのグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! 作られるAnimatorがGroupAnimatorかGroupArrayAnimatorかわからないという問題点が
    //! ありますが、CreateGroupArrayAnimatorと比較してメモリ消費を抑えることができます。
    //!
    //! @param[in] tagName  アニメーションのタグ名です。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合の動作は未定義であり、ASSERT で停止します。
    //!
    Animator* CreateGroupAnimatorAuto(const char* tagName, bool enable = true);

    //! @brief アニメーションを削除します。
    //!
    //! @param[in,out] pAnimTransform   アニメーションへのポインタです。
    //!
    virtual void DeleteAnimTransform(
        AnimTransform* pAnimTransform);

    //! @brief アニメーションを関連付けます。
    //!
    //! @details
    //! レイアウトに属する全てのペインについて関連付けを行います。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //!
    //! @sa UnbindAnimation
    //! @sa UnbindAllAnimation
    //!
    virtual void BindAnimation(AnimTransform* pAnimTrans);

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @details
    //! レイアウトに属する全てのペインについて関連付けを解除します。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //!
    //! @sa BindAnimation
    //! @sa UnbindAllAnimation
    //!
    virtual void UnbindAnimation(AnimTransform* pAnimTrans);

    //! @brief 指定したペインのアニメーションの関連付けを解除します。
    //!
    //! @details
    //! レイアウトに属する全てのアニメについて、指定したペインの
    //! 関連付けを解除します。
    //!
    //! @param[in] pPane    ペインへのポインタです。
    //!
    //! @sa BindAnimation
    //! @sa UnbindAllAnimation
    //!
    virtual void UnbindAnimation(Pane* pPane);

    //! @brief アニメーションとペインの関連付けをすべて解除します。
    //!
    //! @sa BindAnimation
    //! @sa UnbindAnimation
    //!
    virtual void UnbindAllAnimation();

    //! @brief アニメーションの作成と関連付けを一度に行います。
    //!
    //! @details
    //! アニメーションリソースオブジェクトの内容に応じて、
    //! アニメーションオブジェクト(AnimTransform)を作成し、
    //! ペインに関連付けます。
    //!
    //! アニメーションファイルにアニメーション共有情報が含まれている場合は、
    //! 共有対象のペインごとにアニメーションオブジェクト(AnimTransform)が
    //! 作成されます。
    //!
    //! また、アニメーション区間タグが含まれていて、関連グループが
    //! 指定されている場合は、アニメーションを関連付けるペインを関連グループに限定します。
    //!
    //! この関数では、アニメーションは無効状態で関連付けます。
    //!
    //! @param[in] animRes  アニメーションリソースオブジェクトです。
    //!
    //! @return 関数が成功した場合は true、失敗した場合は false を返します。
    //!
    //! @sa CreateAnimTransform
    //!
    virtual bool BindAnimationAuto(const AnimResource& animRes);

    //! @brief 全てのアニメーションの結果を計算します。
    //!
    virtual void Animate();

    //! @brief 全てのアニメーションのフレームを進めます。
    //!
    //! @details
    //! レイアウトが部品を持っている場合は、そのレイアウトのアニメーションのフレームも進めます。
    //!
    //! @param[in] progress_frame 進めるフレーム数です。
    //!
    virtual void UpdateAnimFrame(f32 progress_frame = 1.0f);

    //! @brief 全てのアニメーションについて、結果を計算した後にフレームを進めます。
    //!
    //! @details
    //! レイアウトが部品を持っている場合は、そのレイアウトの結果の計算とアニメーションのフレーム
    //! を進める処理も行います。
    //!
    //! AnimateとUpdateAnimFrameを別々に行うよりも処理が速くなります。別々のタイミングで行う必要
    //! がないのであれば、こちらを使用してください。
    //!
    //! @param[in] progress_frame 進めるフレーム数です。
    //!
    virtual void AnimateAndUpdateAnimFrame(f32 progress_frame = 1.0f);

    //! @brief 指定したタグ名のアニメーションリソースデータを取得します。
    //!
    //! @param[in] tagName  アニメーションのタグ名
    //!
    const void* GetAnimResourceData(const char* tagName);

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief 描画のための行列を計算します。
    //!
    //! @details
    //! 引数drawInfoで与える DrawInfo はメソッド内では書き換わりませんので、同じインスタンスを
    //! 複数のスレッドで別々のレイアウトに与えて計算しても問題ありません。
    //!
    //! このメソッドは、描画のための行列を計算すると共に、TextBoxの頂点バッファの更新も行います。
    //! この頂点バッファの更新は内部でダブルバッファリングしているため、1フレームに2回以上
    //! CalculateMtxを呼び出すと、ダブルバッファリングの切り替えが不正に行われ、描画が崩れてしまいます。
    //! 処理負荷の面からも、CalculateMtxを呼び出すのは1フレームに1回であることが推奨されますが、
    //! どうしても1フレームに複数回呼び出す場合は、2回目以降はTextBoxのClearDirtyFlagメソッドを
    //! 呼び出しておくことでTextBoxの頂点バッファが更新されないようにしてください。
    //!
    //! 引数forceGlbMtxDirtyは、ペインのグローバルマトリックスを強制的に再計算させる場合に
    //! trueを指定します。ペインは、初回計算時もしくは自分自身と親のSRT及びサイズが変更された
    //! 場合のみグローバルマトリックスを再計算する仕組みになっていますが、DrawInfoに設定した
    //! ビュー行列が変更された場合も再計算が必要になります。
    //!
    //! DrawInfoに設定したビュー行列が変更された場合はforceGlbMtxDirtyにtrueを与えるように
    //! してください。
    //!
    //! この引数は、正射影で描画している場合はほぼ指定する必要はありません。透視射影でカメラを
    //! 動かしているような場合に指定する必要があります。
    //!
    //! @param[in] drawInfo         描画情報です。
    //! @param[in] forceGlbMtxDirty 全てのグローバルマトリックスを再計算するか否かです。デフォルトではfalseです。
    //!
    //! @sa TextBox::ClearDirtyFlag
    //!
    virtual void CalculateMtx(const DrawInfo& drawInfo, bool forceGlbMtxDirty = false);

    //! @brief 描画処理を行います。
    //!
    //! @details
    //! drawInfo には、あらかじめ DrawInfo::SetGraphicsResource() により
    //! GraphicsResource を設定しておく必要があります。
    //!
    //! GLの以下の設定はユーザに任せています(lytでは設定しません)。
    //!
    //! ・カリング\n
    //! ・シザー処理\n
    //! ・ステンシルテスト\n
    //! ・マスク処理\n
    //! ・ポリゴンオフセット\n
    //! ・デプステスト\n
    //! ・アーリーデプステスト\n
    //! ・フレームバッファオブジェクト
    //!
    //! @param[in] drawInfo 描画情報です。
    //!
    //! @sa GraphicsResource
    //!
    virtual void Draw(DrawInfo& drawInfo);

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 部品となっているレイアウトか否かを取得します。
    //!
    //! @details
    //! ルートペインが親を持っているか否かという方法で判定していますので、
    //! ライブラリユーザ側で部品以外の理由でルートペインの親を使用している
    //! 場合は、正しく判定できないことがありますのでご注意ください。
    //!
    //! @return
    //! 部品となっている場合はtrue、そうでない場合はfalseを返します。
    //!
    bool IsPartsLayout() const
    {
        return (m_pRootPane->GetParent() != NULL);
    }

    //! @brief ルートペインを取得します。
    //!
    //! @return
    //! ルートペインへのポインタを返します。
    //! ルートペインが存在しない場合は NULL を返します。
    //!
    Pane* GetRootPane() const
    {
        return m_pRootPane;
    }

    //! @brief グループコンテナを取得します。
    //!
    //! @return
    //! グループコンテナへのポインタを返します。
    //! グループコンテナが存在しない場合は NULL を返します。
    //!
    GroupContainer* GetGroupContainer() const
    {
        return m_pGroupContainer;
    }

    //! @brief 表示領域のサイズを取得します。
    //!
    //! @return サイズを返します。
    //!
    const Size& GetLayoutSize() const
    {
        return m_LayoutSize;
    }

    //! @brief レイアウトの表示領域を取得します。
    //!
    //! @return 表示領域の矩形を返します。
    //!
    const ut::Rect GetLayoutRect() const;

    //! @brief レイアウト名を取得します。
    //!
    //! @details
    //! レイアウト名は、レイアウトのファイル名から拡張子を除いたものになります。
    //! 長さは63文字までです。
    //!
    //! @return レイアウト名を返します。
    //!
    const char* GetName() const
    {
        return m_Name;
    }

    //! @brief レイアウト内の全てのテキストボックスペインにタグプロセッサを設定します。
    //!
    //! @param[in] pTagProcessor    タグプロセッサへのポインタです。
    //!
    //! @sa TextBox::GetTagProcessor
    //! @sa TextBox::SetTagProcessor
    //!
    virtual void SetTagProcessor(font::TagProcessorBase<char16>* pTagProcessor);

    //! @brief レイアウトに関連付けているアニメーションのリストを取得します。
    //!
    //! @return AnimTransform オブジェクトのリストを返します。
    //!
    const AnimTransformList& GetAnimTransformList() const
    {
        return m_AnimTransList;
    }

    //! @brief レイアウトに関連付けているアニメーションのリストを取得します。
    //!
    //! @return AnimTransform オブジェクトのリストを返します。
    //!
    AnimTransformList& GetAnimTransformList()
    {
        return m_AnimTransList;
    }

    //! @brief レイアウトに含まれる部品ペインのリストを取得します。
    //!
    //! @return PartsPaneList 部品ペインのリストを返します。
    //!
    const PartsPaneList& GetPartsPaneList() const
    {
        return m_PartsPaneList;
    }

    //! @brief レイアウトに含まれる部品ペインのリストを取得します。
    //!
    //! @details
    //! 構築したレイアウトを他のレイアウトにくっつけるときに、このメソッドで取得した
    //! リストに追加すると、Animate等のアニメを更新するメソッドで更新されるようになり、
    //! またFindPartsPaneByNameで検索対象になります。逆に言うとこれらのメソッドで追加
    //! の処理がかかる事になりますので、機能が必要なければリストに追加する必要はありません。
    //!
    //! また、構築したレイアウトを他のレイアウトにくっつけるときは、BuildOptionの
    //! isRootPanePartsについての説明も参照してください。
    //!
    //! @sa BuildOption::isRootPaneParts
    //!
    //! @return PartsPaneList 部品ペインのリストを返します。
    //!
    PartsPaneList& GetPartsPaneList()
    {
        return m_PartsPaneList;
    }

#if defined(NW_PLATFORM_CAFE)
    //! @brief アーカイブシェーダを取得します。
    //!
    //! @param[in] name シェーダ名
    //!
    //! @return アーカイブシェーダ
    //!
    const ArchiveShaderInfo* GetArchiveShader(const char* name);
#endif

    //! @brief 指定した名前の部品ペインをリストから検索し、取得します。
    //!
    //! @details
    //! レイアウトが部品ペインのみを管理しているリストを使用しますので、
    //! ルートペインからFindPaneByNameするよりも高速に検索できます。
    //!
    //! @param[in] findName 検索するペイン名です。
    //!
    //! @return
    //! findName のペイン名を持つ部品ペインが見つかった場合は部品ペインへのポインタを、
    //! 見つからなかった場合は NULL を返します。
    //!
    Parts* FindPartsPaneByName(const char* findName);

    //! @brief 共有頂点属性配列の領域を取得します。
    //!
    //! @return
    //! 共有頂点属性配列の領域です。
    //!
    nw::font::internal::CharAttribute* GetSharedCharAttrs() const
    {
        return m_pSharedCharAttrs;
    }

    //! @brief 共有頂点属性配列の長さを設定します。
    //!
    //! @param[in] length 共有頂点属性配列の長さです。
    //!
    void SetMaxCharAttributeNum(u32 length)
    {
        m_MaxCharAttributeNum = length;
    }

    //! @brief 共有頂点属性配列の長さを取得します。
    //!
    //! @return
    //! 共有頂点属性配列の長さです。
    //!
    u32 GetMaxCharAttributeNum() const
    {
        return m_MaxCharAttributeNum;
    }

    //@}

    //----------------------------------------
    //! @name オブジェクト生成／破棄
    //@{

    //! @brief オブジェクトを生成します。
    //!
    //! @tparam T 生成するオブジェクトのクラスです。
    //!
    //! @return 生成されたオブジェクトへのポインタを返します。
    //!
    //! @sa DeleteObj
    //!
    template<typename T>
    static T* NewObj()
    {
        if (void* pMem = Layout::AllocMemory(sizeof(T)))
        {
            return new (pMem) T();
        }
        return 0;
    }

    //! @brief オブジェクトを生成します。
    //!
    //! @tparam T 生成するオブジェクトのクラスです。
    //! @tparam Param1 コンストラクタの第1引数の型です。
    //!
    //! @param[in] param1   コンストラクタの第1引数です。
    //!
    //! @return 生成されたオブジェクトへのポインタを返します。
    //!
    //! @sa DeleteObj
    //!
    template<typename T, typename Param1>
    static T* NewObj(Param1 param1)
    {
        if (void* pMem = Layout::AllocMemory(sizeof(T)))
        {
            return new (pMem) T(param1);
        }
        return 0;
    }

    //! @brief オブジェクトを生成します。
    //!
    //! @tparam T 生成するオブジェクトのクラスです。
    //! @tparam Param1 コンストラクタの第1引数の型です。
    //! @tparam Param2 コンストラクタの第2引数の型です。
    //!
    //! @param[in] param1   コンストラクタの第1引数です。
    //! @param[in] param2   コンストラクタの第2引数です。
    //!
    //! @return 生成されたオブジェクトへのポインタを返します。
    //!
    //! @sa DeleteObj
    //!
    template<typename T, typename Param1, typename Param2>
    static T* NewObj(Param1 param1, Param2 param2)
    {
        if (void* pMem = Layout::AllocMemory(sizeof(T)))
        {
            return new (pMem) T(param1, param2);
        }
        return 0;
    }

    //! @brief オブジェクトを生成します。
    //!
    //! @tparam T 生成するオブジェクトのクラスです。
    //! @tparam Param1 コンストラクタの第1引数の型です。
    //! @tparam Param2 コンストラクタの第2引数の型です。
    //! @tparam Param3 コンストラクタの第3引数の型です。
    //!
    //! @param[in] param1   コンストラクタの第1引数です。
    //! @param[in] param2   コンストラクタの第2引数です。
    //! @param[in] param3   コンストラクタの第3引数です。
    //!
    //! @return 生成されたオブジェクトへのポインタを返します。
    //!
    //! @sa DeleteObj
    //!
    template<typename T, typename Param1, typename Param2, typename Param3>
    static T* NewObj(Param1 param1, Param2 param2, Param3 param3)
    {
        if (void* pMem = Layout::AllocMemory(sizeof(T)))
        {
            return new (pMem) T(param1, param2, param3);
        }
        return 0;
    }

    //! @brief オブジェクトを生成します。
    //!
    //! @tparam T 生成するオブジェクトのクラスです。
    //! @tparam Param1 コンストラクタの第1引数の型です。
    //! @tparam Param2 コンストラクタの第2引数の型です。
    //! @tparam Param3 コンストラクタの第3引数の型です。
    //! @tparam Param4 コンストラクタの第4引数の型です。
    //!
    //! @param[in] param1   コンストラクタの第1引数です。
    //! @param[in] param2   コンストラクタの第2引数です。
    //! @param[in] param3   コンストラクタの第3引数です。
    //! @param[in] param4   コンストラクタの第4引数です。
    //!
    //! @return 生成されたオブジェクトへのポインタを返します。
    //!
    //! @sa DeleteObj
    //!
    template<typename T, typename Param1, typename Param2, typename Param3, typename Param4>
    static T* NewObj(Param1 param1, Param2 param2, Param3 param3, Param4 param4)
    {
        if (void* pMem = Layout::AllocMemory(sizeof(T)))
        {
            return new (pMem) T(param1, param2, param3, param4);
        }
        return 0;
    }

    // 配列を生成します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] num  要素数です。
    //!
    //! @return 生成した配列です。
    //!
    template<typename T>
    static T* NewArray(u32 num)
    {
        void* pMem = Layout::AllocMemory(sizeof(T) * num);
        if (! pMem)
        {
            return 0;
        }

        T *const objAry = static_cast<T*>(pMem);

        for (u32 i = 0; i < num; ++i)
        {
            new (&objAry[i]) T();
        }

        return objAry;
    }

    //! @brief オブジェクトを廃棄します。
    //!
    //! @tparam T オブジェクトのクラスです。
    //!
    //! @param[in] pObj オブジェクトへのポインタです。
    //!
    template<typename T>
    static void DeleteObj(T* pObj)
    {
        if (pObj)
        {
            pObj->~T();
            FreeMemory(pObj);
        }
    }

    // 配列を廃棄します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] objAry   廃棄する配列です。
    //! @param[in] num      要素数です。
    //!
    template<typename T>
    static void DeleteArray(T objAry[], u32 num)
    {
        if (objAry)
        {
            for (u32 i = 0; i < num; ++i)
            {
                objAry[i].~T();
            }
            FreeMemory(objAry);
        }
    }

    // 配列を廃棄します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] objAry   廃棄する配列です。
    //!
    template<typename T>
    static void DeletePrimArray(T objAry[])
    {
        if (objAry)
        {
            FreeMemory(objAry);
        }
    }

    //@}

protected:

    /**
     *  部品を構築するためのデータを集めたクラスです。
     */
    class PartsBuildDataSet
    {
    public:
        PartsBuildDataSet(Parts* partsPane, const res::Parts* resParts, const BuildResSet* buildResSet, const res::Vec2* originalSize);

        Parts* GetPartsPane() const { return m_PartsPane; }
        const res::PartsProperty* FindPartsPropertyFromName(const char* name) const;
        const void* GetPropertyResBlock(const res::PartsProperty* prop) const;
        const res::ExtUserDataList* GetExtUserDataListResBlock(bool* isOverride, const res::PartsProperty* prop) const;
        const res::PartsPaneBasicInfo* GetPartsPaneBasicInfoResBlock(const res::PartsProperty* prop) const;
        const math::VEC2& GetMagnify() const { return m_Magnify; }
        const BuildResSet* GetPropertyBuildResSet() const { return m_PropertyBuildResSet; }

    protected:
        u32 m_PropertyNum;
        const res::PartsProperty* m_PropertyTable;
        Parts* m_PartsPane;
        const res::Parts* m_ResParts;
        const BuildResSet* m_PropertyBuildResSet;
        math::VEC2 m_Magnify;

    };

    //! @name 初期化
    // @{

    virtual bool BuildImpl(
        const void* lytResBuf,
        ResourceAccessor* pResAcsr,
        const BuildArgSet& parentBuildArgSet,
        const PartsBuildDataSet* partsBuildDataSet);

    //! @brief リソースからペインを生成します。
    //!
    //! @details
    //! kind には以下のいずれかのデータブロック・シグニチャが指定されます。
    //!
    //! ・res::DATABLOCKKIND_PANE\n
    //! ・res::DATABLOCKKIND_PICTURE\n
    //! ・res::DATABLOCKKIND_TEXTBOX\n
    //! ・res::DATABLOCKKIND_WINDOW\n
    //! ・res::DATABLOCKKIND_BOUNDING\n
    //! ・res::DATABLOCKKIND_PARTS
    //!
    //! @param[in] kind             データブロックのシグニチャです。
    //! @param[in] pBlock           データブロックです。
    //! @param[in] pOverrideBlock   オーバーライドするデータブロックです。
    //! @param[in] buildArgSet      構築時の引数のセットです。
    //!
    //! @return 生成されたペインを返します。
    //!
    virtual Pane* BuildPaneObj(
        ut::SigWord kind,
        const void* pBlock,
        const void* pOverrideBlock,
        const BuildArgSet& buildArgSet);

    virtual Layout* BuildPartsLayout(
        const char* name,
        const PartsBuildDataSet& partsBuildDataSet,
        const BuildArgSet& buildArgSet);

    //@}

    //! @name 設定／取得
    // @{

    //! @brief ルートペインを設定します。
    //!
    //! @param[in] pPane    ペインです。
    //!
    void SetRootPane(Pane* pPane)
    {
        m_pRootPane = pPane;
    }

    //! @brief グループコンテナを設定します。
    //!
    //! @param[in] pGroupContainer  グループコンテナです。
    //!
    void SetGroupContainer(GroupContainer* pGroupContainer)
    {
        m_pGroupContainer = pGroupContainer;
    }

    //! @brief 表示領域のサイズを設定します。
    //!
    //! @param[in] size 表示領域のサイズです。
    //!
    void SetLayoutSize(const Size& size)
    {
        m_LayoutSize = size;
    }

    //! @brief 指定したレイアウト名のレイアウトリソースデータを取得します。
    //!
    //! @param[in] layoutName   レイアウト名
    //!
    const void* GetLayoutResourceData(const char* layoutName);

    //! @brief 指定したタグ名のGroupArrayAnimatorを作成し、アニメタグに関連づけられた全てのグループにバインドします。
    //!
    //! @details
    //! GroupArrayAnimatorを作成し、指定されたタグ名のリソースを関連付けます。
    //! さらに、アニメタグに関連づけられた全てのグループにアニメをバインドします。
    //! リソースのフォーマットが正しくなければ失敗します。
    //!
    //! このメソッドを使用するには、バイナリコンバータで-gオプションを使用し、
    //! アニメーションをアニメタグ個別に出力している必要があります。
    //!
    //! 内部実装の共有のために用意していますので、publicにはしていません。
    //!
    //! @param[in] animRes  アニメーションリソースです。
    //! @param[in] enable   開始時にアニメを有効状態にするか。
    //!
    //! @return
    //! アニメーションの作成に成功した場合はアニメーションへのポインタを、
    //! 失敗した場合は NULL を返します。
    //!
    GroupArrayAnimator* CreateGroupArrayAnimator(const AnimResource& animRes, bool enable);
    // @}

protected:
    static nw::ut::IAllocator* s_pAllocator;

    AnimTransformList m_AnimTransList;
    Pane* m_pRootPane;
    GroupContainer* m_pGroupContainer;
    Size m_LayoutSize;
    const char* m_Name;
    ResourceAccessor* m_ResourceAccessor;
    // 部品ペインのリスト。効率のため、他のペインとは別に管理します。このリストではペインの親子関係は関係なく、
    // 一次元のリストになっています。また、部品ペインがリストに登録される順番は不定です。
    PartsPaneList m_PartsPaneList;
    u32 m_MaxCharAttributeNum;
    nw::font::internal::CharAttribute* m_pSharedCharAttrs;

private:
    // コピー演算子を禁止します。
    const Layout& operator=( const Layout& );

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_LAYOUT_H_
